/* Low-level HDLC driver for Motorola MPC860T Communication Processor */
/* To be used with a generic Linux HDLC driver.                       */
/* Written by Adam Kaczynski for DGT-LAB S.A., 2002                   */
/* <kaczor@dgt-lab.com.pl>                                            */
/* Covered by GNU Public Licence ver. 2, see COPYING                  */
/* Prepared for kernel ver. 2.4.18pre                                 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/hdlc.h>
#include <linux/delay.h>
#include <linux/if_arp.h>
#include <asm/io.h>
#include <asm/cache.h>
#include <asm/commproc.h>
#include "commproc_add.h"

// HW definitions
#define IMAP_ADR 0xfff00000
#define DPMEM_TOP 0x1000


// I/O port setup definitions - hardware dependent.
// Only part "B" of TSA is used
 
#define HW_PAPAR_OR_MASK     0xa0f0
#define HW_PADIR_AND_MASK    0xaf00 // 2 additional bits must be set as input - these pins are hardwired on the blade
#define HW_PADIR_OR_MASK     0x0030
#define HW_PAODR_AND_MASK    0x0030
#define HW_PBDIR_OR_MASK     0x00002000 // Output buffer on the blade is switched on  by this bit
#define HW_PBDAT_OR_MASK     0x00002000
#define HW_PCPAR_OR_MASK     0x0300
#define HW_PCDIR_AND_MASK    0x0300


// SW definitions
#define NUMBER_OF_CHANNELS 2 // Number of SCCs used
#define INDEX_OF_FIRST_SCC 0 // SCC1=0, SCC2=1 ...
#define BUFFER_LEN 2048 // should be a power of two, only 1600B is used
#define BUFFER_NUM 8  // for read and write - must be a power of two
#define MAX_HDLC_FRAME 1600

// Other config parameters
#define SW_SCC_RFCR  0x10
#define SW_SCC_TFCR  0x10
#define SW_SCC_CMASK 0x0000f0b8
#define SW_SCC_CPRES 0x0000ffff
#define SW_SCCE_ALL  0xffff
#define SW_SCCM_MASK 0x001a // allow TXE, RXF, TXB interrupts
#define SW_SC_SDCR   0x0001 // SDMA initialization parameter

#define BLINK_RED_LED()      immap->im_cpm.cp_pbdat ^= 0x00000001
#define BLINK_GREEN_LED()    immap->im_cpm.cp_pbdat ^= 0x00000002

// Each port (link) is implemented on a separate SCC
typedef struct m_port_t {
  hdlc_device hdlc;	/* HDLC device struct - must be first */
  uint read_ind; // next buffer to be read by device_read
  uint write_ind; // next buffer to be written by device_write
  int writer_is_waiting; /* It's better to have a seperate variable for it */
  unsigned received_frames; // counters
  unsigned transmitted_frames;
  char* read_buffers; // main data buffer area
  char* write_buffers; 
  cbd_t* read_cbd; // allocated in dpmem
  cbd_t* write_cbd;
  unsigned int scc_num; 
  unsigned int scc_int_mask;
  unsigned int scc_int_num;
  unsigned int scc_isiram_mask_or;
  unsigned int scc_isiram_mask_and;
  scc_t* scc; 
  sccp_t* scc_pr;
  scc_hdlc_t* scc_hdlc_pr; 	// SCC's HDLC specific parameter RAM - an alias to scc_pr
} m_port_t;

//  Memory for link structures is dynamically allocated 
static m_port_t * m_dev; 

// Globals ("consts") of the module, initialized by init_module
static immap_t* immap;
static cpm8xx_t* cp;		

static int debug=0;            //module verbosity set during startup

// Functions used to disable memory cache for selected pages. Copied from apus_setup.c
static __inline__ pte_t *my_find_pte(struct mm_struct *mm,unsigned long va)
{
  pgd_t *dir = 0;
  pmd_t *pmd = 0;
  pte_t *pte = 0;
  
  va &= PAGE_MASK;
  
  dir = pgd_offset( mm, va );
  if (dir)
    {
      pmd = pmd_offset(dir, va & PAGE_MASK);
      if (pmd && pmd_present(*pmd))
        {
          pte = pte_offset(pmd, va);
        }
    }
  return pte;
}

void local_kernel_set_cachemode( unsigned long address, unsigned long size,int is_on)
{
  unsigned long mask,flags;
  
  if (is_on){
    mask = ~(_PAGE_NO_CACHE | _PAGE_GUARDED);
    flags = 0;
    
  } else {
    mask = ~0;
    flags = (_PAGE_NO_CACHE | _PAGE_GUARDED);    
  }
  
  size --;
  size /= PAGE_SIZE; 
  size++;
  
  address &= PAGE_MASK;
  while (size--)
    {
      pte_t *pte;
      
      pte = my_find_pte(&init_mm, address);
      if ( !pte )
        {
          printk("Error: pte NULL in kernel_set_cachemode()\n");
          return;
        }
      
      pte_val (*pte) &= mask;      
      pte_val (*pte) |= flags;      
      address += PAGE_SIZE;
    }
  flush_tlb_all();
}

// Allocation of memory for buffer descriptors, int. queue and buffers
// These pages must NOT be cached
int alloc_mem(void) // returns 0 on success
{
  int i;
  for(i=0;i<NUMBER_OF_CHANNELS;i++){
    if ((m_dev[i].read_buffers = (char*) __get_free_pages(GFP_KERNEL ,get_order(BUFFER_NUM* BUFFER_LEN))) == 0) { 
      printk("Error: __get_free_pages failed on read_buffers\n"); 
      return -ENOMEM; 
    } 
    local_kernel_set_cachemode((long)m_dev[i].read_buffers,BUFFER_NUM* BUFFER_LEN,0); 
    
    if ((m_dev[i].write_buffers = (char*) __get_free_pages(GFP_KERNEL ,get_order(BUFFER_NUM* BUFFER_LEN))) == 0) { 
      printk("Error: __get_free_pages failed on write_buffers\n"); 
      return -ENOMEM; 
    } 
    local_kernel_set_cachemode((long)m_dev[i].write_buffers,BUFFER_NUM* BUFFER_LEN,0); 
  }
  return 0;
}

void free_mem(void)
{
  int i;
  for(i=0;i<NUMBER_OF_CHANNELS;i++){
    if (m_dev[i].read_buffers){ 
      local_kernel_set_cachemode((long)m_dev[i].read_buffers,BUFFER_NUM*BUFFER_LEN,1);
      free_pages((long)m_dev[i].read_buffers ,get_order(BUFFER_NUM* BUFFER_LEN));
    }
    if (m_dev[i].write_buffers){ 
      local_kernel_set_cachemode((long)m_dev[i].write_buffers,BUFFER_NUM*BUFFER_LEN,1);
      free_pages((long)m_dev[i].write_buffers ,get_order(BUFFER_NUM* BUFFER_LEN));
    }   
  }
}

void display_data(char * data, int len)
{
  int i;
  if (len > 32)
    len = 32;
  for(i=0;i<len;i++)
    printk("%02X ",data[i]);
  printk("\n");
}

void receive(m_port_t * _m_dev)
{    
  int len;
  struct sk_buff *skb;

  len = _m_dev->read_cbd[_m_dev->read_ind].cbd_datlen;
  len-=2; // CRC is not very interesting ...

  if(len>MAX_HDLC_FRAME) { // should never happen, mpc868 has set max. length
      _m_dev->hdlc.stats.rx_errors++;
      _m_dev->hdlc.stats.rx_length_errors++;
    return;
  }
 
  skb = dev_alloc_skb(len);
  if (!skb) {
    if (debug)
      printk("Packet dropped\n");
    _m_dev->hdlc.stats.rx_dropped++;
    return;
  }
  
  BLINK_GREEN_LED();
  // copy data from internal buffer to sk_buf
  memcpy(skb->data,__va(_m_dev->read_cbd[_m_dev->read_ind].cbd_bufaddr), len); 
  BLINK_GREEN_LED();

  if(debug>1){
    printk("Rx: ");
    display_data(skb->data,len);
  }
  skb_put(skb, len);

  _m_dev->hdlc.stats.rx_packets++;
  _m_dev->hdlc.stats.rx_bytes += skb->len;

  hdlc_netif_rx(&_m_dev->hdlc, skb); // pass frame through hdlc stack
}

void hdlc_interrupt (void* par, struct pt_regs *regs)
{
  ushort events;

  m_port_t * _m_dev = par;
  events = _m_dev->scc->scc_scce & _m_dev->scc->scc_sccm;
  
  if (events &  SCCE_HDLC_RXF) { // RFTHR frames received
    _m_dev->scc->scc_scce |= SCCE_HDLC_RXF;

    while (!(_m_dev->read_cbd[_m_dev->read_ind].cbd_sc & BD_SC_RD )) { // Process all descriptors
      if (!(_m_dev->read_cbd[_m_dev->read_ind].cbd_sc & BD_SC_ALL_ERR))    
        receive(_m_dev);   //frame reception
      else // error reported
        {
          if(debug)
            printk("SCC_HDLC_RXF error flags=%04X\n",_m_dev->read_cbd[_m_dev->read_ind].cbd_sc);
          _m_dev->read_cbd[_m_dev->read_ind].cbd_sc &= ~BD_SC_ALL_ERR; // clear error flags
          _m_dev->hdlc.stats.rx_errors++;
          if(_m_dev->read_cbd[_m_dev->read_ind].cbd_sc & BD_SC_OV) // overrun
            _m_dev->hdlc.stats.rx_over_errors++;
          if(_m_dev->read_cbd[_m_dev->read_ind].cbd_sc & BD_SC_CR) // CRC
            _m_dev->hdlc.stats.rx_crc_errors++;
          if(_m_dev->read_cbd[_m_dev->read_ind].cbd_sc & (BD_SC_NO | BD_SC_AB))// nonoctet alignment or abort
            _m_dev->hdlc.stats.rx_missed_errors++; //???
          if(_m_dev->read_cbd[_m_dev->read_ind].cbd_sc & BD_SC_LG) // length
            _m_dev->hdlc.stats.rx_length_errors++; //???
          
        }            
      _m_dev->read_cbd[_m_dev->read_ind].cbd_sc|= BD_SC_RD; // Buffer marked as empty
      _m_dev->read_ind++;
      _m_dev->read_ind &= (BUFFER_NUM-1);    
    }
  }
  
  if (events &  SCCE_HDLC_BSY) { // busy (lack of Rx buffers) - should never happen
    if (debug)
      printk("Error: SCCE_HDLC_BSY HDLC interrupt\n");
    while (!(_m_dev->read_cbd[_m_dev->read_ind].cbd_sc & BD_SC_RD)) { // Process all descriptors - dropping frames
      _m_dev->read_cbd[_m_dev->read_ind].cbd_sc|=BD_SC_RD; // Buffer is ready for next frame
      _m_dev->read_cbd[_m_dev->read_ind].cbd_sc &= ~BD_SC_ALL_ERR; // clear error flags
      _m_dev->read_ind++;
      _m_dev->read_ind &= (BUFFER_NUM-1);    
      _m_dev->hdlc.stats.rx_errors++;
      //      printk("c");
      _m_dev->hdlc.stats.rx_dropped++; // receievr fifo overrun
    }
    _m_dev->scc->scc_scce |= SCCE_HDLC_BSY;
  }
  
  if (events &  SCCE_HDLC_TXB) { // Tx buffer sent
    //frame has been transmitted - wake up queue if necessary
    if(_m_dev->writer_is_waiting){
      _m_dev->writer_is_waiting = 0;
      netif_wake_queue(hdlc_to_dev(&_m_dev->hdlc));
      if(debug>1)
        printk("+");
    }        
    _m_dev->scc->scc_scce |= SCCE_HDLC_TXB;
  }
  
  if (events &  SCCE_HDLC_RXB) { // Rx (incomplete) buffer received SHOULD NEVER HAPPEN
    if (debug)
      printk ("Error: SCCE_HDLC_RXB HDLC interrupt\n");
    if(!(_m_dev->read_cbd[_m_dev->read_ind].cbd_sc & BD_SC_RD)) { // Process only one descriptor      
      _m_dev->read_cbd[_m_dev->read_ind].cbd_sc|=BD_SC_RD; // Buffer is ready for next frame
      _m_dev->read_cbd[_m_dev->read_ind].cbd_sc &= ~BD_SC_ALL_ERR; // clear error flags
      _m_dev->read_ind++;
      _m_dev->read_ind &= (BUFFER_NUM-1);    
      _m_dev->hdlc.stats.rx_errors++;
      //      printk("d");
      _m_dev->hdlc.stats.rx_missed_errors++; // ???
      _m_dev->scc->scc_scce |= SCCE_HDLC_RXB;
    }
  }   
}

static int m_ioctl(hdlc_device *hdlc, struct ifreq *ifr, int cmd)
{
  int value = ifr->ifr_ifru.ifru_ivalue;
  int result = 0;
  unsigned int mask = 1 << 31; // MUST BE unsigned !!!
  int i;
  int *isiram;
  m_port_t * _m_dev = (m_port_t*) hdlc;

  isiram = (int*) &cp->cp_siram;

  if(debug) 
    printk("m_ioctl\n");

  if(!capable(CAP_NET_ADMIN))
    return -EPERM;

  switch(cmd) {
  case HDLCGCLOCK:
    value = 0; // CLOCK EXT
    break;
    
  case HDLCGCLOCKRATE:
    value = 2048000; // E1 clock rate
    break;
    
  case HDLCGLINE:
    value = 0; // default
    break;

  case HDLCGSLOTMAP:
    value = 0;
    
    for(i=0;i<32;i++, mask >>= 1){ // only receiver checked
      if((isiram[i+32] & SI_RAM_ROUT_MSK) == _m_dev->scc_isiram_mask_or) 
        value |= mask;
    }
    break;

  case HDLCSSLOTMAP:
    if (debug)
      printk("ioctl HDLCSSLOTMAP requested mask: %08X\n",value);
    // TSA "B" part must be used due to hadrware design.
    cp->cp_sigmr = 0;//Disable TSA during change
    
    for(i=0;i<32;i++, mask >>= 1){ // only receiver checked
      if(value & mask) {
        isiram[i+32] |= _m_dev->scc_isiram_mask_or; // mask for SCC2
        isiram[i+96] |= _m_dev->scc_isiram_mask_or; // mask for SCC2
      }
      else {
        isiram[i+32] &= ~_m_dev->scc_isiram_mask_and; // not connected
        isiram[i+96] &= ~_m_dev->scc_isiram_mask_and; // not connected
      }
    }
    cp->cp_sigmr =  SIGMR_ENB | SIGMR_STATIC_32; // enable TSA

    break;

  default:
    return -EINVAL;
  } 
  
  ifr->ifr_ifru.ifru_ivalue = value;
  return result;
}

static int m_open(hdlc_device *hdlc)
{
  m_port_t * _m_dev = (m_port_t*) hdlc;

  if(debug)
    printk("m_open\n");

  _m_dev->scc->scc_gsmrl |= (SCC_GSMRL_ENT | SCC_GSMRL_ENR); //Enable receiver & transmitter
  netif_start_queue(hdlc_to_dev(hdlc));
  return 0;
}

static void m_close(hdlc_device *hdlc)
{
  m_port_t * _m_dev = (m_port_t*) hdlc;

  if(debug)
    printk("m_close\n");
  netif_stop_queue(hdlc_to_dev(hdlc));
  _m_dev->scc->scc_gsmrl &= ~(SCC_GSMRL_ENT | SCC_GSMRL_ENR); //Disable receiver & transmitter

}

static int m_xmit(hdlc_device *hdlc, struct sk_buff *skb)
{
  int len;

  m_port_t * _m_dev = (m_port_t*) hdlc;

  if(_m_dev->write_cbd[_m_dev->write_ind].cbd_sc & BD_SC_OV){  // previous error found
    _m_dev->write_cbd[_m_dev->write_ind].cbd_sc &= ~BD_SC_OV;
    _m_dev->hdlc.stats.tx_aborted_errors++; // ???
    _m_dev->hdlc.stats.tx_errors++;
  }
  
  if(_m_dev->write_cbd[_m_dev->write_ind].cbd_sc & BD_SC_RD){   //Buffer ready for transmission found == queue is full
     if(debug>1)
      printk("-");
    
    netif_stop_queue(hdlc_to_dev(hdlc));
    _m_dev->writer_is_waiting = 1; 
    return 1; // packet is queued
  }

  if (skb->len > MAX_HDLC_FRAME) {
    if (debug)
      printk("Tx is dropping too long packet\n");
    _m_dev->hdlc.stats.tx_dropped++;    
    return 0; // packet dropped
  }
  else
    len = skb->len;
  
  _m_dev->hdlc.stats.tx_packets++;
  _m_dev->hdlc.stats.tx_bytes += len;
	
  BLINK_RED_LED();
  memcpy(__va(_m_dev->write_cbd[_m_dev->write_ind].cbd_bufaddr),skb->data,len);
  BLINK_RED_LED();
  dev_kfree_skb_any(skb);
  
  if(debug > 1){
    printk("Tx: ");
    display_data(__va(_m_dev->write_cbd[_m_dev->write_ind].cbd_bufaddr),len);
  }
  _m_dev->write_cbd[_m_dev->write_ind].cbd_datlen = len;

  // The buffer is ready for transmission + Last in frame + CRC +Intr
  _m_dev->write_cbd[_m_dev->write_ind].cbd_sc |= (BD_SC_RD | BD_SC_INT| BD_SC_LST | BD_SC_TC);
  _m_dev->write_ind++;
  _m_dev->write_ind &= (BUFFER_NUM-1);  
  
  //  dev_kfree_skb_any(skb);
  // for(len=0;len<50;len++);
  
  //dev_kfree_skb(skb);
  return 0;
}

static void setup_ports(void)
{
  // Only port B of TSA is used
  // If you want to use port A, change settings
  immap->im_ioport.iop_papar |=  HW_PAPAR_OR_MASK; 
  immap->im_ioport.iop_padir &= ~HW_PADIR_AND_MASK; 
  immap->im_ioport.iop_padir |=  HW_PADIR_OR_MASK;  
  immap->im_ioport.iop_paodr &= ~HW_PAODR_AND_MASK;
  immap->im_cpm.cp_pbdir |= HW_PBDIR_OR_MASK; 
  immap->im_cpm.cp_pbdat |= HW_PBDAT_OR_MASK;  
  immap->im_ioport.iop_pcpar |= HW_PCPAR_OR_MASK;
  immap->im_ioport.iop_pcdir &= ~HW_PCDIR_AND_MASK;

  if (debug) 
    printk("setup_ports finished \n"); 
}


void init_tsa(void) // Step 1,2,3,~4,5,6,7,8
{
  int *isiram; // right size is int
  int i;
  // settings for SI Clock Route register
  // The real mask for SCCx will be generated by shifting bits...
  unsigned int mask_or  = 0x00000040;
  unsigned int mask_and = 0x000000bf;

  isiram = (int*)cp->cp_siram;

  cp->cp_sigmr = 0; // disable TSA
  cp->cp_simode = 0;

  mask_or <<= (8*INDEX_OF_FIRST_SCC);
  mask_and <<= (8*INDEX_OF_FIRST_SCC);
  for(i=0;i<NUMBER_OF_CHANNELS;i++){
    cp->cp_sicr &= ~mask_and;
    cp->cp_sicr |= mask_or;
    mask_or <<= 8;  
    mask_and <<= 8;
  }

  for (i=0;i<128;i++)
    isiram[i]= SI_RAM_BYTE;    

  isiram[31] |= SI_RAM_LAST; 
  isiram[63] |= SI_RAM_LAST;
  isiram[95] |= SI_RAM_LAST;
  isiram[127]|= SI_RAM_LAST;
  // SI RAM routing will be set up by ioctl 
  if(debug)
    printk("init_tsa finished\n"); 
}

static int init_link_port(int port_num,int scc_num) 
{  
  int i;
  m_port_t * _m_dev = &m_dev[port_num];

  // STEP 8
  while (cp->cp_cpcr & CPM_CR_FLG)
    schedule();
  switch(scc_num) {
  case 0:
    cp->cp_cpcr =  mk_cr_cmd(CPM_CR_CH_SCC1,  CPM_CR_INIT_TRX) | CPM_CR_FLG;
    break;
  case 1:
    cp->cp_cpcr =  mk_cr_cmd(CPM_CR_CH_SCC2,  CPM_CR_INIT_TRX) | CPM_CR_FLG;
    break;
  case 2:
    cp->cp_cpcr =  mk_cr_cmd(CPM_CR_CH_SCC3,  CPM_CR_INIT_TRX) | CPM_CR_FLG;
    break;
  case 3:
    cp->cp_cpcr =  mk_cr_cmd(CPM_CR_CH_SCC4,  CPM_CR_INIT_TRX) | CPM_CR_FLG;
    break;
  }  
  while (cp->cp_cpcr & CPM_CR_FLG)
    schedule();
  
  //STEP 9
  _m_dev->scc_pr->scc_rfcr = SW_SCC_RFCR;
  _m_dev->scc_pr->scc_tfcr = SW_SCC_TFCR;   
  
   
   //STEP 10
  _m_dev->scc_pr->scc_mrblr = MAX_HDLC_FRAME; 
   
  //STEP 11
  _m_dev->scc_hdlc_pr->scc_cmask = SW_SCC_CMASK; // CRC mask
  
  //STEP 12
  _m_dev->scc_hdlc_pr->scc_cpres = SW_SCC_CPRES; // CRC preset
  
   //STEP 13 
  _m_dev->scc_hdlc_pr->scc_disfc = 0; 	// discarded frame counter
  _m_dev->scc_hdlc_pr->scc_crcec = 0; 	// CRC error counter
  _m_dev->scc_hdlc_pr->scc_abtsc = 0; 	// abort sequence counter
  _m_dev->scc_hdlc_pr->scc_nmarc = 0; 	// nonmatching address Rx counter
  _m_dev->scc_hdlc_pr->scc_retrc = 0; 	// frame retransmission counter
  
   //STEP 14
  _m_dev->scc_hdlc_pr->scc_mflr  = MAX_HDLC_FRAME; // Max frame length
   
   //STEP 15
  _m_dev->scc_hdlc_pr->scc_rfthr = 1; // Allow interrups after each frame
   
   //STEP 16
   // HDLC frame address recognition
  _m_dev->scc_hdlc_pr->scc_hmask  = 0; 	// address mask register: allow all addresses (no address recognition)
   
   //STEP 17
  _m_dev->scc_hdlc_pr->scc_haddr1 = 0; 	// address register 1
  _m_dev->scc_hdlc_pr->scc_haddr2 = 0; 	// address register 2
  _m_dev->scc_hdlc_pr->scc_haddr3 = 0; 	// address register 3
  _m_dev->scc_hdlc_pr->scc_haddr4 = 0; 	// address register 4
   
   //STEP 18 and 19 
  for(i=0;i<BUFFER_NUM;i++) {
    _m_dev->read_cbd[i].cbd_datlen=0;
    _m_dev->write_cbd[i].cbd_datlen=0;
    _m_dev->read_cbd[i].cbd_bufaddr = __pa(&_m_dev->read_buffers[i*BUFFER_LEN]);

    _m_dev->write_cbd[i].cbd_bufaddr = __pa(&_m_dev->write_buffers[i*BUFFER_LEN]);
    _m_dev->read_cbd[i].cbd_sc = BD_SC_RD; // ready
    _m_dev->write_cbd[i].cbd_sc = 0; 
  }
  _m_dev->read_cbd[BUFFER_NUM-1].cbd_sc |= BD_SC_WRP; // wrap - last buffer in the queue
  _m_dev->write_cbd[BUFFER_NUM-1].cbd_sc |=BD_SC_WRP ;
  
   //STEP 20
  _m_dev->scc->scc_scce = SW_SCCE_ALL; // clear all pending events
  
  //STEP 21
  _m_dev->scc->scc_sccm = SW_SCCM_MASK; 
  
  //STEP 22
  immap->im_cpic.cpic_cisr |= _m_dev->scc_int_mask; // clear SCCx interrupts at CPIC
  immap->im_cpic.cpic_cipr |= _m_dev->scc_int_mask; // clear SCCx interrupts at CPIC

  cpm_install_handler(_m_dev->scc_int_num,hdlc_interrupt,_m_dev);
  
  //STEP 23
  _m_dev->scc->scc_gsmrh = 0;
  
  //STEP 24
  _m_dev->scc->scc_gsmrl = 0;
  
  //STEP 25
  _m_dev->scc->scc_pmsr = 0;
  
  // STEP 26 moved to m_open

  // Setting up driver info 
 
  hdlc_to_dev(&_m_dev->hdlc)->irq = _m_dev->scc_int_num;
  hdlc_to_dev(&_m_dev->hdlc)->mem_start = 0; // not applicable
  hdlc_to_dev(&_m_dev->hdlc)->mem_end = 0;
  hdlc_to_dev(&_m_dev->hdlc)->tx_queue_len = BUFFER_NUM ;
  hdlc_to_dev(&_m_dev->hdlc)->mtu = MAX_HDLC_FRAME;
 
  return 0;
}


static int __init mpc860hdlc_init(void)
{
  int i,result;
  if (debug)
    printk("debug=%i\n", debug);
  
  immap = (immap_t*) IMAP_ADR ; // pointer to internal registers
  cp = &immap->im_cpm; // pointer to Communication Processor

  m_dev  = kmalloc(NUMBER_OF_CHANNELS*sizeof(m_port_t), GFP_KERNEL); 
  memset(m_dev, 0, NUMBER_OF_CHANNELS*sizeof(m_port_t));

  // some parameters (masks) for SCC are generated here
  for(i=0;i<NUMBER_OF_CHANNELS;i++){
    m_dev[i].hdlc.ioctl = m_ioctl;
    m_dev[i].hdlc.open = m_open;
    m_dev[i].hdlc.close = m_close;
    m_dev[i].hdlc.xmit = m_xmit;
    m_dev[i].scc_num = i+INDEX_OF_FIRST_SCC;
    m_dev[i].scc_int_mask = 0x40000000 >> ( i+INDEX_OF_FIRST_SCC);
    m_dev[i].scc_int_num = 30 - (i+INDEX_OF_FIRST_SCC) ;
    // generation of masks used later for connecting timeslots 
    m_dev[i].scc_isiram_mask_or = 0x00400000 * (1+i+INDEX_OF_FIRST_SCC); // 001, 010, 011, 100
    m_dev[i].scc_isiram_mask_and = 0x001c0000 ^ m_dev[i].scc_isiram_mask_or;
  
    m_dev[i].read_ind = 0;  
    m_dev[i].write_ind = 0; 
    m_dev[i].writer_is_waiting = 0; 
    m_dev[i].scc = &cp->cp_scc[m_dev[i].scc_num]; // pointer to SCC parameters
    // 0x100 is size of SCC parameters
    m_dev[i].scc_pr = (sccp_t*)&cp->cp_dparam[0x100 * m_dev[i].scc_num]; 
    m_dev[i].scc_hdlc_pr = (scc_hdlc_t*)(m_dev[i].scc_pr);
  }

  result = alloc_mem();
  if (result) {
    printk("mpc860hdlc_init memory error\n");
    return result;
  }

  for(i=0;i<NUMBER_OF_CHANNELS;i++){
    result = register_hdlc_device(&m_dev[i].hdlc); 
    
    if(result){
      free_mem(); 
      kfree(m_dev);
      printk("mpc860hdlc_init registration error\n");
      return result;
    }
  }

  setup_ports();
  init_tsa();

  // STEP 6 SDMA initialization
  immap->im_siu_conf.sc_sdcr = SW_SC_SDCR;
  
  // STEP 7 
  // WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING WARNING 
  // Here I perform manual allocation of dpmem. I decided not to use alloc function supplied with the kernel
  // since there is no deallocation procedure for it. This module can be easily deallocated.
  // I grab memory from the top. Allocator works from the bottom. 
 
  for (i=0;i<NUMBER_OF_CHANNELS;i++){    
    m_dev[i].scc_pr->scc_rbase = DPMEM_TOP - (2*i+2)*sizeof(cbd_t)*BUFFER_NUM; 
    m_dev[i].scc_pr->scc_tbase = DPMEM_TOP - (2*i+1)*sizeof(cbd_t)*BUFFER_NUM;
    
    m_dev[i].read_cbd = (cbd_t*)&cp->cp_dpmem[m_dev[i].scc_pr->scc_rbase];
    m_dev[i].write_cbd = (cbd_t*)&cp->cp_dpmem[m_dev[i].scc_pr->scc_tbase];
  }
  // END OF UGLY STEP 7

  // The rest of initialzation is hidden here
  for (i=0;i<NUMBER_OF_CHANNELS;i++) {
   init_link_port(i, i+INDEX_OF_FIRST_SCC);
  }

  BLINK_GREEN_LED();
  
  printk("mpc860hdlc_init finished\n");
  return 0;
}

static void __exit mpc860hdlc_cleanup(void)
{
  int i;
  for (i=0;i<NUMBER_OF_CHANNELS;i++) {
    unregister_hdlc_device(&m_dev[i].hdlc);
    m_dev[i].scc->scc_gsmrl &= ~0x00000030; // disable SCC 
    immap->im_cpic.cpic_cimr &= ~(m_dev[i].scc_int_mask); // this is done by cpm_free_handler but too late!
    cpm_free_handler(m_dev[i].scc_int_num);
  }
  free_mem();
  kfree(m_dev);
  BLINK_GREEN_LED();
  printk("mpc860hdlc_cleanup finished\n");  
}

module_init(mpc860hdlc_init);
module_exit(mpc860hdlc_cleanup); 

MODULE_AUTHOR("Adam Kaczynski");
MODULE_DESCRIPTION("MPC860 HDLC driver");
MODULE_PARM(debug,"i");
MODULE_LICENSE("GPL");

EXPORT_NO_SYMBOLS;

