/*
 * HDLC (synchronous) driver for Motorola MPC8xx, 825x and 826x SCC.
 * Copyright (c) 2002-2003 R SCOP Consult. (scop@plugin.com.br)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This driver is based on Daris Nevil's HDLC driver for MPC8xx SCC
 * and Dan Malek's Ethernet driver for MPC8260 SCC. It's also based on
 * some K. Halasa synchronous boards' drivers.
 *
 * ORIGINAL COPYRIGHT MESSAGES:
 * -------------------------------------------------------------------------
 * Copyright 2000 SNMC
 * Simple Network Magic Corporation
 *		
 * Copyright (c) 1999 Dan Malek (dmalek@jlc.net)
 * Copyright (c) 2000 MontaVista Software Inc. (source@mvista.com)
 *
 * Copyright (C) 1998-2002 Krzysztof Halasa <khc@pm.waw.pl>
 * -------------------------------------------------------------------------
 *
 * Buffer descriptors are kept in the CPM dual port RAM, and the frame
 * buffers are in the host memory.
 *
 * At first, this was basically a work of updating the driver code to 2.4.x
 * kernels, and inserting conditional code to work with MPC82xx platforms
 * (826x and 825x, that is, not 824x).
 *
 * After that, some optimizations were made in the tx ring management code.
 *
 * Now, it interfaces to K. Halasa's Generic HDLC Layer, with PPP, Cisco
 * HDLC, Frame Relay and X25 protocols.
 *
 * SCOP: My Gosh! I'll have to use netif_carrier_ok _and_ IFF_RUNNING state to
 *       "notify" SNMP agents the state of the link protocol (link state),
 *       so that they can set correctly their "Interface Operational Status".
 *
 * TODO list:
 *  - Implement DCD handling (to call netif_carrier_on/off when DCD changes)
 *  - Implement DTR handling (where the hw permits)
 *  - Standardize board dependent stuff.
 *
 */

#define HAVE_CHANGE_MTU

#include <linux/config.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/netdevice.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/reboot.h>
#include <linux/spinlock.h>

#include <asm/uaccess.h>
#include <asm/processor.h>
#include <asm/system.h>
#include <asm/semaphore.h>
#include <asm/irq.h>

#ifdef CONFIG_8xx
# include <asm/8xx_immap.h>
# include <asm/mpc8xx.h>
#else  /* 8260 */
# include <asm/immap_8260.h>
# include <asm/mpc8260.h>
# include <asm/cpm_8260.h>
#endif

#include <asm/cpm_hdlc.h>

#include <asm/scchdlc.h>
#include <linux/hdlc.h>

/* Include board dependent definitions */

#ifdef CONFIG_NR2G
#include <asm/nr2g.h>
#endif

#ifdef CONFIG_TRITON
#include <asm/triton.h>
#endif

/*
 *				Theory of Operation
 *
 * The MPC8xx and MPC82xx CPMs perform the HDLC processing on an SCC.
 * It can use an arbitrary number of buffers on byte boundaries, but 
 * must have at least two receive buffers to prevent constant overrun
 * conditions.
 *
 * The buffer descriptors are allocated from the CPM dual port memory
 * with the data buffers allocated from host memory, just like all other
 * serial communication protocols.  This creates a ring buffer
 * structure similar to the LANCE and other controllers.
 *
 * The driver runs as two independent, single-threaded flows of control.  One
 * is the send-packet routine, which enforces single-threaded use by the
 * port->tx_full flag.  The other thread is the interrupt handler + bottom half
 * tasks, which is single threaded by the hardware and other software.
 */

#ifdef CONFIG_SCC_HDLC_DEBUG
#define DEBUG(args...) printk(KERN_DEBUG args)
#else
#define DEBUG(args...)
#endif

/* The transmitter timeout
 */
#define TX_TIMEOUT	(2*HZ)

/* Tx and Rx ring sizes */
#define RX_RING_SIZE 8
#define TX_RING_SIZE 2 /* SCOP: testing at 64k only!!!! */

/* Maximum size for RX Buffers
 * Must be a multiple of 4 */
#define MAX_RX_BUFF_SIZE 1560 // 390*4

/* This value must be <= MAX_RX_BUFF_SIZE */
#define SCC_HDLC_MTU (1550+8)

#define SCC_PARAM_SIZE	0x100	/* SCC Parameter RAM size */

/* Defines number of RX Frames before an interrupt is generated */
#define RX_FRAME_THRESHOLD 1

/* HDLC Addresses and Masks */
#define HDLC_C_MASK	0xF0B8	/*	CRC Constant */
#define HDLC_C_PRES	0xFFFF	/* CRC Preset */
#define HDLC_ADDR_MASK	0x0000	/* Address Mask (0-bit: we need to
						 accept any address bits for F. Relay to work) */
#define HDLC_AS_ADDR	0x00FF	/* All-Stations Address */
#define HDLC_CM_ADDR	0x008F	/* Cisco Multicast Address */
#define HDLC_CU_ADDR	0x000F	/* Cisco Unicast Address */
#define HDLC_ZE_ADDR	0x0000  /* Let's enable address 0 to see if
								   it still works... -- Scop */

/*-------------------------------------------------------------------
 * Local structures */

/* The CPM buffer descriptors track the ring buffers.  The rx_bd_base and
 * tx_bd_base always point to the base of the buffer descriptors.  The
 * rx_tail and tx_head point to the currently available buffer.
 * The dirty_tx tracks the current buffer that is being sent by the
 * controller.  The tx_head and dirty_tx are equal under both completely
 * empty and completely full conditions.  The port->tx_full flag
 * determines the actual condition.
 */
typedef struct {
	
    /* HDLC device struct - must be first */
    hdlc_device	hdlc;
	
	/* Synchronous port parameters */
	/* FIXME: We could use these latter -- Scop */
	sync_serial_settings settings;
	unsigned short encoding;
	unsigned short parity;
	
	/* SCC device number (1-4) */
	int			scc_num;
	int			scc_param_index;

	/* Buffer descriptors */
	/* TX Buffers and descriptors */
	volatile cbd_t*	tx_bd_base; /* Start of TX Buffer Descriptors */
	uint 	scc_tbase;
	volatile cbd_t*	tx_bd[TX_RING_SIZE];
	int		tx_head;
	struct sk_buff*	tx_skbuff[TX_RING_SIZE];

	/* RX Buffers and descriptors */
	volatile cbd_t*	rx_bd_base; /* Start of RX Buffer Descriptors */
	uint	scc_rbase;
	volatile cbd_t*	rx_bd[RX_RING_SIZE];
	int		rx_tail;
	struct sk_buff*	rx_skbuff[RX_RING_SIZE];

	/* Bottom Half task queues */
	struct tq_struct	rx_bh_tq;
	struct tq_struct	tx_bh_tq;
	
	/* The tx ring entries to be free()ed. */
	int	dirty_tx;
		
	/* transmission ring full flag */
	uint	tx_full; 

	/* mutual exclusion access control */
	spinlock_t lock; 

	} hdlc_port_t;

typedef struct
	{
	ushort		scc_number;
	uint		pram_index;
	uint		chan_cmd; /* must have a typecast to ushort
						     when used with 8xx */
	ushort		irq_vector;
	uint		clock_route_mask; /* signicant bits must be set to 1 */
	uint		clock_route;
	} SCC_Params_t;

/*-------------------------------------------------------------------
 * Function Prototypes */

#ifdef CONFIG_8xx
static void scc_hdlc_interrupt(void *dev_id , struct pt_regs * regs);
#else /* 8260 */
static void scc_hdlc_interrupt(int irq, void *dev_id, struct pt_regs * regs);
#endif

static int scc_hdlc_attach ( hdlc_device *hdlc, unsigned short encoding, unsigned short parity);
static int scc_hdlc_hard_start_xmit(struct sk_buff* skb, struct net_device* dev);
static int scc_hdlc_ioctl(struct net_device* dev, struct ifreq* ifr, int cmd);
static int scc_hdlc_change_mtu(struct net_device* dev, int new_mtu);
static int scc_hdlc_stop(struct net_device* dev);
static int scc_hdlc_alloc_rx_ring(struct net_device* dev);
static struct sk_buff* scc_hdlc_alloc_skb(int mtu);
static void scc_hdlc_restart_rx(struct net_device* dev);
static void scc_hdlc_restart_tx(struct net_device* dev);
static int scc_hdlc_open(struct net_device* dev);
static void scc_hdlc_tx_bh(void* dev_id);
static void scc_hdlc_rx_bh(void* dev_id);
static void scc_hdlc_timeout(struct net_device *dev);
static void scc_set_iface(hdlc_port_t *port);
static void set_multicast_list(struct net_device *dev);

/*-------------------------------------------------------------------
  This is BOARD-dependent stuff */

#if defined(CONFIG_NR2G_PROTO_ALPHA)

# define HDLC_SCC1_NUMBER	1
# define HDLC_SCC2_NUMBER	2
# define HDLC_SCC3_NUMBER	0
# define HDLC_SCC4_NUMBER	0
/* See page 15-15 of the MPC8260 User's Manual */
/* SCC1 RX CLK3 and TX CLK4 */
# define CMX_HDLC_CLK_ROUTE_SCC1	((uint)0x37000000)
# define CMX_HDLC_CLK_MASK_SCC1		((uint)0xff000000)
/* SCC2 RX CLK11 and TX CLK12 */
# define CMX_HDLC_CLK_ROUTE_SCC2	((uint)0x00250000)
# define CMX_HDLC_CLK_MASK_SCC2		((uint)0x00ff0000)
/* SCC3 RX and TX clocks undefined (no pins for that...) */
# define CMX_HDLC_CLK_ROUTE_SCC3	((uint)0x00000000)
# define CMX_HDLC_CLK_MASK_SCC3		((uint)0x0000ff00)
/* SCC4 RX and TX clocks undefined (no pins for that...) */
# define CMX_HDLC_CLK_ROUTE_SCC4	((uint)0x00000000)
# define CMX_HDLC_CLK_MASK_SCC4		((uint)0x000000ff)

#elif defined(CONFIG_NR2G) /* PROTO_BETA and further ... */

# ifdef CONFIG_NR2G_SCC1
#  define HDLC_SCC1_NUMBER	1
/* See page 15-15 of the MPC8260 User's Manual */
/* SCC1 RX CLK3 and TX CLK4 */
#  define CMX_HDLC_CLK_ROUTE_SCC1	((uint)0x37000000)
# else
#  define HDLC_SCC1_NUMBER	0
/* See page 15-15 of the MPC8260 User's Manual */
/* SCC1 RX and TX clocks undefined (no pins for that...) */
#  define CMX_HDLC_CLK_ROUTE_SCC1	((uint)0x00000000)
# endif

# define HDLC_SCC2_NUMBER	2

# ifdef CONFIG_NR2G_SCC3
#  define HDLC_SCC3_NUMBER	3
/* SCC3 RX CLK7 and TX CLK8 */
#  define CMX_HDLC_CLK_ROUTE_SCC3	((uint)0x00003700)
# else
#  define HDLC_SCC3_NUMBER	0
/* SCC3 RX and TX clocks undefined (no pins for that...) */
#  define CMX_HDLC_CLK_ROUTE_SCC3	((uint)0x00000000)
# endif

# define HDLC_SCC4_NUMBER	4

# define CMX_HDLC_CLK_MASK_SCC1		((uint)0xff000000)

/* SCC2 RX CLK11 and TX CLK12 */
# define CMX_HDLC_CLK_ROUTE_SCC2	((uint)0x00250000)
# define CMX_HDLC_CLK_MASK_SCC2		((uint)0x00ff0000)

# define CMX_HDLC_CLK_MASK_SCC3		((uint)0x0000ff00)

/* SCC4 RX CLK5 and TX CLK6 */
# define CMX_HDLC_CLK_ROUTE_SCC4	((uint)0x00000025) /* 2d for RC=TC */
# define CMX_HDLC_CLK_MASK_SCC4		((uint)0x000000ff)


#elif defined(CONFIG_SDH)

# define HDLC_SCC1_NUMBER		0
# define HDLC_SCC2_NUMBER		2
# define HDLC_SCC3_NUMBER		3
# define HDLC_SCC4_NUMBER		0

# define SICR_HDLC_MASK_SCC1		0x000000FF
# define SICR_HDLC_MASK_SCC2		0x0000FF00
# define SICR_HDLC_MASK_SCC3		0x00FF0000
# define SICR_HDLC_MASK_SCC4		0xFF000000

# define SICR_HDLC_CLKRT_SCC1	0x00000000	//undefined
# define SICR_HDLC_CLKRT_SCC2	0x00003700
# define SICR_HDLC_CLKRT_SCC3	0x00250000
# define SICR_HDLC_CLKRT_SCC4	0x00000000	//undefined

#elif defined(CONFIG_TRITON)

# define HDLC_SCC1_NUMBER		1
# define HDLC_SCC2_NUMBER		2
# define HDLC_SCC3_NUMBER		0
# define HDLC_SCC4_NUMBER		0

# define SICR_HDLC_MASK_SCC1		0x000000FF
# define SICR_HDLC_MASK_SCC2		0x0000FF00
# define SICR_HDLC_MASK_SCC3		0x00FF0000
# define SICR_HDLC_MASK_SCC4		0xFF000000

# define SICR_HDLC_CLKRT_SCC1	0x00000025
#ifdef CONFIG_SCC2_HDLC   /* Titan ModRt */
# define SICR_HDLC_CLKRT_SCC2	0x00003700  //undefined
#else	
# define SICR_HDLC_CLKRT_SCC2	0x00000000  //undefined
#endif
# define SICR_HDLC_CLKRT_SCC3	0x00000000  //undefined
# define SICR_HDLC_CLKRT_SCC4	0x00000000	//undefined

#else /* Test = SCC1 */
/* SICR_HDLC_MASK_SCCx and SICR_HDLC_CLKRT_SCCx are board
   dependent stuff. Please define them according
   to your needs. -- Scop */

# define HDLC_SCC1_NUMBER	1
# define HDLC_SCC2_NUMBER	0
# define HDLC_SCC3_NUMBER	0
# define HDLC_SCC4_NUMBER	0
/* See page 15-15 of the MPC8260 User's Manual */
/* SCC1 RX CLK3 and TX CLK4 */
# define CMX_HDLC_CLK_ROUTE_SCC1	((uint)0x37000000)
# define CMX_HDLC_CLK_MASK_SCC1	((uint)0xff000000)
/* SCC2 RX RX and TX clocks undefined  */
# define CMX_HDLC_CLK_ROUTE_SCC2	((uint)0x00000000)
# define CMX_HDLC_CLK_MASK_SCC2		((uint)0x00ff0000)
/* SCC3 RX and TX clocks undefined (no pins for that...) */
# define CMX_HDLC_CLK_ROUTE_SCC3	((uint)0x00000000)
# define CMX_HDLC_CLK_MASK_SCC3		((uint)0x0000ff00)
/* SCC4 RX and TX clocks undefined (no pins for that...) */
# define CMX_HDLC_CLK_ROUTE_SCC4	((uint)0x00000000)
# define CMX_HDLC_CLK_MASK_SCC4		((uint)0x000000ff)


#endif

/*-------------------------------------------------------------------
 * Global initializers */

/* Convert an HDLC device pointer into a port pointer and similar */
#define hdlc_to_port(H) ((hdlc_port_t *)(H))
#define dev_to_port(D)  hdlc_to_port(dev_to_hdlc(D))
#define port_to_dev(P)  hdlc_to_dev(&(P)->hdlc)

#define MAX_HDLC_CHANNELS	4

static hdlc_port_t HdlcPorts[MAX_HDLC_CHANNELS];


#ifdef CONFIG_8xx

# define CPM_T 		cpm8xx_t
# define MK_CR_CMD(CH, CMD)	mk_cr_cmd((ushort) CH, CMD)
# define CPM_BDRAM_BASE		cp->cp_dpmem	/* SCC BD Ram base */
# define SCC_PARAM_BASE		cp->cp_dparam	/* SCC Parameter RAM base */


static SCC_Params_t SCC_Params[MAX_HDLC_CHANNELS] = {
	{HDLC_SCC1_NUMBER, PROFF_SCC1, CPM_CR_CH_SCC1, CPMVEC_SCC1,
		SICR_HDLC_MASK_SCC1, SICR_HDLC_CLKRT_SCC1},
	{HDLC_SCC2_NUMBER, PROFF_SCC2, CPM_CR_CH_SCC2, CPMVEC_SCC2,
		SICR_HDLC_MASK_SCC2, SICR_HDLC_CLKRT_SCC2},
	{HDLC_SCC3_NUMBER, PROFF_SCC3, CPM_CR_CH_SCC3, CPMVEC_SCC3,
		SICR_HDLC_MASK_SCC3, SICR_HDLC_CLKRT_SCC3},
	{HDLC_SCC4_NUMBER, PROFF_SCC4, CPM_CR_CH_SCC4, CPMVEC_SCC4,
		SICR_HDLC_MASK_SCC4, SICR_HDLC_CLKRT_SCC4},
};

#else /* 8260 */

# define CPM_T 		cpm8260_t
# define MK_CR_CMD(CH, CMD)	(CH | CMD)
# define CPM_BDRAM_BASE	immap->im_dprambase
# define SCC_PARAM_BASE	CPM_BDRAM_BASE

# ifdef CONFIG_NR2G_PROTO_ALPHA
static SCC_Params_t SCC_Params[MAX_HDLC_CHANNELS] = {
	{HDLC_SCC1_NUMBER, PROFF_SCC1,
		mk_cr_cmd(CPM_CR_SCC1_PAGE, CPM_CR_SCC1_SBLOCK, 0, 0),
		SIU_INT_SCC1, CMX_HDLC_CLK_MASK_SCC1, CMX_HDLC_CLK_ROUTE_SCC1},
	{HDLC_SCC2_NUMBER, PROFF_SCC2,
		mk_cr_cmd(CPM_CR_SCC2_PAGE, CPM_CR_SCC2_SBLOCK, 0, 0), 
		SIU_INT_SCC2, CMX_HDLC_CLK_MASK_SCC2, CMX_HDLC_CLK_ROUTE_SCC2},
	{HDLC_SCC3_NUMBER, PROFF_SCC3,
		mk_cr_cmd(CPM_CR_SCC3_PAGE, CPM_CR_SCC3_SBLOCK, 0, 0), 
		SIU_INT_SCC3, CMX_HDLC_CLK_MASK_SCC3, CMX_HDLC_CLK_ROUTE_SCC3},
	{HDLC_SCC4_NUMBER, PROFF_SCC4,
		mk_cr_cmd(CPM_CR_SCC4_PAGE, CPM_CR_SCC4_SBLOCK, 0, 0), 
		SIU_INT_SCC4, CMX_HDLC_CLK_MASK_SCC4, CMX_HDLC_CLK_ROUTE_SCC4},
};
# elif defined(CONFIG_NR2G)
static SCC_Params_t SCC_Params[MAX_HDLC_CHANNELS] = {
	{HDLC_SCC4_NUMBER, PROFF_SCC4,
		mk_cr_cmd(CPM_CR_SCC4_PAGE, CPM_CR_SCC4_SBLOCK, 0, 0),
		SIU_INT_SCC4, CMX_HDLC_CLK_MASK_SCC4, CMX_HDLC_CLK_ROUTE_SCC4},
	{HDLC_SCC2_NUMBER, PROFF_SCC2,
		mk_cr_cmd(CPM_CR_SCC2_PAGE, CPM_CR_SCC2_SBLOCK, 0, 0),
		SIU_INT_SCC2, CMX_HDLC_CLK_MASK_SCC2, CMX_HDLC_CLK_ROUTE_SCC2},
	{HDLC_SCC1_NUMBER, PROFF_SCC1,
		mk_cr_cmd(CPM_CR_SCC1_PAGE, CPM_CR_SCC1_SBLOCK, 0, 0),
		SIU_INT_SCC1, CMX_HDLC_CLK_MASK_SCC1, CMX_HDLC_CLK_ROUTE_SCC1},
	{HDLC_SCC3_NUMBER, PROFF_SCC3,
		mk_cr_cmd(CPM_CR_SCC3_PAGE, CPM_CR_SCC3_SBLOCK, 0, 0),
		SIU_INT_SCC3, CMX_HDLC_CLK_MASK_SCC3, CMX_HDLC_CLK_ROUTE_SCC3},
};
# endif /* CONFIG_NR2G_PROTO_ALPHA */

#endif /* CONFIG_8xx */



/*--------------------------------------------------------------
 * scc_alloc_cpm_dpram()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *
 * Allocate DPRAM for the device's BDs.
 */

static int scc_alloc_cpm_dpram(hdlc_port_t *port)
{
	struct net_device *dev = port_to_dev(port);
	uint i;

#ifdef CONFIG_8xx
	volatile CPM_T* cp = cpmp;
# define CPM_DPALLOC(X, Y)	m8xx_cpm_dpalloc(X)
#else /* 8260 */
	volatile	immap_t		*immap;
# define CPM_DPALLOC(X, Y)	m8260_cpm_dpalloc(X, Y)
	immap = (immap_t *)IMAP_ADDR;	/* Get pointer to internal registers */
#endif
	/* Allocate space for the buffer descriptors in the DP ram.
	 * These are relative offsets in the DP ram address space.
	 * Initialize base addresses for the buffer descriptors.
	 */
	
	/* This must be called at init time, for now, because we simply
	   can't do it dinamically until we have a way to deallocate it ...
	   -- Scop */
	
	/* Allocate RX Buffer descriptors */
	i = CPM_DPALLOC(sizeof(cbd_t) * RX_RING_SIZE, 8);
	if (i == CPM_DP_NOSPACE)
		{
		printk(KERN_WARNING "%s: No more CPM DPRAM for Rx\n", dev->name);
		return -ENOMEM;
		}

	port->scc_rbase = i;
	port->rx_bd_base = (cbd_t*)&CPM_BDRAM_BASE[i];

	/* Allocate TX Buffer descriptors */
	i = CPM_DPALLOC(sizeof(cbd_t) * TX_RING_SIZE, 8);
	if (i == CPM_DP_NOSPACE)
		{
		printk(KERN_WARNING "%s: No more CPM DPRAM for Tx\n", dev->name);
		return -ENOMEM;
		}

	port->scc_tbase = i;
	port->tx_bd_base = (cbd_t*)&CPM_BDRAM_BASE[i];
	
	return 0;

}

static void set_multicast_list(struct net_device *dev)
{
	return;
}

/*--------------------------------------------------------------
 * scc_hdlc_init()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	None.
 *
 *  Initialize the SCC port for operation in HDLC mode compatible
 *	to K. Halasa's Generic HDLC Layer.
 */

#ifdef CONFIG_8xx

# define CLOCK_ROUTE_REG	cp->cp_sicr
# define FCR_BIG_ENDIAN		SCC_EB

#else /* 8260 */

# define CLOCK_ROUTE_REG	immap->im_cpmux.cmx_scr
# define FCR_BIG_ENDIAN		(CPMFCR_GBL | CPMFCR_EB)

#endif

static int __init scc_hdlc_init(void)
{
	int i;
	int result;
#ifdef CONFIG_8xx
	volatile CPM_T* cp = cpmp;
#else /* 8260 */
	volatile	immap_t		*immap;
#endif
	
	DEBUG("SCC_HDLC: scc_hdlc_init()\n");
	printk(KERN_INFO "SCC HDLC Driver for MPC8xx and 82xx, v1.4\n");
	printk(KERN_INFO "(c) Copyright 2002-2003, R SCOP Consult., scop@plugin.com.br\n");
/* DEBUG("Exiting ...\n"); return 0; */

#ifdef CONFIG_8xx
# define BASE_ADDR_SCC	cp->cp_scc 
#else /* 8260 */
# define BASE_ADDR_SCC	immap->im_scc
	immap = (immap_t *)IMAP_ADDR;	/* Get pointer to internal registers */
#endif

	for (i = 0; i < MAX_HDLC_CHANNELS; ++i) {
 		hdlc_port_t *port = &HdlcPorts[i];
		struct net_device *dev = hdlc_to_dev(&port->hdlc);
		
		__clear_user(&HdlcPorts[i], sizeof HdlcPorts[0]);

		/* Initialize only hardcode-enabled SCCs */
		if (SCC_Params[i].scc_number == 0)
			continue;

#ifdef CONFIG_TRITON
    	{
    	volatile ulong *p = (ulong *) ((ulong) ADDR_INP1);
	    ulong inp = *p;

		    if ((inp & 0x0f) != 1 &&  /* Not DUAL WAN model - no SCC2 for WAN */
	    		SCC_Params[i].scc_number == 2) {
				SCC_Params[i].scc_number = 0;
				continue;
    		}
    	}
#endif			

#ifdef CONFIG_NR2G
/* Caruso - Krypton does not have an expansion bus. */
# ifndef CONFIG_EXTIO_16BIT
    	{
    	volatile ulong *p = (OUT_PORT_TYPE *) ((ulong) IN_PORT_0);
	    ulong inp = *p;

            /* SCOP: revisar! */
		    if ((inp & 0xF0000000) != 0x00000000 &&  /* No Wendy in slot 0 (X1) */
	    		SCC_Params[i].scc_number == 1) {
				SCC_Params[i].scc_number = 0;
				continue;
    		}
            else if ((inp & 0x0F000000) != 0x00000000 &&  /* No Wendy in slot 1 (X2) */
	    		SCC_Params[i].scc_number == 3) {
				SCC_Params[i].scc_number = 0;
				continue;
    		}
    	}
# endif
#endif
		spin_lock_init(&HdlcPorts[i].lock);

		/* Save the SCC Channel number */
		port->scc_num = SCC_Params[i].scc_number; /* Hack: double used
													variable -- Scop */
		port->scc_param_index = i;

		/* Fill in the net device info */
		/* --------------------------- */
		
		/* Get pointer to base of SCC control registers
		 * (index, not number) */
		dev->base_addr = (unsigned long)&BASE_ADDR_SCC[port->scc_num - 1];

		/* Get beginning of my SCC's Parameter Ram */ 
		dev->mem_start = (unsigned long)&SCC_PARAM_BASE[SCC_Params[i].pram_index];

		/* Get end of my SCC's Parameter Ram */
		dev->mem_end = dev->mem_start + SCC_PARAM_SIZE - 1;

		/* Dinamically assign a name to the device
		result = dev_alloc_name(dev, "hdlc%d");
		if (result<0)
			return result;
		*/

		/* Fill out the function pointers */
		dev->open = scc_hdlc_open;
		dev->stop = scc_hdlc_stop;
		dev->do_ioctl = scc_hdlc_ioctl;
		port->hdlc.attach = scc_hdlc_attach;
		port->hdlc.xmit = scc_hdlc_hard_start_xmit;
		port->settings.clock_type = CLOCK_EXT;

		dev->change_mtu = scc_hdlc_change_mtu;

		dev->tx_timeout = scc_hdlc_timeout;
		dev->watchdog_timeo = TX_TIMEOUT;
		dev->set_multicast_list = set_multicast_list;

		/* SET_MODULE_OWNER(dev); */

		dev->tx_queue_len = 100; /* I completely forgot that - Scop */
		
		/* Register my HDLC device */
		result = register_hdlc_device(&port->hdlc);
		if (result) {
			printk(KERN_WARNING "SCC_HDLC: unable to register hdlc "
			       "device (error=%d), so port hdlc%d disabled.\n",
			       result, i);
			SCC_Params[i].scc_number = 0;
			continue;
		}
		
		/* Let's allocate BDs in DPRAM here, for now */
		result = scc_alloc_cpm_dpram(port);
		if (result) {
			return result;
		}
		
		/* Configure Serial Interface clock routing.
		 * First, clear all SCC bits to zero, then set the ones we want.
		 *
		 * These are default (external clocks' sources) values -- Scop.
		CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
		CLOCK_ROUTE_REG |= SCC_Params[port->scc_param_index].clock_route;
		 */
	
	} /* for */

	/* Success */
	return(0);
}

/*--------------------------------------------------------------
 * scc_hdlc_exit()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	None.
 *
 * Remove the device from the list of network devices.
 * Return allocated resources back to the system pool.
 */

static void __exit scc_hdlc_exit(void)
{
	int i;

	DEBUG("SCC_HDLC: scc_hdlc_exit()\n");

	for (i = 0; i < MAX_HDLC_CHANNELS; ++i)
		{
		/* Cleanup only still enabled SCCs */
		if (SCC_Params[i].scc_number == 0)
			continue;

		/* Unregister the HDLC device */
		unregister_hdlc_device(&HdlcPorts[i].hdlc);

		}
}


/*--------------------------------------------------------------
 * scc_hdlc_open()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) port - pointer to this driver's port structure
 *
 * Open the device
 */

static int scc_open_hdlc_port(hdlc_port_t *port)
{
	struct net_device *dev = port_to_dev(port);
	/* hdlc_port_t* port = (hdlc_port_t*)dev->priv; */
	volatile scc_t* sccp = (scc_t*) dev->base_addr;
	volatile scc_hdlc_t* hp = (scc_hdlc_t*) dev->mem_start;
	int result;
	volatile immap_t* immap = (immap_t *)IMAP_ADDR;	/* Get pointer to internal registers */
#ifdef CONFIG_8xx
	/* volatile CPM_T* cp = cpmp; */
#else /* 8260 */
	volatile	iop8260_t	*io = &immap->im_ioport;
#endif
	
	DEBUG("%s: scc_open_hdlc_port()\n", dev->name);

	/* Make sure the parameter RAM area is not already allocated */
	if ((result = (check_region(dev->mem_start, SCC_PARAM_SIZE))))
		return(result);

	/* Allocate the parameter RAM */
	request_region(dev->mem_start, SCC_PARAM_SIZE, dev->name);

	/* Disable rx & tx */
	sccp->scc_gsmrl &= ~(SCC_GSMRL_ENR | SCC_GSMRL_ENT);
	
	/* Configure I/O Pins
	 * For me, this was already done by PPCBoot, hopefully -- Scop */
	
#if defined(CONFIG_NR2G_PROTO_ALPHA)
	if (port->scc_num == 2) {
		io->iop_pparc &= ~0x00080000;
		io->iop_pdirc &= ~0x00080000;
		io->iop_psorc &= ~0x00080000;
		io->iop_podrc &= ~0x00080000;
		io->iop_pdatc &= ~0x00080000;
	}
	else {  /* SCC1 */
		io->iop_pparc &= ~0x00020000;
		io->iop_pdirc &= ~0x00020000;
		io->iop_psorc &= ~0x00020000;
		io->iop_podrc &= ~0x00020000;
		io->iop_pdatc &= ~0x00020000;
	}

#elif defined(CONFIG_NR2G)

#ifdef CONFIG_NR2G_SCC1
    if (port->scc_num == 1) {
	/* Force DCD ON - CTS, TC and RC are inputs - This is board dependent */
		io->iop_pparc &= ~0x00020000;
		io->iop_pparc |=  0x0001000C;
		io->iop_pdirc &= ~0x0002000C;
		io->iop_psorc &= ~0x0002000C;
		io->iop_podrc &= ~0x0003000C;
		io->iop_pdatc &= ~0x00020000;

    /* Set XTC, RTS and TD as outputs - RD is input - This is board dependent */
    	io->iop_ppard |=  0x00200007;
		io->iop_pdird &= ~0x00000001;
		io->iop_pdird |=  0x00200006;
		io->iop_psord &= ~0x00000005;
		io->iop_psord |=  0x00200002;
		io->iop_podrd &= ~0x00200007;
    }
	else
#endif

# ifdef CONFIG_NR2G_SCC3
    if (port->scc_num == 3) {
	/* Force DCD ON - CTS, TC and RC are inputs - This is board dependent */
		io->iop_pparc &= ~0x00200000;
		io->iop_pparc |=  0x001000C0;
		io->iop_pdirc &= ~0x002000C0;
		io->iop_psorc &= ~0x002000C0;
		io->iop_podrc &= ~0x003000C0;
		io->iop_pdatc &= ~0x00200000;

    /* Set XTC, RTS and TD as outputs - RD is input - This is board dependent */
    	io->iop_ppard |=  0x004001C0;
		io->iop_pdird &= ~0x00000040;
		io->iop_pdird |=  0x00400180;
		io->iop_psord &= ~0x000001C0;
		io->iop_psord |=  0x00400000;
		io->iop_podrd &= ~0x004001C0;
	}
	else
# endif

	/* Force DCD ON - This is board dependent */
    if (port->scc_num == 2) {
		io->iop_pparc &= ~0x00080000;
		io->iop_pdirc &= ~0x00080000;
		io->iop_psorc &= ~0x00080000;
		io->iop_podrc &= ~0x00080000;
		io->iop_pdatc &= ~0x00080000;
	}
	else {  /* SCC4 */
		io->iop_pparc &= ~0x00800000;
		io->iop_pdirc &= ~0x00800000;
		io->iop_psorc &= ~0x00800000;
		io->iop_podrc &= ~0x00800000;
		io->iop_pdatc &= ~0x00800000;
	}
#endif  /* CONFIG_NR2G */

#if defined(CONFIG_TRITON)
	/* Force DCD ON - This is board dependent */
	if (port->scc_num == 1) {
		immap->im_ioport.iop_pcdir &= ~0x0020;
		immap->im_ioport.iop_pcpar &= ~0x0020;
		immap->im_ioport.iop_pcso &= ~0x0020;
		immap->im_ioport.iop_pcdat &= ~0x0020;
	}
# ifdef CONFIG_SCC2_HDLC  /* Titan ModRt */
    if (port->scc_num == 2) {
		immap->im_ioport.iop_pcdir &= ~0x0080;
		immap->im_ioport.iop_pcpar &= ~0x0080;
		immap->im_ioport.iop_pcso &= ~0x0080;
		immap->im_ioport.iop_pcdat &= ~0x0080;
	}
# endif
#endif
	
	/* Set the GSMR_H and GSMR_L registers */
	/* FIXME: Shouldn't we set transmit flags when idle? -- Scop */
	/* SCOP: YES, let's try it */
	/* FIXME: in the future, this should be programmable */
	/* ATTENTION!!! These two lines must be here, before the following
	   scc_set_iface call, which modifies some bits of scc_gsmrl -- Scop */
	sccp->scc_gsmrh = 0 | SCC_GSMRH_RTSM;
	sccp->scc_gsmrl = SCC_GSMRL_MODE_HDLC;

    /* Configure clocks and BRGs */
	scc_set_iface(port);
	
	/* Initialize Tx and Rx rings' bases */
	hp->genscc.scc_rbase = port->scc_rbase;
	hp->genscc.scc_tbase = port->scc_tbase;
	
	/* FIXME: Here we would have to put DPRAM allocation when we can
	   free it dinamically... -- Scop */
	   /* if (result = scc_alloc_cpm_dpram(port)) {
			return result;
		}
	   ... */
	
	/* Init Function Code registers (Big Endian) */
	hp->genscc.scc_rfcr = hp->genscc.scc_tfcr = FCR_BIG_ENDIAN; 

	/* Set the RX Buffer maximum size 
	 * This appears to be a frame size, not the buffer
	 * fragment size.  It must be a multiple of four.
	 */
	hp->genscc.scc_mrblr = MAX_RX_BUFF_SIZE;

	/* Set the MTU */
	hp->mflr = SCC_HDLC_MTU;

	/* Set the CRC Constant and Preset registers */
	hp->c_mask = HDLC_C_MASK;
	hp->c_pres = HDLC_C_PRES;

	/* Set the All-Stations Address (used for PPP)
	 * Also set Cisco Multicast and Cisco Unicast addresses
	 */
	hp->hmask = HDLC_ADDR_MASK;
	hp->haddr1 = HDLC_ZE_ADDR;
	hp->haddr2 = HDLC_ZE_ADDR;
	hp->haddr3 = HDLC_ZE_ADDR;
	hp->haddr4 = HDLC_ZE_ADDR;
	
	/* Clear the pm counters */
	hp->disfc = 0;
	hp->crcec = 0;
	hp->abtsc = 0;
	hp->nmarc = 0;
	hp->retrc = 0;

	/* Set the RX Frame Threshold */
	hp->rfthr = RX_FRAME_THRESHOLD;

	/* Set the Protocol Specific Mode Register
	 * NOF = 0 (1 flag between adjacent frames)
	 * CRC = 16-bit
	 * No Retransmission
	 * FSE = Normal
	 * DRT = Normal
	 * No HDLC Bus mode
	 * BRM = Ignored
	 * MFF = Normal */
	sccp->scc_pmsr = 0x0000;

	/* Disable All Interrupts for this SCC */
	sccp->scc_sccm = 0x0000;

	/* Clear pending events */
	sccp->scc_scce = 0xFFFF;

	/* Install my interrupt handler */
#ifdef CONFIG_8xx
	cpm_install_handler(SCC_Params[port->scc_param_index].irq_vector,
		scc_hdlc_interrupt, dev);
#else /* 8260 */
	request_8xxirq(SCC_Params[port->scc_param_index].irq_vector,
		scc_hdlc_interrupt, 0, "hdlc", dev);
#endif

	/* Enable Interrupts for Transmit Error, Rx Received, and TXB
	   Let's enable BSY dectection, too. -- Scop */
	sccp->scc_sccm = (SCCM_TXE | SCCM_RX | SCCM_RXF | SCCM_TX | SCCM_BSY);

	/* Set the DSR register, just it case it has been modified
	 * by another driver since reset */
	sccp->scc_dsr = 0x7E7E;

	/* Allocate buffers for all RX Buf Descr */
	if ((result = scc_hdlc_alloc_rx_ring(dev)))
		goto fail_buffer;

#if 0
    /* SCOP: Now this is controlled by the protocol layer */
    /* FIXME: DCD always on,for now... -- Scop */
	netif_carrier_on(dev);
#endif

	/* Initialize and start the Receiver and Transmitter */
	scc_hdlc_restart_rx(dev);
	scc_hdlc_restart_tx(dev);

	DEBUG("%s: scc_hdlc_open() --> 001\n", dev->name);

	goto done;

fail_clocks:
	printk("fail_clocks\n");
	goto fail_release;

fail_buffer:
	printk("fail_buffer\n");

fail_release:	
	release_region(dev->mem_start, SCC_PARAM_SIZE);

done:
	/* Done */
	return(result);
}

static int scc_hdlc_open(struct net_device *dev)
{
	hdlc_device *hdlc = dev_to_hdlc(dev);
	hdlc_port_t *port = hdlc_to_port(hdlc);
    int result;

	DEBUG("%s: scc_hdlc_open()\n", dev->name);
	
#ifdef CONFIG_TRITON
# ifndef CONFIG_SCC2_HDLC   /* not a Titan ModRt */
    if (port->scc_num != HDLC_SCC1_NUMBER)
		return -EINVAL;	/* Hack: SCC2 was enabled just to allow async PPP */
# endif
#endif
	
	result = hdlc_open(hdlc);
	if (result)
		return result;

	result = scc_open_hdlc_port(port);
	if (result) {
		hdlc_close(hdlc);
		return result;
	}
	
	MOD_INC_USE_COUNT;
	netif_start_queue(dev);

    printk("%s: admin status: UP.\n", dev->name);

#ifdef CONFIG_EXT_DEVSTATS
	port->hdlc.ext_stats.admin_status = 1; /* UP */
	port->hdlc.ext_stats.admin_last_change = jiffies;
#endif

	return 0;
}

/*--------------------------------------------------------------
 * scc_hdlc_stop()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *
 * Close the device and free all resources
 */

static void scc_close_hdlc_port(hdlc_port_t *port)
{
	struct net_device *dev = port_to_dev(port);
	volatile scc_t* sccp = (scc_t*)dev->base_addr;
	volatile CPM_T* cp = cpmp;
	int i;

	DEBUG("%s: scc_close_hdlc_port()\n", dev->name);

	/* Send a Stop Transmit command to the CPM */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
		CPM_CR_STOP_TX) | CPM_CR_FLG;
	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG,CPM_CR_FLG,scc_close_hdlc_port());

	/* Disable rx & tx */
	sccp->scc_gsmrl &= ~(SCC_GSMRL_ENR | SCC_GSMRL_ENT);

	/* Free the socket buffers associated with the RX buffer rings */
	for (i = 0; i < RX_RING_SIZE; ++i)
		{
		/* Free any allocated socket buffers */
		if (port->rx_skbuff[i])
			dev_kfree_skb(port->rx_skbuff[i]);

		port->rx_skbuff[i] = 0;
		}

	/* Free the socket buffers associated with the TX buffer rings */
	for (i = 0; i < TX_RING_SIZE; ++i)
		{
		/* Free any allocated socket buffers */
		if (port->tx_skbuff[i])
			dev_kfree_skb(port->tx_skbuff[i]);

		port->tx_skbuff[i] = 0;
		}

	/* Release the Parameter Ram associated with this channel */
	release_region(dev->mem_start, SCC_PARAM_SIZE);

	/* FIXME: Release RX & TX Buffer descriptors */
	/* m8xx_cpm_dpfree(hp->genscc.scc_rbase); */
	/* m8xx_cpm_dpfree(hp->genscc.scc_tbase); */

	/* Disable All Interrupts for this SCC */
	sccp->scc_sccm = 0x0000;

	/* Clear pending events */
	sccp->scc_scce = 0xFFFF;


	/* Uninstall the interrupt handler */
	/* Just calling a request with a null handler does the trick */
#ifdef CONFIG_8xx
	cpm_install_handler(SCC_Params[port->scc_param_index].irq_vector,
		0, dev);
#else /* 8260 */
	request_8xxirq(SCC_Params[port->scc_param_index].irq_vector,
		0, 0, "hdlc", dev);
#endif

	/* Update statistics */
	/* FIXME */

	/* Done */
	return;
}

static int scc_hdlc_stop(struct net_device *dev)
{
	hdlc_device *hdlc = dev_to_hdlc(dev);
	hdlc_port_t *port = hdlc_to_port(hdlc);

	DEBUG("%s: scc_hdlc_stop()\n", dev->name);

#ifdef CONFIG_EXT_DEVSTATS
	port->hdlc.ext_stats.admin_status = 0; /* DOWN */
	port->hdlc.ext_stats.admin_last_change = jiffies;
#endif

    printk("%s: admin status: DOWN.\n", dev->name);

    netif_stop_queue(dev);
	scc_close_hdlc_port(port);
	hdlc_close(hdlc);
	MOD_DEC_USE_COUNT;
	return 0;
}

static void scc_encoding_setting(hdlc_port_t *port)
{
   /* Nothing to do here */

}

static void scc_clock_setting(hdlc_port_t *port)
{
#ifdef CONFIG_8xx
	volatile CPM_T* cp = cpmp;
#else /* 8260 */
	volatile immap_t* immap = (immap_t *)IMAP_ADDR;	/* Get pointer to internal registers */
#endif
	struct net_device *dev = port_to_dev(port);
	uint bps = port->settings.clock_rate;

	switch(port->settings.clock_type) {
	case CLOCK_TXINT:
		switch(port->scc_num) {
#if defined(CONFIG_NR2G_PROTO_ALPHA)
		case 1:
			CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
			CLOCK_ROUTE_REG |= (CMX_HDLC_CLK_ROUTE_SCC1 & ~0x07000000) | 0x00000000; /* SCC1 = BRG1 */
			break;
		case 2:
			CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
			CLOCK_ROUTE_REG |= (CMX_HDLC_CLK_ROUTE_SCC2 & ~0x00070000) | 0x00010000; /* SCC2 = BRG2 */
			break;
#elif defined(CONFIG_NR2G)	/* PROTO_BETA and further */	
# ifdef CONFIG_NR2G_SCC1
        case 1:
			CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
			CLOCK_ROUTE_REG |= (CMX_HDLC_CLK_ROUTE_SCC1 & ~0x07000000) | 0x03000000; /* SCC1 = BRG4 */
			break;
# endif
# ifdef CONFIG_NR2G_SCC3
        case 3:
			CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
			CLOCK_ROUTE_REG |= (CMX_HDLC_CLK_ROUTE_SCC3 & ~0x00000700) | 0x00000200; /* SCC3 = BRG3 */
			break;
# endif
        case 2:
			CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
			CLOCK_ROUTE_REG |= (CMX_HDLC_CLK_ROUTE_SCC2 & ~0x00070000) | 0x00010000; /* SCC2 = BRG2 */
			break;
		case 4:
			CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
			CLOCK_ROUTE_REG |= (CMX_HDLC_CLK_ROUTE_SCC4 & ~0x00000007) | 0x00000000; /* SCC4 = BRG1 */
			break;
#elif defined(CONFIG_TRITON)	/* TRITON boards and similars */	
		case 1:
			CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
			CLOCK_ROUTE_REG |= (SICR_HDLC_CLKRT_SCC1 & ~0x00000007) | 0x00000003; /* SCC1 = BRG4 */
			break;
# ifdef CONFIG_SCC2_HDLC         /* Titan ModRt */
		case 2:
			CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
			CLOCK_ROUTE_REG |= (SICR_HDLC_CLKRT_SCC2 & ~0x00000700) | 0x00000200; /* SCC2 = BRG3 */
			break;
# endif
#endif		
		default:
   		    printk(KERN_WARNING "%s: Invalid port number = %d \n",
	  		    dev->name, port->scc_num);
			return;
		}
		break;

	default:
	case CLOCK_EXT:
		CLOCK_ROUTE_REG &= ~SCC_Params[port->scc_param_index].clock_route_mask;
		CLOCK_ROUTE_REG |= SCC_Params[port->scc_param_index].clock_route;
		return;
	}

#ifdef CONFIG_8xx
# ifdef CONFIG_TRITON
   {
   int tmp;
   tmp = port->scc_param_index+3;
#  ifdef CONFIG_SCC2_HDLC  /* Titan ModRt */
   if (tmp == 4) tmp=2; /* SCOP: grrr... Caruso inverted that in the HW */
#  endif
   if (bps != m8xx_cpm_getfastbrg(tmp)) {
   		m8xx_cpm_fastbrg(tmp, bps);
   		port->settings.clock_rate = m8xx_cpm_getfastbrg(tmp);
   }
# else
   if (bps != m8xx_cpm_getfastbrg(port->scc_param_index)) {
		m8xx_cpm_fastbrg(port->scc_param_index, bps);
		port->settings.clock_rate = m8xx_cpm_getfastbrg(port->scc_param_index);
# endif		
#else
   {
   int tmp;
   tmp = port->scc_param_index;
#ifdef CONFIG_NR2G
   if (tmp == 2) tmp=3; /* SCOP: grrr... Caruso inverted that in the HW */
   else if (tmp == 3) tmp=2;
#endif
   if (bps != m8260_cpm_getfastbrg(tmp)) {
   		m8260_cpm_fastbrg(tmp, bps);
   		port->settings.clock_rate = m8260_cpm_getfastbrg(tmp);
   }
#endif
   		
   		printk(KERN_INFO "%s: XTC frequency adjusted from %d to %d Hz\n",
	  		dev->name, bps, port->settings.clock_rate);
   }

}

static void scc_loopback_setting(hdlc_port_t *port)
{
	struct net_device *dev = port_to_dev(port);
	volatile scc_t* sccp = (scc_t*)dev->base_addr;
	
	if (port->settings.loopback) {
		printk(KERN_INFO "%s: loopback on\n", dev->name);
		sccp->scc_gsmrl |= SCC_GSMRL_DIAG_LOOP | SCC_GSMRL_DIAG_ECHO;
	}
	else {
		printk(KERN_INFO "%s: loopback off\n", dev->name);
		sccp->scc_gsmrl &= ~(SCC_GSMRL_DIAG_LOOP | SCC_GSMRL_DIAG_ECHO);
	}

}

static void scc_txinv_setting(hdlc_port_t *port)
{
	struct net_device *dev = port_to_dev(port);
	volatile scc_t* sccp = (scc_t*)dev->base_addr;
	
	if (port->settings.txinv) {
		printk(KERN_INFO "%s: tx clock invertion on\n", dev->name);
		sccp->scc_gsmrl |= SCC_GSMRL_TCI;
	}
	else {
		printk(KERN_INFO "%s: tx clock invertion off\n", dev->name);
		sccp->scc_gsmrl &= ~SCC_GSMRL_TCI;
	}

}

static void scc_crc_setting(hdlc_port_t *port)
{
#if 0
	/* This is from Rodolfo Giacometti's driver */
	
	uint parity = port->settings.clock_rate; /* double-used ??? */
	struct net_device *dev = port_to_dev(port);
	volatile scc_hdlc_t *hp = (scc_hdlc_t *) dev->base_addr;

   if (parity == PARITY_CRC16_PR0_CCITT ||
       parity == PARITY_CRC16_PR1_CCITT) {
      hp->c_pres = 0x0000ffff;
      hp->c_mask = 0x0000f0b8;
      printk(KERN_INFO "%s: using 16-bit CRC-CCITT\n", dev->name);
   }
   if (parity == PARITY_CRC32_PR0_CCITT ||
       parity == PARITY_CRC32_PR1_CCITT) {
      hp->c_pres = 0xffffffff;
      hp->c_mask = 0xdebb20e3;
      printk(KERN_INFO "%s: using 32-bit CRC-CCITT\n", dev->name);
   }
#endif

}

static void scc_set_iface(hdlc_port_t *port)
{
   struct {
      void (*action)(hdlc_port_t *);
   } *p, do_setting[] = {
      { scc_encoding_setting },
      { scc_clock_setting },
      { scc_loopback_setting },
      { scc_txinv_setting },
      { scc_crc_setting },
      { NULL }
   };

   for (p = do_setting; p->action; p++)
       p->action(port);

}

static unsigned int board_get_iface(hdlc_port_t *port)
{

#ifdef CONFIG_NR2G	
static unsigned int nr2gwan_phy[] = {
	IF_IFACE_V24,
	IF_IFACE_V11,
	IF_IFACE_V35,
	IF_IFACE_SYNC_SERIAL
};

# if defined(CONFIG_NR2G_SCC1) || defined (CONFIG_NR2G_SCC3)
static unsigned int nr2gslot_phy[] = {
	IF_IFACE_SYNC_SERIAL,
	IF_IFACE_V24,
	IF_IFACE_G703,
	IF_IFACE_V35,
	IF_IFACE_V24,
	IF_IFACE_V11,
	IF_IFACE_V35,
	IF_IFACE_SYNC_SERIAL
};
# endif

/* Caruso - Krypton does not have WAN ports. */
# ifndef CONFIG_EXTIO_16BIT
    volatile OUT_PORT_TYPE *inp = (OUT_PORT_TYPE *) IN_PORT_0;
	
	switch (port->scc_param_index) {
# ifdef CONFIG_NR2G_SCC3
    case 3:
	    {
	    volatile immap_t* immap = (immap_t *)IMAP_ADDR;	/* Get pointer to internal registers */
        volatile	iop8260_t	*io = &immap->im_ioport;
   		return nr2gslot_phy[(io->iop_pdata & 0x38000000) >> 27];
        }
# endif
# ifdef CONFIG_NR2G_SCC1
    case 2:
	    {
	    volatile immap_t* immap = (immap_t *)IMAP_ADDR;	/* Get pointer to internal registers */
        volatile	iop8260_t	*io = &immap->im_ioport;
   		return nr2gslot_phy[(io->iop_pdatc & 0x38000000) >> 27];
        }
# endif
    case 1:
   		return nr2gwan_phy[(*inp & 0x0C000) >> 14];
	case 0:
		return nr2gwan_phy[(*inp & 0x01800) >> 11];
	default:
		break;
	}
# endif /* CONFIG_EXTIO_16BIT */
#endif  /* CONFIG_NR2G */

#ifdef CONFIG_TRITON	
static unsigned int nr2gwan_phy[] = {
	IF_IFACE_SYNC_SERIAL,
	IF_IFACE_SYNC_SERIAL,
	IF_IFACE_G703,
	IF_IFACE_SYNC_SERIAL,
	IF_IFACE_V24,
	IF_IFACE_V11,
	IF_IFACE_V35,
	IF_IFACE_SYNC_SERIAL
};

    volatile unsigned long *inp = (unsigned long *) ADDR_INP0;
	
	switch (port->scc_param_index) {
	case 0:
		return nr2gwan_phy[(*inp & 0x07000) >> 12];
	default:
		break;
	}
#endif

   return IF_IFACE_SYNC_SERIAL;
}

/*--------------------------------------------------------------
 * scc_hdlc_ioctl()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *	(IO) ifr - pointer to a structure for passing arguments
 *	(I) cmd - ioctl() function specifier
 *
 * Perform special device control functions
 */

static int scc_hdlc_ioctl(struct net_device* dev,
	struct ifreq* ifr, int cmd)
{
#if 0	
	volatile scc_t* sccp = (scc_t*)dev->base_addr;
#endif	
	const size_t size = sizeof(sync_serial_settings);
	sync_serial_settings new_line, *line = ifr->ifr_settings.ifs_ifsu.sync;
	hdlc_device *hdlc = dev_to_hdlc(dev);
	hdlc_port_t *port = hdlc_to_port(hdlc);

	DEBUG("%s: scc_hdlc_ioctl()\n", dev->name);

#if	0	/* defined(CONFIG_HDLC_DEBUG_RINGS) */
	/* FIXME: Latter we can see what's usefull here -- Scop */
	if (cmd == SIOCDEVPRIVATE) {
		sca_dump_rings(hdlc);
		return 0;
	}
#endif

#if 0
	/* This was from the original Daris' driver */	
	switch (cmd) {
	case SCCHDLCIOCNOLOOPECHO:
		sccp->scc_gsmrl &= ~(SCC_GSMRL_DIAG_LOOP | SCC_GSMRL_DIAG_ECHO);
		break;

	case SCCHDLCIOCLOOP:
		sccp->scc_gsmrl &= ~SCC_GSMRL_DIAG_ECHO;
		sccp->scc_gsmrl |= SCC_GSMRL_DIAG_LOOP;
		break;

	case SCCHDLCIOCECHO:
		sccp->scc_gsmrl &= ~SCC_GSMRL_DIAG_LOOP;
		sccp->scc_gsmrl |= SCC_GSMRL_DIAG_ECHO;
		break;

	case SCCHDLCIOCLOOPECHO:
		sccp->scc_gsmrl |= SCC_GSMRL_DIAG_LOOP | SCC_GSMRL_DIAG_ECHO;
		break;

	case SCCHDLCIOCTXCLKFALL:
		sccp->scc_gsmrl &= ~SCC_GSMRL_TCI;
		break;

	case SCCHDLCIOCTXCLKRISE:
		sccp->scc_gsmrl |= SCC_GSMRL_TCI;
		break;

	default:
		break;
	}
#endif

	if (cmd != SIOCWANDEV)
		return hdlc_ioctl(dev, ifr, cmd);

	switch(ifr->ifr_settings.type) {
	case IF_GET_IFACE:
#if 0		
		ifr->ifr_settings.type = IF_IFACE_SYNC_SERIAL;
#else
		ifr->ifr_settings.type = board_get_iface(port);
#endif		
		if (ifr->ifr_settings.size < size) {
			ifr->ifr_settings.size = size; /* data size wanted */
			return -ENOBUFS;
		}
		if (copy_to_user(line, &port->settings, size))
 			return -EFAULT;
		return 0;

	case IF_IFACE_SYNC_SERIAL:
		if(!capable(CAP_NET_ADMIN))
			return -EPERM;

		if (copy_from_user(&new_line, line, size))
			return -EFAULT;

        if ( (new_line.clock_type != CLOCK_EXT &&
		      new_line.clock_type != CLOCK_TXINT ) ||
             (new_line.clock_type == CLOCK_TXINT &&
		      (new_line.clock_rate < 1000 ||
		       new_line.clock_rate > 5500000))
           )
		    return -EINVAL;

		if (new_line.loopback != 0 && new_line.loopback != 1)
			return -EINVAL;

		memcpy(&port->settings, &new_line, size); /* Update settings */
		scc_set_iface(port);
        return 0;

    default:
		return hdlc_ioctl(dev, ifr, cmd);
	}
	
	/* Success */
	return(0);
}


/*--------------------------------------------------------------
 * scc_hdlc_change_mtu()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *	(I) new_mtu - new size of the mtu
 *
 * Sets a new value for Maximum Transmission Unit (MTU).
 */

static int scc_hdlc_change_mtu(struct net_device* dev, int new_mtu)
{
	hdlc_port_t* port = dev_to_port(dev);
	volatile scc_hdlc_t* hp = (scc_hdlc_t*)dev->mem_start;
	volatile scc_t* sccp = (scc_t*)dev->base_addr;
	int result;
	int i;

	DEBUG("%s: scc_hdlc_change_mtu: new mtu=%d, old mtu=%d\n",
		dev->name, new_mtu, hp->mflr);

	/* Stop invalid changes */
	if ((new_mtu < 32) || (new_mtu > 65534))
		return(-EINVAL);

	/* Disable the Receiver */
	sccp->scc_gsmrl &= ~SCC_GSMRL_ENR;

	/* Set the MTU */
	hp->mflr = new_mtu;

	/* Set the RX Buffer maximum size
	 * Must be a multiple of 4 for the SCC in HDLC mode */
	new_mtu /= 4;
	new_mtu += 2; /* Add a little cushion */
	new_mtu *= 4;
	hp->genscc.scc_mrblr = new_mtu;

	/* Free the socket buffers associated with the RX buffer rings */
	for (i = 0; i < RX_RING_SIZE; ++i)
		{
		/* Free any allocated socket buffers */
		if (port->rx_skbuff[i])
			dev_kfree_skb(port->rx_skbuff[i]);

		port->rx_skbuff[i] = 0;
		}

	/* Allocate new buffers	*/
	if ((result = scc_hdlc_alloc_rx_ring(dev)))
		return(result);

	/* Restart Receiver	*/
	scc_hdlc_restart_rx(dev);

	/* Success */
	return(0);
}


/*--------------------------------------------------------------
 * scc_hdlc_alloc_rx_ring()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *
 * Allocate buffers for each buffer descriptor in the RX ring.
 * The receiver must be restarted after calling this function.
 */

static int scc_hdlc_alloc_rx_ring(struct net_device* dev)
{
	int i;
	volatile cbd_t* bdp;
	struct sk_buff*	newskb;
	hdlc_port_t* port = dev_to_port(dev);
	volatile scc_hdlc_t* hp = (scc_hdlc_t*)dev->mem_start;

	/* Initialize the RX Buffer Descriptor Ring */
	bdp = port->rx_bd_base;
	for (i = 0; i < RX_RING_SIZE; ++i, ++bdp)
		{
		newskb = scc_hdlc_alloc_skb(hp->genscc.scc_mrblr);
		if (!newskb)
			{
			printk("cannot alloc socket buffer\n");
			goto fail_rx_buffer;
			}

		/* Mark in use by the device -- Scop */
		newskb->dev = dev;
		
		/* Assign physical memory to each RX Buffer Descriptor */
		bdp->cbd_bufaddr = __pa(newskb->data);
		port->rx_skbuff[i] = newskb;
		}

	/* Success */
	return(0);

fail_rx_buffer:

	for (i = 0; i < RX_RING_SIZE; ++i)
		{
		/* Free any allocated socket buffers */
		if (port->rx_skbuff[i])
			dev_kfree_skb(port->rx_skbuff[i]);

		port->rx_skbuff[i] = 0;
		}

	return(-ENOMEM);
}


/*--------------------------------------------------------------
 * scc_hdlc_alloc_skb()
 *
 * Return:
 *	Pointer to a new socket buffer.
 *
 * Input Parameters:
 *	(I) mtu - specifies the size in bytes of the current MTU
 *
 * Allocates a new socket buffer large enough to receive a complete
 * frame of the current MTU size.  For the MPC8xx, the buffer is 
 * pre-flush from the cache so that data received from the SCC is
 * not corrupted later.
 */

static struct sk_buff* scc_hdlc_alloc_skb(int mtu)
{
	struct sk_buff*	newskb;

	/* Allocate a socket buffer */
	newskb = dev_alloc_skb(mtu);

#ifdef CONFIG_8xx
	if (newskb)
		{
		/* Push the data cache so updates are made now from cache
		 * (we don't want data clobbered later) */
		flush_dcache_range((ulong)newskb->data,
			(ulong)(newskb->data + /* newskb->len */ mtu));
		}
#endif

	return(newskb);
}

/*--------------------------------------------------------------
 * scc_hdlc_restart_rx()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *
 * Restarts (or starts for the first time) the Receiver of the
 * specified SCC.
 */

static void scc_hdlc_restart_rx(struct net_device* dev)
{
	volatile CPM_T* cp = cpmp;
	volatile scc_t* sccp = (scc_t*)dev->base_addr;
	hdlc_port_t* port = dev_to_port(dev);
	volatile cbd_t* bdp;
	int i;

	/* Disable the receiver */
	sccp->scc_gsmrl &= ~SCC_GSMRL_ENR;

	/* Re-INIT RX
	 * This command lets the CPM know it must reinitialize the SCC
	 * with the new parameter RAM values.
	 * This intializes the rbptr register */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
		CPM_CR_INIT_RX) | CPM_CR_FLG;
	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG,CPM_CR_FLG,scc_hdlc_restart_rx());

	/* Synchronize the local state machine with the CPM */
	port->rx_tail = 0;

	/* Initialize the RX Buffer Descriptor Ring */
	bdp = port->rx_bd_base;
	for (i = 0; i < RX_RING_SIZE; ++i, ++bdp)
		{
		/* Save the indexed pointer to the bd */
		port->rx_bd[i] = bdp;

		if (i == (RX_RING_SIZE - 1))
			bdp->cbd_sc = BD_HDLC_RX_EMPTY | BD_HDLC_RX_INTRPT |
				BD_HDLC_RX_WRAP;
		else
			bdp->cbd_sc = BD_HDLC_RX_EMPTY | BD_HDLC_RX_INTRPT;
		}

	/* Enable Receive processing
	 * Hunt Mode is automatically enetered */
	sccp->scc_gsmrl |= SCC_GSMRL_ENR;
}

/*--------------------------------------------------------------
 * scc_hdlc_restart_tx()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *
 * Restarts (or starts for the first time) the Transmitter of the
 * specified SCC.
 */

static void scc_hdlc_restart_tx(struct net_device* dev)
{
	volatile CPM_T* cp = cpmp;
	volatile scc_t* sccp = (scc_t*)dev->base_addr;
	hdlc_port_t* port = dev_to_port(dev);
	volatile cbd_t* bdp;
	int i;

	/* Send the Stop Transmission command to the CPM for this SCC */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
		CPM_CR_STOP_TX) | CPM_CR_FLG;
	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG,CPM_CR_FLG,scc_hdlc_restart_tx());

	/* Disable transmission */
	sccp->scc_gsmrl &= ~SCC_GSMRL_ENT;

	/* Re-INIT TX command
	 * This command lets the CPM know it must reinitialize the SCC
	 * with the new parameter RAM values.
	 * This intializes the tbptr register */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
		CPM_CR_INIT_TX) | CPM_CR_FLG;
	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG,CPM_CR_FLG,scc_hdlc_restart_tx());

	/* Synchronize the local state machine with the CPM */
	port->tx_head = port->dirty_tx = 0;
	port->tx_full = 0;

	/* Initialize the TX Buffer Descriptor Ring */
	bdp = port->tx_bd_base;

	for (i = 0; i < TX_RING_SIZE; ++i, ++bdp) {
		/* Save the indexed pointer to the bd */
		port->tx_bd[i] = bdp;
#if 1
		if (i == (TX_RING_SIZE - 1))
			{
			bdp->cbd_sc = BD_HDLC_TX_WRAP;
			}
		else
			{
			bdp->cbd_sc = 0;
			}
#else
		bdp->cbd_sc = 0;
#endif
		/* No pointers yet */
		bdp->cbd_bufaddr = 0;

		/* Free any allocated socket buffers */
		if (port->tx_skbuff[i])
			{
			dev_kfree_skb(port->tx_skbuff[i]);
			port->hdlc.stats.tx_dropped++;
			}

		port->tx_skbuff[i] = 0;
	}

	/* SCOP: After a HW reset, tx seems not to be starting automatically...	
    cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
        CPM_CR_RESTART_TX) | CPM_CR_FLG;
	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG,CPM_CR_FLG,scc_hdlc_restart_tx());
    */
    
	/* Enable Transmit processing */
	sccp->scc_gsmrl |= SCC_GSMRL_ENT;

    /* Record start time of transmission */
	dev->trans_start = jiffies;

    if (netif_queue_stopped(dev))
		netif_wake_queue(dev);

}

/*--------------------------------------------------------------
 * scc_hdlc_attach()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) hdlc - pointer to hdlc device
 *	(I) encoding - line encoding
 *  (I) parity - crc check type
 *
 * Used by HDLC Generic Layer to gain control over HDLC hw device
 */

static int scc_hdlc_attach ( hdlc_device *hdlc, unsigned short encoding, unsigned short parity )
{
	DEBUG("%s: scc_hdlc_attach()\n", hdlc_to_name(hdlc));

    /* FIXME: Setting currently fixed in SCC so we check and forget */
    if ( encoding != ENCODING_NRZ || parity != PARITY_CRC16_PR1_CCITT )
        return -EINVAL;
    return 0;
}

/*--------------------------------------------------------------
 * scc_hdlc_hard_start_xmit()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) skb - pointer to a buffer to transmit
 *	(I) dev - pointer to this driver's device structure
 *
 * Start transmission of a new socket buffer
 */

static int scc_hdlc_hard_start_xmit(struct sk_buff* skb, struct net_device* dev)
{
	hdlc_port_t* port = dev_to_port(dev);
	volatile cbd_t* bdp;
	int i;

	DEBUG("%s: scc_hdlc_hard_start_xmit()\n", dev->name);

	/* Not exactly sure what this code does.  It was illustrated
	 * on p321 of "Linux Device Drivers", 1st. edition */
	if (skb == NULL)  {	/* paranoid check */
		printk("%s: transmit null pointer\n", dev->name);
		return(-EINVAL); /* was (0) */
	}

#if 0
    /* SCOP: Now carrier status is actually link protocol status */
    /* Drop packet with error if we don't have carrier */
    if (!netif_carrier_ok(dev))
    {
        dev_kfree_skb ( skb );
        port->hdlc.stats.tx_errors++;
        port->hdlc.stats.tx_carrier_errors++;
        return 0;
    }
#endif

	/* Get the head of the queue */
	i = port->tx_head;
	bdp = port->tx_bd[i];

	if (bdp->cbd_sc & BD_HDLC_TX_READY)	 /* paranoid: shouldn't happen */
		{
		printk("%s: no tx bd's available\n", dev->name);

		netif_stop_queue(dev);

		/* We loose this packet */
		dev_kfree_skb(skb);
		port->hdlc.stats.tx_dropped++;

		return(-ENOMEM);
		}

#ifdef CONFIG_HDLC_DEBUG_PKT
	printk(KERN_DEBUG "%s TX(%i):", dev->name, skb->len);
	debug_frame(skb);
#endif

	/* Set buffer length and pointer */
	port->tx_skbuff[i] = skb;
	bdp->cbd_datlen = skb->len;

	bdp->cbd_bufaddr = __pa(skb->data); /* Use the physical address */

#ifdef CONFIG_8xx
	/* Push the data cache so the CPM does not get stale data */
	flush_dcache_range((ulong)skb->data, (ulong)(skb->data + skb->len));
#endif

	/* Clear status flags */
	bdp->cbd_sc &= ~BD_HDLC_TX_STATS;

	spin_lock_irq(&port->lock);

	/* We found a bd we can use, bump the head index */
	++port->tx_head;
	if (port->tx_head >= TX_RING_SIZE)
		port->tx_head = 0;

	/* Record start time of transmission */
	dev->trans_start = jiffies;

	/* Send the packet */
	/* if (i == (TX_RING_SIZE - 1))
		bdp->cbd_sc |= BD_HDLC_TX_READY | BD_HDLC_TX_INTRPT |
			BD_HDLC_TX_LAST | BD_HDLC_TX_CRC | BD_HDLC_TX_WRAP;
	else */
		bdp->cbd_sc |= BD_HDLC_TX_READY | BD_HDLC_TX_INTRPT |
			BD_HDLC_TX_LAST | BD_HDLC_TX_CRC;

	if (port->tx_head == port->dirty_tx)  {
		netif_stop_queue(dev);
		port->tx_full = 1;
	}

	spin_unlock_irq(&port->lock);

	/* Update statistics */
	port->hdlc.stats.tx_bytes += skb->len;
	port->hdlc.stats.tx_packets++;
#ifdef CONFIG_EXT_DEVSTATS
    port->hdlc.ext_stats.tx_packet_time = jiffies;
#endif

	/* Done */
	return(0);
}

/*--------------------------------------------------------------
 * scc_hdlc_timeout()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *
 * Transmission timeout, we could have missed an interrupt.
 * Re-wake transmission.
 */

static void scc_hdlc_timeout(struct net_device *dev)
{
	hdlc_port_t* port = dev_to_port(dev);

	DEBUG("%s: scc_hdlc_timeout()\n", dev->name);

	printk("%s: transmit timed out, restarting tx.\n", dev->name);
	port->hdlc.stats.tx_errors++;
	port->hdlc.stats.tx_aborted_errors++;
	
	/* Restart the transmitter */
	scc_hdlc_restart_tx(dev);

/* maybe?
	printk("%s: transmit timed out.\n", dev->name);
	port->hdlc.stats.tx_errors++;
	if (!port->tx_full)
		netif_wake_queue(dev);
*/
}


/*--------------------------------------------------------------
 * scc_hdlc_interrupt()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *  (I) irq    - not used
 *	(I) dev_id - pointer to this driver's device structure
 * 	(I) regs   - not used
 *
 * Interrupt service routine, deals with Tx & Rx asynchronous
 * events and errors.
 */

#ifdef CONFIG_8xx
static void scc_hdlc_interrupt(void *dev_id, struct pt_regs * regs)
#else /* 8260 */
static void scc_hdlc_interrupt(int irq, void *dev_id, struct pt_regs * regs)
#endif
{
	struct net_device* dev = (struct net_device*)dev_id;
	hdlc_port_t* port = dev_to_port(dev);
	volatile scc_t* sccp = (scc_t*)dev->base_addr;
	ushort int_events;

	DEBUG("%s: Entering ISR\n", dev->name);

	/* Get the interrupt events */
	int_events = sccp->scc_scce;

	/* DEBUG("%s: sccp->scc_scce=0x%x\n", dev->name, int_events); */

	/* Clear off these event */
	sccp->scc_scce = int_events;

	if (int_events & SCCM_BSY)
		{
		port->hdlc.stats.rx_errors++;
		port->hdlc.stats.rx_dropped++;
		DEBUG("%s: SCCM_BSY=%ld\n", dev->name, port->hdlc.stats.rx_errors);
		}

	if (int_events & (SCCM_RX|SCCM_RXF))
		{
		/* Queue the task into the immediate bottom half queue */
		port->rx_bh_tq.routine = scc_hdlc_rx_bh;
		port->rx_bh_tq.data = (void*)dev;
		queue_task(&port->rx_bh_tq, &tq_immediate);
		mark_bh(IMMEDIATE_BH);
		}

	if (int_events & SCCM_TXE)
		{
		port->hdlc.stats.tx_errors++;
		DEBUG("%s: SCCM_TXE=%ld\n", dev->name, port->hdlc.stats.tx_errors);
		}

	if (int_events & (SCCM_TX | SCCM_TXE))
		{
		/* Queue the task into the immediate bottom half queue */
		port->tx_bh_tq.routine = scc_hdlc_tx_bh;
		port->tx_bh_tq.data = (void*)dev;
		queue_task(&port->tx_bh_tq, &tq_immediate);
		mark_bh(IMMEDIATE_BH);
		}

}


/*--------------------------------------------------------------
 * scc_hdlc_tx_bh()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) dev_id - pointer to this driver's device structure
 *
 * This is one of the bottom half handlers for scc_hdlc_interrupt().
 * It completes the service of TX buffers that have been transmitted.
 */

static void scc_hdlc_tx_bh(void* dev_id)
{
	struct net_device* dev = (struct net_device*)dev_id;
	hdlc_port_t* port = dev_to_port(dev);
	volatile scc_hdlc_t* hp = (scc_hdlc_t*)dev->mem_start;
	volatile cbd_t* bdp;
	int i;
	int must_restart;

	DEBUG("%s: scc_hdlc_tx_bh()\n", dev->name);

	/* Get some CPM error counters */
	if ((i = hp->disfc + hp->crcec + hp->nmarc) != 0) {
		DEBUG("%s: scc_hdlc_tx_bh(): rxerr=0x%x\n",
			dev->name, i);
		port->hdlc.stats.rx_errors += i;
		port->hdlc.stats.rx_dropped += i;
		/* FIXME: non-atomic! we may be loosing some counting here,
		         but better count some then none... -- Scop*/
		hp->disfc = hp->crcec =	hp->nmarc = 0;
	}
	
	spin_lock(&port->lock);	/* FIXME: should it be spin_lock_irq??? */
	
	must_restart = 0;
	
	/* Scan TX bd's and cleanup	*/
	while (1)  {
		/* Get the tail of the queue */
		i = port->dirty_tx;
		bdp = port->tx_bd[i];

		/* Don't touch buffers waiting to be transmitted */
		if (bdp->cbd_sc & BD_HDLC_TX_READY)	 {
			break;
		}

		/* Tx ring is empty */
		if ((i==port->tx_head) && (port->tx_full == 0))	 {
			break;
		}

		/* We found a bd with data, bump the tail index */
		if (++port->dirty_tx >= TX_RING_SIZE)
			port->dirty_tx = 0;

		/* Unused bd if its pointer is null; paranoid */
		if (port->tx_skbuff[i])	{
			/* Free the socket buffer */
			dev_kfree_skb(port->tx_skbuff[i]);
			port->tx_skbuff[i] = 0;
		}

		/* We have at least one TX Buff Descr available now */
		if (port->tx_full) {
			port->tx_full = 0;
			if (netif_queue_stopped(dev)) {
				netif_wake_queue(dev);
			}
		}

		/* Accumulate TX error statistics */
		if (bdp->cbd_sc & BD_HDLC_TX_UN)
			{
			must_restart = 1;
			DEBUG("%s: BD_HDLC_TX_UN\n", dev->name);
			port->hdlc.stats.tx_fifo_errors++;
			}

		if (bdp->cbd_sc & BD_HDLC_TX_CL)
			{
			must_restart = 1;
			DEBUG("%s: BD_HDLC_TX_CL\n", dev->name);
			port->hdlc.stats.tx_carrier_errors++;
			}

	}
	
	if (must_restart) {
		volatile CPM_T* cp = cpmp;

		/* Some transmit errors cause the transmitter to shut
		 * down.  We now issue a restart transmit.  Since the
		 * errors close the BD and update the pointers, the restart
		 * _should_ pick up without having to reset any of our
		 * pointers either.
		*/
		cp->cp_cpcr = MK_CR_CMD(
			SCC_Params[port->scc_param_index].chan_cmd,
			CPM_CR_RESTART_TX) | CPM_CR_FLG;
		WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG,CPM_CR_FLG,scc_hdlc_tx_bh());
	}
	
	spin_unlock(&port->lock); /* FIXME: should it be spin_unlock_irq??? */

}


/*--------------------------------------------------------------
 * scc_hdlc_rx_bh()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) dev_id - pointer to this driver's device structure
 *
 * This is one of the bottom half handlers for scc_hdlc_interrupt().
 * It completes the service of RX buffers that have been received.
 */

static void scc_hdlc_rx_bh(void* dev_id)
{
	struct net_device* dev = (struct net_device*)dev_id;
	hdlc_port_t* port = dev_to_port(dev);
	volatile scc_hdlc_t* hp = (scc_hdlc_t*)dev->mem_start;
	volatile cbd_t* bdp;
	struct sk_buff*	skb;
	struct sk_buff*	newskb;
	ushort pkt_len;
	ushort rx_stat;
	int rxerr;
	int i;

	DEBUG("%s: scc_hdlc_rx_bh()\n", dev->name);

	while (1)  {
		
		/* Get the tail of the queue */
		i = port->rx_tail;
		bdp = port->rx_bd[i];

		if (bdp->cbd_sc & BD_HDLC_RX_EMPTY)
			return; /* Nothing more to do */

		/* We found a bd with data, bump the tail index */
		if (++port->rx_tail >= RX_RING_SIZE)
			port->rx_tail = 0;

		/* Get the status of the RX packet */
		rx_stat = bdp->cbd_sc;

		/* Update error statistics */
		rxerr = 0;

		if (rx_stat & BD_HDLC_RX_DE)
			{
			rxerr = 2;
			DEBUG("%s: scc_hdlc_rx_bh(): RX_DE\n", dev->name);
			}

		if (rx_stat & BD_HDLC_RX_LG)
			{
			port->hdlc.stats.rx_length_errors++;
			rxerr = 2;
			DEBUG("%s: scc_hdlc_rx_bh(): RX_LG\n", dev->name);
			}

		if (rx_stat & BD_HDLC_RX_NO)
			{
			port->hdlc.stats.rx_frame_errors++;
			rxerr = 2;
			DEBUG("%s: scc_hdlc_rx_bh(): RX_NO\n", dev->name);
			}

		if (rx_stat & BD_HDLC_RX_AB)
			{
			rxerr = 2;
			DEBUG("%s: scc_hdlc_rx_bh(): RX_AB\n", dev->name);
			}

		if (rx_stat & BD_HDLC_RX_CRC)
			{
			port->hdlc.stats.rx_crc_errors++;
			rxerr = 2;
			DEBUG("%s: scc_hdlc_rx_bh(): RX_CRC\n", dev->name);
			}

		if (rx_stat & BD_HDLC_RX_OV)
			{
			port->hdlc.stats.rx_over_errors++;
			rxerr = 1;
			DEBUG("%s: scc_hdlc_rx_bh(): RX_OV\n", dev->name);
			}

		if (rx_stat & BD_HDLC_RX_CD)
			{
			port->hdlc.stats.rx_missed_errors++;
			rxerr = 1;
			DEBUG("%s: scc_hdlc_rx_bh(): RX_CD\n", dev->name);
			}

		if (rxerr)
			{
			DEBUG("%s: scc_hdlc_rx_bh(): rxerr=0x%x\n",
				dev->name, rxerr);
			port->hdlc.stats.rx_errors++;
			}

		if (rxerr <= 1) {
			/* Get a socket buffer to replace the one just consumed */
			newskb = scc_hdlc_alloc_skb(hp->genscc.scc_mrblr);
			if (!newskb)
				{
				printk("%s: no rx buffers, dropping packet\n",
					dev->name);
				port->hdlc.stats.rx_dropped++;
            	
				/* Return here so we don't get out of sync w/cpm ???
				return;	*/
				}
			else {
				/* Mark in use by the device -- Scop */
				newskb->dev = dev;
		
				/* Get the socket data buffer */
				skb = port->rx_skbuff[i];
				port->rx_skbuff[i] = newskb;
				pkt_len = bdp->cbd_datlen;
    	
				/* Set the physical address of the new socket buffer */
				bdp->cbd_bufaddr = __pa(newskb->data);
			}
    	
		}
		else
			newskb = NULL; /* We'll simply drop packets with critical errors
								--Scop */
		
		/* Clear status flags */
		bdp->cbd_sc &= ~BD_HDLC_RX_STATS;

		/* Mark ready to receive data */
		bdp->cbd_sc |= BD_HDLC_RX_EMPTY;

		if (newskb)  {
			/* Set the length of the received data, ()minus the CRC ???) */
			skb_put(skb, pkt_len-2);  /* -2 ??? */

#ifdef CONFIG_HDLC_DEBUG_PKT
			printk(KERN_DEBUG "%s RX(%i):", dev->name,
				skb->len);
			debug_frame(skb);
#endif
			/* Update statistics */
			port->hdlc.stats.rx_packets++;
			port->hdlc.stats.rx_bytes += pkt_len;
#ifdef CONFIG_EXT_DEVSTATS
            port->hdlc.ext_stats.rx_packet_time = jiffies;
#endif

			/* Set the protocol type */
			skb->protocol=htons(ETH_P_HDLC);

            /* Point to the mac address (first HDLC byte) */
			skb->mac.raw = skb->data;
        	skb->dev = dev;
        	dev->last_rx = jiffies;

#ifdef CONFIG_EXT_DEVSTATS
            /* SCOP: This MUST go here, AFTER eth_type_trans() return ??? 
            if (skb->pkt_type == PACKET_BROADCAST)
	            fep->ext_stats.rx_broadcasts++; */
#endif

            /* Send it to the upper layer */
			netif_rx(skb);

		}

	}

}

module_init(scc_hdlc_init);

module_exit(scc_hdlc_exit);  /* FIXME: not really usefull, right now
								-- Scop */

/* Module Configuration Parameters */
MODULE_AUTHOR("Ricardo Scop, R SCOP Consult. <scop@plugin.com.br>");
MODULE_DESCRIPTION("SCC HDLC Driver for MPC8xx and 82xx");
EXPORT_NO_SYMBOLS;
MODULE_LICENSE("GPL");



