/*
 * HDLC driver for Motorola MPC8xx in synchronous/asynchronous mode.
 *      (c) Copyright 2002 Rodolfo Giometti <giometti@ascensit.com>
 *      (c) Copyright 2002 Christian Pellegrin <pellegrin@ascensit.com>
 *      (c) Copyright 2002 Ascensit <support@ascensit.com>
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 *      Neither Rodolfo Giometti nor Christian Pellegrin nor Ascensit Srl.
 *      admit liability nor provide warranty for any of this software.
 *      This material is provided "AS-IS" and at no charge.
 *
 * Based on ./enet.c.
 * Original copyright messages:
 *
 * Copyright (c) 1997 Dan Malek (dmalek@jlc.net)
 *
 * Based on linux/drivers/net/wan/dscc4.c.
 * Original copyright messages:
 *
 * This software may be used and distributed according to the terms of the
 * GNU General Public License. 
 * 
 * The author may be reached as romieu@cogenit.fr.
 * Specific bug reports/asian food will be welcome.
 *      
 */

/*
 * ChangeLog
 * 
 * 22/10/2002 chri
 *     * reworked the structure to have a BH
 *     * support for kernel FRF
 *     * support for Alyseo specific LEDs and I/O lines
 *     * many bugfixes
 *
 * 08/04/2002		Rodolfo Giometti
 *     * BRG/CLK params added.
 *     * 16/32 bit CRC-CCIT for sync mode added.
 *
 * 05/04/2002		Rodolfo Giometti
 *     * Interrupt disabling on close() method.
 *
 * 22/03/2002		Christian Pellegrin
 *     * Big buffers added.
 *     * Interrupt enabling on open() method.
 *
 * 14/03/2002		Rodolfo Giometti
 *     * Synchronous HDLC added.
 *
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/ptrace.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/hdlc.h>
#include <linux/spinlock.h>

#include <asm/8xx_immap.h>
#include <asm/pgtable.h>
#include <asm/mpc8xx.h>
#include <asm/bitops.h>
#include <asm/uaccess.h>
#include <asm/commproc.h>

#define CONFIG_KFRF 1
#ifdef CONFIG_KFRF
#include "kfrf.h"
#endif

/* IRQ test */
//#define IRQ_TEST 1

/* selta custom version */
#define SELTA 1

/* new clocks routed via the CPLD */
#define NEW_CLOCKS

/* define this tu use high 3k of 850 dpram*/
//#define DPRAM_850 1

#ifdef SELTA
/* Leds & C. definitions */
#define SELTA_C107_PORT immap->im_cpm.cp_pbdat
#define SELTA_C107_MASK (0x00008000)
/* C107 e' DSR */
#define SELTA_SBIL_PORT immap->im_ioport.iop_pcdat
#define SELTA_SBIL_MASK (0x0010)

#define SELTA_TERM_PORT immap->im_ioport.iop_pcdat
#define SELTA_TERM_MASK (0x002)

#define SELTA_C106_PORT immap->im_ioport.iop_pcdat
#define SELTA_C106_MASK (0x0400)
/* C107 e' CTS */

#define SELTA_TESTMODE_PORT immap->im_cpm.cp_pbdat
#define SELTA_TESTMODE_MASK (0x00000001)
/* testmode will be handled by eocd */

#define SELTA_POWER_PORT immap->im_ioport.iop_pcdat
#define SELTA_POWER_MASK (0x100)

#include "alyseo_ioctls.h"

static int cts_mode = 1;
MODULE_PARM(cts_mode, "i");
MODULE_PARM_DESC(cts_mode, "0 off 1 on");

static int dcd_mode = 1;
MODULE_PARM(dcd_mode, "i");
MODULE_PARM_DESC(dcd_mode, "0 off 1 on");

static int dsr_mode = 1;
MODULE_PARM(dsr_mode, "i");
MODULE_PARM_DESC(dsr_mode, "0 off 1 on");

static int term_mode = 0;
MODULE_PARM(term_mode, "i");
MODULE_PARM_DESC(term_mode, "0 off 1 on");

static int loopled_mode = 0;
MODULE_PARM(loopled_mode, "i");
MODULE_PARM_DESC(loopled_mode, "0 off 1 on");

/* timer for LEDS */
static struct timer_list poll_timer = {{NULL, NULL}, 0L, 0L, NULL};
static int hy_tx=0;
static int hy_rx=0;

/* prints that we are out of buffers */
static int say_once_busy = 0;

/* debug status */
static int hdlc_loopback_software = 0;
static int hdlc_loopback_hardware = 0;
extern int hdlc_loopback;

static void ehdlc_timer(unsigned long dummy) {
  volatile immap_t *immap = (immap_t *)(mfspr(IMMR)&0xFFFF0000);
  
  if (say_once_busy > 0)
    say_once_busy --;

  if (hy_tx>0)
    hy_tx--;
  else {
    /* TX off */
    immap->im_ioport.iop_padat |=  0x80;
  }

  if (hy_rx>0)
    hy_rx--;
  else {
    /* RX off */
    immap->im_ioport.iop_padat |=  0x040;
  }

  if (!(immap->im_ioport.iop_pcdat & 0x4))				/* RTS read*/
    immap->im_ioport.iop_padat |= 0x0001; /* RTS on */
  else
    immap->im_ioport.iop_padat &= ~0x0001; /* RTS off */

  if (!(immap->im_cpm.cp_pbdat & 0x00004000)) /* DTR read */
    immap->im_ioport.iop_padat |= 0x0002; /* DTR on */
  else
    immap->im_ioport.iop_padat &= ~0x0002; /* DTR off */

  if (dcd_mode)
    immap->im_ioport.iop_pcdat |= 0x800; /* assert dcd */
  else
    immap->im_ioport.iop_pcdat &= ~0x800; /* de-assert dcd */

  if (cts_mode)
    immap->im_ioport.iop_pcdat &= ~0x400; /* cts on */
  else
    immap->im_ioport.iop_pcdat |= 0x400; /* cts off */

  if (dsr_mode)
    SELTA_C107_PORT |= SELTA_C107_MASK; /* asserted DSR */
  else
    SELTA_C107_PORT &= ~SELTA_C107_MASK; /* de-asserted DSR */

  if (term_mode)
    SELTA_TERM_PORT &= ~SELTA_TERM_MASK; /* de-asserted TERM */
  else
    SELTA_TERM_PORT |= SELTA_TERM_MASK; /* asserted TERM */

   if (loopled_mode)
     immap->im_cpm.cp_pbdat &= ~0x00000001; /* loopled on */
   else
     immap->im_cpm.cp_pbdat |=  0x00000001; /* loopled off */

  /* TODO: fill hdlc_loopback_hardware */
#if THE_BOARD_IS_SOLDIERED
  if (!(immap->im_ioport.iop_pcdat & 0x00000200))				/* LOOP read*/
    hdlc_loopback_hardware = 1; /* LOOP on */
  else
    hdlc_loopback_hardware = 0; /* LOOP off */
#else
  hdlc_loopback_hardware = 0; /* LOOP off */
#endif

  if (hdlc_loopback_hardware || hdlc_loopback_software) 
    hdlc_loopback = 1;
  else
    hdlc_loopback = 0;

  poll_timer.expires = jiffies + HZ/2; /* schedule in half second */
  add_timer(&poll_timer);
}

#endif

/* 
   These are parameters to setup the clocks/brgs of the SCC at runtime
 */

#ifndef NEW_CLOCKS
static int tclk = 2;
MODULE_PARM(tclk, "i");
MODULE_PARM_DESC(tclk, "Tx clock (BRG) - can be 1 or 2 or 3 (default is 2)");
#if !defined(CONFIG_EHDLC_ASYNC)
static int rclk = 3;
MODULE_PARM(rclk, "i");
MODULE_PARM_DESC(rclk, "Rx clock (CLK) - can be 1, 2, 3 or 4 (default is 3)");
#endif
#endif

#ifdef NEW_CLOCKS
unsigned long clocks_base = 0xff018000;
unsigned long clocks_size = 32*1024;

void 
set_clocks(int n, int from_dte, int polarity) 
     /* n is n*64, from_dte if clock from outside, polarity 0 is direct, 1 is inverted */
{
  volatile unsigned char * addr = (unsigned char *) clocks_base;
  volatile immap_t *immap = (immap_t *)(mfspr(IMMR)&0xFFFF0000);
  volatile cpm8xx_t *cp = cpmp;

  cp->cp_sicr &= ~0x00ff0000;   /* we reset GR3 and SC3 */
  immap->im_ioport.iop_papar |= 0x0a00;
  immap->im_ioport.iop_padir &= ~0xa00;
  if (from_dte) {
    cp->cp_sicr |= 0x002f0000;
  }
  else {
    cp->cp_sicr |= 0x003f0000;
  }
  //printk("sicr is %x, %p is %x, %p is %x\n", cp->cp_sicr, &addr[0], n-1, &addr[1], polarity); /* REMOVEME */
  addr[0] = n-1;
  //printk("done, speed should be ok\n"); /* REMOVEME */
  addr[1] = polarity;
  //printk("done, also polarity should be ok\n"); /* REMOVEME */
}
#endif

/* The number of Tx and Rx buffers.  These are allocated from the page
 * pool.  The code may assume these are power of two, so it is best
 * to keep them that size.
 * We don't need to allocate pages for the transmitter.  We just use
 * the skbuffer directly.
 */

#define CONFIG_EHDLC_BIG_BUFFERS 1

#ifdef CONFIG_EHDLC_BIG_BUFFERS
#define CPM_EHDLC_RX_PAGES	32
#define CPM_EHDLC_RX_FRSIZE	2048
#define CPM_EHDLC_RX_FRPPG	(PAGE_SIZE / CPM_EHDLC_RX_FRSIZE)
#define RX_RING_SIZE		(CPM_EHDLC_RX_FRPPG * CPM_EHDLC_RX_PAGES)
#define TX_RING_SIZE		64	/* Must be power of two */
#define TX_RING_MOD_MASK	63	/*   for this to work */
#else
#define CPM_EHDLC_RX_PAGES	4
#define CPM_EHDLC_RX_FRSIZE	2048
#define CPM_EHDLC_RX_FRPPG	(PAGE_SIZE / CPM_EHDLC_RX_FRSIZE)
#define RX_RING_SIZE		(CPM_EHDLC_RX_FRPPG * CPM_EHDLC_RX_PAGES)
#define TX_RING_SIZE		8	/* Must be power of two */
#define TX_RING_MOD_MASK	7	/*   for this to work */
#endif

/* The CPM stores dest/src/type, data, and checksum for receive packets.
 */
#define PKT_MAXBLR_SIZE		1520

/* The CPM buffer descriptors track the ring buffers.  The rx_bd_base and
 * tx_bd_base always point to the base of the buffer descriptors.  The
 * cur_rx and cur_tx point to the currently available buffer.
 * The dirty_tx tracks the current buffer that is being sent by the
 * controller.  The cur_tx and dirty_tx are equal under both completely
 * empty and completely full conditions.  The empty/ready indicator in
 * the buffer descriptor determines the actual condition.
 */
struct scc_hdlc_private {
   /* The saved address of a sent-in-place packet/buffer, for skfree(). */
   struct	sk_buff* tx_skbuff[TX_RING_SIZE];
   ushort	skb_cur;
   ushort	skb_dirty;

   /* CPM dual port RAM relative addresses */
   cbd_t	*rx_bd_base;		/* Address of Rx and Tx buffers. */
   cbd_t	*tx_bd_base;
   cbd_t	*cur_rx, *cur_tx;	/* The next free ring entry */
   cbd_t	*dirty_tx;		/* The ring entries to be free()ed. */
   scc_t	*sccp;
   uint	tx_full;
   spinlock_t lock;

   /* CPM BRG & CLK used */
   char		tclk;
   char		rclk;

   /* HDLC data */
   hdlc_device hdlc;
   sync_serial_settings settings;
   unsigned short encoding;
   unsigned short parity;
   u32 pad __attribute__ ((aligned (4)));
} scc_private_data;

/* We use SCC3 for the HDLC.
 * See enet.c to know howto extend */
#define CPM_CR_EHDLC	CPM_CR_CH_SCC3
#define PROFF_EHDLC	PROFF_SCC3
#define SCC_EHDLC	2		/* Index, not number! */
#define CPMVEC_EHDLC	CPMVEC_SCC3

/* During a receive, the cur_rx points to the current incoming buffer.
 * When we update through the ring, if the next incoming buffer has
 * not been given to the system, we just set the empty indicator,
 * effectively tossing the packet.
 */
static int scc_hdlc_rx(struct net_device *dev)
{
   volatile struct scc_hdlc_private *cep = &scc_private_data;
   volatile cbd_t *bdp;
   struct sk_buff *skb;
   ushort pkt_len;
   volatile immap_t *immap = (immap_t *)(mfspr(IMMR)&0xFFFF0000);
   ulong len;

   //printk("rx bh\n"); //REMOVEME

#ifdef SELTA
   /* RX on */
   hy_rx = 2;
   immap->im_ioport.iop_padat &=  ~0x040;
#endif

   /* First, grab all of the stats for the incoming packet.
    * These get messed up if we get called due to a busy condition */
   bdp = cep->cur_rx;

   for (;;) {
     int error = 0;

     //printk("rx bh bdp %p\n", bdp); //REMOVEME
     
      if (bdp->cbd_sc&BD_HDLC_RX_EMPTY)
	 break;
		
      /* Since we have allocated space to hold a complete frame, both
       * the first and last indicators should be set */
      if ((bdp->cbd_sc&(BD_HDLC_RX_FIRST|BD_HDLC_RX_LAST)) !=
	  (BD_HDLC_RX_FIRST|BD_HDLC_RX_LAST))
	printk("CPM HDLC: rcv is not first+last\n");
      
#if !defined(CONFIG_EHDLC_ASYNC)
      if (bdp->cbd_sc&BD_HDLC_RX_LG) {   /* Frame length error */
         dev_to_hdlc(dev)->stats.rx_length_errors++;
	 error = 1;
      }
      if (bdp->cbd_sc&BD_HDLC_RX_NO) {  /* Frame alignment */
	dev_to_hdlc(dev)->stats.rx_frame_errors++;
	error = 1;
      }
#endif
      if (bdp->cbd_sc&BD_HDLC_RX_CR) {  /* CRC Error */
         dev_to_hdlc(dev)->stats.rx_crc_errors++;
	 error = 1;
      }
      if (bdp->cbd_sc&BD_HDLC_RX_OV) {  /* FIFO overrun */
         dev_to_hdlc(dev)->stats.rx_over_errors++;
	 error = 1;
      }
   
      /* Report frame too short, carrier detect lost and Rx abort
       * sequence as a frame error (and in async mode we consider BRK
       * and BOF errors also).
       * On these errors, the BD is closed, but we don't know what we
       * have in the buffer.  So, just drop this frame on the floor but
       * we use different stats entry */
      if ((bdp->cbd_datlen < 2) ||
	  (bdp->cbd_sc&(BD_HDLC_RX_CD|BD_HDLC_RX_AB
#if defined(CONFIG_EHDLC_ASYNC)
                        |BD_AHDLC_RX_BR|BD_AHDLC_RX_BO
#endif
         )))
         dev_to_hdlc(dev)->stats.rx_frame_errors++;
      else {
         /* Process the incoming frame */
	if (!error) {
	  dev_to_hdlc(dev)->stats.rx_packets++;
	  pkt_len = bdp->cbd_datlen;
	  dev_to_hdlc(dev)->stats.rx_bytes += pkt_len;
   
         /* This does 16 byte alignment, much more than we need.
          * The packet length includes FCS, but we don't want to
          * include that when passing upstream as it messes up
          * bridging applications */
	  pkt_len -= 2;                   /* we use 16-bit for CRC */
#define CONFIG_KFRF 1
#ifdef CONFIG_KFRF
	  skb = dev_alloc_skb(pkt_len + 16 + 15);
	  if (skb) {
	    if ((len = (ulong)skb->data & (ulong)0x0f)){
	      skb_reserve(skb, 16 - len);
	    }
	    skb_reserve(skb, 16);
	  }
#else
	  skb = dev_alloc_skb(pkt_len);
#endif

	  //printk("rx bh skb %p\n"); //REMOVEME
	  if (skb == NULL) {
   	    //CHRI-MUTE printk("%s: Memory squeeze (ehdlc), dropping packet.\n", dev->name);
   	    dev_to_hdlc(dev)->stats.rx_dropped++;
	  }
	  else {
   	    skb_put(skb, pkt_len);	/* Make room */
   	    eth_copy_and_sum(skb, (unsigned char *)__va(bdp->cbd_bufaddr),
			     pkt_len, 0);
	    if (netif_running(dev))
	      skb->protocol = htons(ETH_P_IP);
	    skb->dev = dev;
   	    dev_to_hdlc(dev)->netif_rx(skb);
	  }
	}
      }
   
      /* Clear the status flags for this buffer */
#if defined(CONFIG_EHDLC_ASYNC)
      bdp->cbd_sc &= ~BD_AHDLC_RX_STATS;
#else
      bdp->cbd_sc &= ~BD_HDLC_RX_STATS;
#endif
      
      /* Mark the buffer empty */
      bdp->cbd_sc |= BD_HDLC_RX_EMPTY;
   
      /* Update BD pointer to next entry */
      if (bdp->cbd_sc & BD_HDLC_RX_WRAP)
	bdp = cep->rx_bd_base;
      else
	bdp++;
   }
   cep->cur_rx = (cbd_t *)bdp;
   
   return 0;
}

/* The interrupt handler bhs. */
static void scc_hdlc_interrupt_rx_bh(void *param)
{
   hdlc_device *hdlc = &scc_private_data.hdlc;

   scc_hdlc_rx(hdlc_to_dev(hdlc));
}

static void scc_hdlc_interrupt_tx_bh(void *param)
{
   volatile struct scc_hdlc_private *cep = &scc_private_data;
   hdlc_device *hdlc = &scc_private_data.hdlc;
   volatile cbd_t *bdp;
   int must_restart;
   volatile cpm8xx_t *cp;
   volatile scc_hdlc_t *ep;

   cp = cpmp;		
   ep = (scc_hdlc_t *)(&cp->cp_dparam[PROFF_EHDLC]);

   /* Check for a transmit error.  The manual is a little unclear
    * about this, so the debug code until I get it figured out.  It
    * appears that if TXE is set, then TXB is not set.  However,
    * if carrier sense is lost during frame transmission, the TXE
    * bit is set, "and continues the buffer transmission normally."
    * I don't know if "normally" implies TXB is set when the buffer
    * descriptor is closed.....trial and error :-).
    */

   /* Transmit OK, or non-fatal error. Update the buffer descriptors */
   spin_lock(&cep->lock);
   bdp = cep->dirty_tx;
   while ((bdp->cbd_sc&BD_HDLC_TX_READY) == 0) {
     if ((bdp == cep->cur_tx) && (cep->tx_full == 0)) /* CHRI: test this test: why tx_full part? */
       break;

#if !defined(CONFIG_EHDLC_ASYNC)
     if (bdp->cbd_sc&BD_HDLC_TX_UN)         /* Underrun */
       hdlc->stats.tx_fifo_errors++;
#endif
     if (bdp->cbd_sc&BD_HDLC_TX_CT)         /* CTS lost */
       hdlc->stats.tx_carrier_errors++;

     hdlc->stats.tx_packets++;

     /* Free the sk buffer associated with this last transmit */
     //     atomic_set(&cep->tx_skbuff[cep->skb_dirty]->users,1); /* CHRI-USERS */
     dev_kfree_skb_irq(cep->tx_skbuff[cep->skb_dirty]);
     cep->skb_dirty = (cep->skb_dirty+1)&TX_RING_MOD_MASK;

     /* Update pointer to next buffer descriptor to be transmitted */
     if (bdp->cbd_sc&BD_HDLC_TX_WRAP)
       bdp = cep->tx_bd_base;
     else
       bdp++;

     /* I don't know if we can be held off from processing these
      * interrupts for more than one frame time.  I really hope
      * not.  In such a case, we would now want to check the
      * currently available BD (cur_tx) and determine if any
      * buffers between the dirty_tx and cur_tx have also been
      * sent.  We would want to process anything in between that
      * does not have BD_HDLC_TX_READY set */
     
     /* Since we have freed up a buffer, the ring is no longer
      * full */
     if (cep->tx_full) {
       cep->tx_full = 0;
       if (netif_queue_stopped(hdlc_to_dev(hdlc)))
	 netif_wake_queue(hdlc_to_dev(hdlc));
     }

     cep->dirty_tx = (cbd_t *)bdp;
   }

   if (must_restart) {
     volatile cpm8xx_t *cp;

     /* Some transmit errors cause the transmitter to shut
      * down.  We now issue a restart transmit.  Since the
      * errors close the BD and update the pointers, the restart
      * _should_ pick up without having to reset any of our
      * pointers either */
     cp = cpmp;
     cp->cp_cpcr = mk_cr_cmd(CPM_CR_EHDLC, CPM_CR_RESTART_TX)|CPM_CR_FLG;
     while (cp->cp_cpcr&CPM_CR_FLG);
   }
   spin_unlock(&cep->lock);

   return;
}

struct tq_struct rx_bh;
struct tq_struct tx_bh;

static void scc_hdlc_interrupt(void *devptr, struct pt_regs *regs)
{
   volatile struct scc_hdlc_private *cep = &scc_private_data;
   hdlc_device *hdlc = &scc_private_data.hdlc;
   ushort int_events;
#ifdef IRQ_TEST
   static int int_rx = 0, int_tx = 0, int_busy = 0, int_tot = 0;
   static long int_jiffies = 0;
#endif

#ifdef IRQ_TEST
   if ( (int_jiffies + HZ) < jiffies ) {
     printk("CPM EHDLC: in %ld jiffies got %d rx int, %d tx int, %d busy int, %d total int\n", jiffies-int_jiffies, int_rx, int_tx, int_busy, int_tot);
     int_rx = int_tx = int_busy = int_tot = 0;
     int_jiffies = jiffies;
   }
   int_tot++;
#endif
   //printk("hdlc irq\n"); //REMOVEME
 
   /* Get the interrupt events that caused us to be here */
   int_events = cep->sccp->scc_scce;
   cep->sccp->scc_scce = int_events;
 
   /* Handle receive event in its own bh */
   if (int_events&SCCE_HDLC_RXF) {
#ifdef IRQ_TEST
     int_rx++;
#endif
     queue_task(&rx_bh, &tq_immediate);
     mark_bh(IMMEDIATE_BH);
   }

   /* Handle tx event in its own bh */
   if (int_events&(SCCE_HDLC_TXE|SCCE_HDLC_TXB)) {
#ifdef IRQ_TEST
     int_tx++;
#endif
     queue_task(&tx_bh, &tq_immediate);
     mark_bh(IMMEDIATE_BH);
   }

   /* Check for receive busy, i.e. packets coming but no place to
    * put them.  This "can't happen" because the receive interrupt
    * is tossing previous frames */
   if (int_events&SCCE_HDLC_BSY) {
#ifdef IRQ_TEST
     int_busy++;
#endif
      hdlc->stats.rx_dropped++;
      if (!say_once_busy) {
	//CHRI-MUTE printk(KERN_INFO "CPM HDLC: Out of buffers\n");
	say_once_busy = 3;
      }
   }

   return;
}

static int scc_encoding_setting(struct net_device *dev)
{
   /* Nothing to do here */

   return 0;
}

static int scc_clock_setting(struct net_device *dev)
{
#ifndef NEW_CLOCKS
   volatile struct scc_hdlc_private *cep = &scc_private_data;
   uint bps = cep->settings.clock_rate; 
#endif

#ifndef NEW_CLOCKS
   //if (bps < 50 || bps > 115200)
   //   return -EINVAL;

   m8xx_cpm_setbrg(cep->tclk-1, bps/16);

   cep->settings.clock_rate = 16*m8xx_cpm_getbrg(cep->tclk-1);
   printk(KERN_INFO "%s: BRG%d clock adjusted from %08d to %08d\n",
	  dev->name, cep->tclk, bps, cep->settings.clock_rate);

#endif
   return 0;
}

static int scc_loopback_setting(struct net_device *dev)
{
   /* Nothing to do here */

   return 0;
}

static int scc_crc_setting(struct net_device *dev)
{
#if !defined(CONFIG_EHDLC_ASYNC)   /* fixed CRC for async mode */
   volatile struct scc_hdlc_private *cep = &scc_private_data;
   uint parity = cep->settings.clock_rate; 
   volatile scc_hdlc_t *ep = (scc_hdlc_t *) dev->base_addr;
 
   if (parity == PARITY_CRC16_PR0_CCITT ||
       parity == PARITY_CRC16_PR1_CCITT) {
      ep->sen_cpres = 0x0000ffff;
      ep->sen_cmask = 0x0000f0b8;
      printk(KERN_INFO "%s: using 16-bit CRC-CCITT\n", dev->name);
   }
   if (parity == PARITY_CRC32_PR0_CCITT ||
       parity == PARITY_CRC32_PR1_CCITT) {
      ep->sen_cpres = 0xffffffff;
      ep->sen_cmask = 0xdebb20e3;
      printk(KERN_INFO "%s: using 32-bit CRC-CCITT\n", dev->name);
   }
#endif

   return 0;
}

static int scc_set_iface(struct net_device *dev)
{
   struct {
      int (*action)(struct net_device *);
   } *p, do_setting[] = {
      { scc_encoding_setting },
      { scc_clock_setting },
      { scc_loopback_setting },
      { scc_crc_setting },
      { NULL }
   };
   int ret = 0;

   for (p = do_setting; p->action; p++) {
       if ((ret = p->action(dev)) < 0)
       break;
   }

   return ret;
}


static int scc_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
   struct scc_hdlc_private *cep = &scc_private_data;
   struct if_settings *if_s = &ifr->ifr_settings;
   const size_t size = sizeof(cep->settings);
   int ret = 0;
#ifdef SELTA
   struct led_status_s led_status;
#ifdef NEW_CLOCKS
   struct clock_status_s clocks;
#endif
#endif

   switch(ifr->ifr_settings.type) {
      case IF_GET_IFACE : {
         if_s->type = IF_IFACE_SYNC_SERIAL;

	 if (if_s->data_length == 0)
	    return 0;
	 if (if_s->data_length < size)
	    return -ENOMEM;
	 if (copy_to_user(if_s->data, &cep->settings, size))
	    return -EFAULT;

	 if_s->data_length = size;

	 break;
      }

      case IF_IFACE_SYNC_SERIAL : {
         if (!capable(CAP_NET_ADMIN))
            return -EPERM;
         if (if_s->data_length != size)
            return -ENOMEM;
         if (copy_from_user(&cep->settings, if_s->data, size))
            return -EFAULT;

         ret = scc_set_iface(dev);

	 break;
      }


#ifdef SELTA
   case IF_ALY_LEDS: {
     volatile immap_t *immap = (immap_t *)(mfspr(IMMR)&0xFFFF0000);

     if (!capable(CAP_NET_ADMIN))
       return -EPERM;
     if (copy_from_user(&led_status, if_s->data, sizeof(led_status)))
            return -EFAULT;

     if (!(immap->im_ioport.iop_pcdat & 0x4))				/* RTS read*/
       led_status.rts = 1; /* RTS on */
     else
       led_status.rts = 0; /* RTS off */
     
     if (!(immap->im_cpm.cp_pbdat & 0x00004000)) /* DTR read */
       led_status.dtr = 1; /* DTR on */
     else
       led_status.dtr = 0; /* DTR off */

     if (led_status.set_cts)
       cts_mode = led_status.cts;

     if (led_status.set_dsr)
       dsr_mode = led_status.dsr;

     if (led_status.set_dcd)
       dcd_mode = led_status.dcd;

     if (led_status.set_term)
       term_mode = led_status.term;

     if (led_status.set_loopled)
       loopled_mode = led_status.loopled;

     led_status.cts = cts_mode;
     led_status.dsr = dsr_mode;
     led_status.dcd = dcd_mode;
     led_status.term = term_mode;

     led_status.loopback = hdlc_loopback_hardware;

     if (copy_to_user(if_s->data, &led_status, sizeof(led_status)))
       return -EFAULT;
     break;
   }
   case IF_ALY_LOOP_ON:
     hdlc_loopback_software = 1;
     break;
   case IF_ALY_LOOP_OFF:
     hdlc_loopback_software = 0;
     break;
#ifdef NEW_CLOCKS
   case IF_ALY_CLOCKS: {
     if (!capable(CAP_NET_ADMIN))
       return -EPERM;
     if (copy_from_user(&clocks, if_s->data, sizeof(clocks)))
       return -EFAULT;
     set_clocks(clocks.n, clocks.from_dte, clocks.polarity);
     break;
   }
#endif
#endif

      default: {
         ret = hdlc_ioctl(dev, ifr, cmd);
      }
   }

   return ret;
} 

static int scc_hdlc_attach(hdlc_device *hdlc, unsigned short encoding,
              unsigned short parity)
{
   volatile struct scc_hdlc_private *cep = &scc_private_data;

   if (encoding != ENCODING_NRZ &&
       encoding != ENCODING_NRZI &&
       encoding != ENCODING_FM_MARK &&
       encoding != ENCODING_FM_SPACE &&
       encoding != ENCODING_MANCHESTER)
      return -EINVAL;

   if (parity != PARITY_CRC16_PR0_CCITT &&
       parity != PARITY_CRC16_PR1_CCITT &&
       parity != PARITY_CRC32_PR0_CCITT &&
       parity != PARITY_CRC32_PR1_CCITT)
      return -EINVAL;

   cep->encoding = encoding;
   cep->parity = parity;

   return 0;
}

#ifdef SELTA
typedef void (*kfrf_stats_push_t) (struct sk_buff *);
kfrf_stats_push_t kfrf_stats_push = NULL;
EXPORT_SYMBOL(kfrf_stats_push);

#endif

static int scc_hdlc_xmit(struct sk_buff *skb, struct net_device *dev)
{
   volatile struct scc_hdlc_private *cep = &scc_private_data;
   volatile cbd_t *bdp, *next_bdp;
   int ret;
   volatile immap_t *immap = (immap_t *)(mfspr(IMMR)&0xFFFF0000);

#ifdef SELTA
   /* TX on */
   hy_tx = 2;
   immap->im_ioport.iop_padat &=  ~0x80;

   if (kfrf_stats_push)
     kfrf_stats_push(skb);
#endif

   /* Fill in a Tx ring entry */
   bdp = cep->cur_tx;

   /* If this was the last BD in the ring, start at the beginning again */
   if (bdp->cbd_sc&BD_HDLC_TX_WRAP)
      next_bdp = cep->tx_bd_base;
   else
      next_bdp = bdp+1;

   /* CHRI: this is the logic to stop the device in time */
   {
#define NEED_FREE 2
      /* TUNE this number! means how many buffers must be free */
     int i;
     volatile cbd_t *next = cep->cur_tx;

     /* TODO: could optimize this loop */
     if (next->cbd_sc & BD_HDLC_TX_WRAP)
       next = cep->tx_bd_base;
     else
       next = next + 1;
     for(i=0; i<NEED_FREE; i++) {
       if (next->cbd_sc & BD_HDLC_TX_WRAP)
	 next = cep->tx_bd_base;
       else
	 next = next + 1;
       if ( next_bdp == cep->dirty_tx) {
	 netif_stop_queue(dev);
	 break;
       }
     }
   }

   //   if (bdp->cbd_sc&BD_HDLC_TX_READY) {
   /* AAAAARRRGGGGGGHHHHHHHH this leads to terrible race condition */
   if ( next_bdp == cep->dirty_tx) {
     /* Ooops.  All transmit buffers are full.  Bail out.
      * This should not happen, since cep->tx_busy should be set */
     //printk("%s: tx queue full! (users is %d).\n", dev->name, atomic_read(&skb->users));
     netif_stop_queue(dev);
     cep->tx_full = 1;
     ret = 1;
     goto tx_finished;
   }

   /* Clear all of the status flags */
#if defined(CONFIG_EHDLC_ASYNC)
   bdp->cbd_sc &= ~BD_AHDLC_TX_STATS;
#else
   bdp->cbd_sc &= ~BD_HDLC_TX_STATS;
#endif

   /* Set buffer length and buffer pointer */
   bdp->cbd_datlen = skb->len;

   bdp->cbd_bufaddr = __pa(skb->data);

   /* Save skb pointer */
   cep->tx_skbuff[cep->skb_cur] = skb;

   dev_to_hdlc(dev)->stats.tx_bytes += skb->len;
   cep->skb_cur = (cep->skb_cur+1)&TX_RING_MOD_MASK;
	
   /* Push the data cache so the CPM does not get stale memory data */
   flush_dcache_range((unsigned long)(skb->data),
		      (unsigned long)(skb->data+skb->len));

   spin_lock_irq(&cep->lock);

   /* Send it on its way.  Tell CPM its ready, interrupt when done,
    * its the last BD of the frame, and to put the CRC on the end (in
    * sunc HDLC) */
   bdp->cbd_sc |= (BD_HDLC_TX_READY|BD_HDLC_TX_INTR|BD_HDLC_TX_LAST);
#if !defined(CONFIG_EHDLC_ASYNC)
   bdp->cbd_sc |= BD_HDLC_TX_TC;
#endif

   dev->trans_start = jiffies;

   cep->cur_tx = (cbd_t *) next_bdp;

   spin_unlock_irq(&cep->lock);

   ret = 0;
 tx_finished:

   return ret;
}

static int scc_open(struct net_device *dev)
{
   int ret;
   volatile cpm8xx_t *cp;
   volatile scc_t *sccp;

   /* Get pointer to Communication Processor... */
   cp = cpmp;

   /* ...and to internal registers */
   sccp = (volatile scc_t *)(&cp->cp_scc[SCC_EHDLC]);

   /* I should reset the ring buffers here, but I don't yet know
    * a simple way to do that */

   if (netif_running(dev)) {
      printk(KERN_WARNING "%s: already running\n", dev->name);
      return 0;
   }

   /* Check the open method */
   if (dev_to_hdlc(dev)->open) {
      ret = dev_to_hdlc(dev)->open(dev_to_hdlc(dev));
      if (ret < 0)
         return ret;
   }

   /* And last, enable the transmit and receive processing */
   sccp->scc_gsmrl |= (SCC_GSMRL_ENR|SCC_GSMRL_ENT);

   MOD_INC_USE_COUNT;

   return 0;					/* Always succeed */
}

static int scc_close(struct net_device *dev)
{
   volatile cpm8xx_t *cp;
   volatile scc_t *sccp;

   /* Get pointer to Communication Processor... */
   cp = cpmp;

   /* ...and to internal registers */
   sccp = (volatile scc_t *)(&cp->cp_scc[SCC_EHDLC]);

   /* Check the stop method */
   if (dev_to_hdlc(dev)->stop)
      dev_to_hdlc(dev)->stop(dev_to_hdlc(dev));

   /* And last, disable the transmit and receive processing */
   sccp->scc_gsmrl &= ~(SCC_GSMRL_ENR|SCC_GSMRL_ENT);

   MOD_DEC_USE_COUNT;

   return 0;
}

/* ------------------------------------------------------------------------ */

#define SCC_EHDLC_RBASE_DPALLOC		0x00010
#define SCC_EHDLC_TBASE_DPALLOC		0x00020
#define SCC_EHDLC_HDLC_PAGE_ALLOC	0x00100
#define SCC_EHDLC_HDLC_DEV_REG		0x00800
static __initdata int scc_hdlc_steps;

/* Remove whatever the scc_hdlc_init() function requested. :-) */
static void __exit scc_hdlc_exit(void)
{
   struct scc_hdlc_private *cep = &scc_private_data;
   hdlc_device *hdlc = &scc_private_data.hdlc;
   volatile scc_hdlc_t *ep = (scc_hdlc_t *) hdlc_to_dev(hdlc)->base_addr;
   unsigned long mem_addr;
   pte_t *pte;
   volatile cbd_t *bdp;
   int i,j;

   if (scc_hdlc_steps&SCC_EHDLC_HDLC_DEV_REG) {
      unregister_hdlc_device(hdlc);
      cpm_free_handler(CPMVEC_EHDLC);
   }
   if (scc_hdlc_steps&SCC_EHDLC_HDLC_PAGE_ALLOC) {
      bdp = cep->rx_bd_base;
      for (i = 0; i < CPM_EHDLC_RX_PAGES; i++) {
	 /* Get the memory address */
         mem_addr = (uint) __va(bdp->cbd_bufaddr);

         /* Make it cached */
         pte = va_to_pte(mem_addr);
         pte_val(*pte) &= ~_PAGE_NO_CACHE;
         flush_tlb_page(init_mm.mmap, mem_addr);
   
         /* Deallocate the page */
         __free_page((struct page *) mem_addr);

         /* Jump to the next page */
         for (j = 0; j < CPM_EHDLC_RX_FRPPG; j++)
   	    bdp++;
      }
   }
#ifndef DPRAM_850
   if (scc_hdlc_steps&SCC_EHDLC_TBASE_DPALLOC)
      m8xx_cpm_dpfree(ep->sen_genscc.scc_tbase);
   if (scc_hdlc_steps&SCC_EHDLC_RBASE_DPALLOC)
      m8xx_cpm_dpfree(ep->sen_genscc.scc_rbase);
#endif
}

/* Initialize the CPM HDLC on SCC.  If EPPC-Bug loaded us, or performed
 * some other network I/O, a whole bunch of this has already been set up.
 * It is no big deal if we do it again, we just have to disable the
 * transmit and receive to make sure we don't catch the CPM with some
 * inconsistent control information.
 */
static int __init scc_hdlc_init(void)
{
   struct scc_hdlc_private *cep = &scc_private_data;
   hdlc_device *hdlc = &scc_private_data.hdlc;
   int i, j;
   unsigned long mem_addr;
   pte_t *pte;
   bd_t	*bd;
   volatile cbd_t *bdp;
   volatile cpm8xx_t *cp;
   volatile scc_t *sccp;
   volatile scc_hdlc_t *ep;
   volatile immap_t *immap;

#ifndef NEW_CLOCKS
#if !defined(CONFIG_EHDLC_ASYNC)
#if 0
   unsigned short tclk2pa[] = {0 /* invalid! */,
      0x0100, 0x0400, 0, 0};
   unsigned short rclk2pa[] = {0 /* invalid! */,
      0x0100, 0x0200, 0x0400, 0x0800};
   unsigned int rclk2sicr[] = {0 /* invalid! */,
      0x04, 0x5, 0x6, 0x7};
#else
   unsigned short tclk2pa[] = {0 /* invalid! */,
      0x0100, 0x0400, 0, 0};
   unsigned short rclk2pa[] = {0 /* invalid! */,
      0x0100, 0x0200, 0x0100, 0x0100};
   unsigned int rclk2sicr[] = {0 /* invalid! */,
      0x04, 0x5, 0x1, 0x2};
#endif
#endif
   unsigned int tclk2sicr[] = {0 /* invalid! */,
      0x0, 0x1, 0x2};
#endif

   /* Get pointer to Communication Processor... */
   cp = cpmp;

   /* ...and to internal registers */
   immap = (immap_t *)(mfspr(IMMR)&0xFFFF0000);

#ifdef SELTA 
   immap->im_cpm.cp_pbpar &=  ~(SELTA_C107_MASK);
   immap->im_cpm.cp_pbdir |=   (SELTA_C107_MASK);

   immap->im_ioport.iop_pcpar &=  ~(SELTA_SBIL_MASK | SELTA_TERM_MASK | 
				    SELTA_C106_MASK | SELTA_POWER_MASK);
   immap->im_ioport.iop_pcdir |=   (SELTA_SBIL_MASK | SELTA_TERM_MASK | 
				    SELTA_C106_MASK | SELTA_POWER_MASK);
   immap->im_ioport.iop_pcso  &=  ~(SELTA_SBIL_MASK | SELTA_TERM_MASK | 
				    SELTA_C106_MASK | SELTA_POWER_MASK);

   /* this is for alyseo102 after recabling */
   /* dcd led as output */
   immap->im_ioport.iop_pcpar &= ~0x800;
   immap->im_ioport.iop_pcdir |= 0x800;
   immap->im_ioport.iop_pcso &= ~0x800;
   if (dcd_mode)
     immap->im_ioport.iop_pcdat |= 0x800; /* assert dcd */
   else
     immap->im_ioport.iop_pcdat &= ~0x800; /* de-assert dcd */
   /* cts led as output */
   immap->im_ioport.iop_pcpar &= ~0x400;
   immap->im_ioport.iop_pcdir |= 0x400;
   immap->im_ioport.iop_pcso &= ~0x400;
   if (cts_mode)
     immap->im_ioport.iop_pcdat &= ~0x400; /* cts on */
   else
     immap->im_ioport.iop_pcdat |= 0x400; /* cts off */
   if (dsr_mode)
     SELTA_C107_PORT |= SELTA_C107_MASK; /* asserted DSR */
   else
     SELTA_C107_PORT &= ~SELTA_C107_MASK; /* de-asserted DSR */
   /* these are valid for new boards only */
   /* TX data 0 on 1 off*/
   immap->im_ioport.iop_papar &= ~0x80;
   immap->im_ioport.iop_padir |= 0x80;
   immap->im_ioport.iop_paodr &= ~0x80;
   immap->im_ioport.iop_padat |=  0x80;
   /* RX data 0 on 1 off*/
   immap->im_ioport.iop_papar &= ~0x40;
   immap->im_ioport.iop_padir |= 0x40;
   immap->im_ioport.iop_paodr &= ~0x40;
   immap->im_ioport.iop_padat |=  0x40;
   /* DTR data 0 off 1 on*/
   immap->im_ioport.iop_papar &= ~0x0002;
   immap->im_ioport.iop_padir |= 0x0002;
   immap->im_ioport.iop_paodr &= ~0x0002;
   immap->im_ioport.iop_padat &= ~0x0002;
   /* RTS data 0 off 1 on*/
   immap->im_ioport.iop_papar &= ~0x0001;
   immap->im_ioport.iop_padir |= 0x0001;
   immap->im_ioport.iop_paodr &= ~0x0001;
   immap->im_ioport.iop_padat &= ~0x0001;
   /* LOOPLED */
   immap->im_cpm.cp_pbpar &=   ~0x00000001;
   immap->im_cpm.cp_pbodr &=   ~0x00000001;
   immap->im_cpm.cp_pbdir |=    0x00000001;
   if (loopled_mode)
     immap->im_cpm.cp_pbdat &= ~0x00000001; /* loopled on */
   else
     immap->im_cpm.cp_pbdat |=  0x00000001; /* loopled off */
   /* DTR read */
   immap->im_cpm.cp_pbpar &=  ~0x00004000;
   immap->im_cpm.cp_pbodr &=   ~0x00004000;
   immap->im_cpm.cp_pbdir &=  ~0x00004000;
   /* RTS read */
   immap->im_ioport.iop_pcpar &= ~0x4;
   immap->im_ioport.iop_pcdir &= ~0x4;
   immap->im_ioport.iop_pcso  &= ~0x4;
   /* LOOP read */
   immap->im_cpm.cp_pbpar &=   ~0x00000200;
   immap->im_cpm.cp_pbodr &=   ~0x00000200;
   immap->im_cpm.cp_pbdir &=   ~0x00000200;

   poll_timer.expires = jiffies + HZ/2;
   poll_timer.function = ehdlc_timer;
   add_timer(&poll_timer);
#endif

   bd = (bd_t *)__res;

   /* Init the spinlock */
   spin_lock_init(&cep->lock);

   /* Get pointer to SCC area in parameter RAM */
   ep = (scc_hdlc_t *)(&cp->cp_dparam[PROFF_EHDLC]);

   /* And another to the SCC register area */
   sccp = (volatile scc_t *)(&cp->cp_scc[SCC_EHDLC]);
   cep->sccp = (scc_t *)sccp;		/* Keep the pointer handy */

   /* Disable receive and transmit in case EPPC-Bug started it */
   sccp->scc_gsmrl &= ~(SCC_GSMRL_ENR|SCC_GSMRL_ENT);

   /* Cookbook style from the MPC860 manual.....
    * Not all of this is necessary if EPPC-Bug has initialized
    * the network.
    * So far we are lucky, all board configurations use the same
    * pins, or at least the same I/O Port for these functions.....
    * It can't last though......
    */

   /* Configure port B pins for TXD and RXD */
   immap->im_cpm.cp_pbpar |=  (PB_HDLC_RXD|PB_HDLC_TXD);
   immap->im_cpm.cp_pbdir &= ~(PB_HDLC_RXD|PB_HDLC_TXD);
#if defined(CONFIG_EHDLC_ODRAIN)
   immap->im_cpm.cp_pbodr |=  PB_HDLC_TXD;   /* open drain */
#elseif
   immap->im_cpm.cp_pbodr &= ~PB_HDLC_TXD;   /* normal */
#endif
   immap->im_cpm.cp_pbodr &= ~PB_HDLC_RXD;

#ifndef SELTA
   /* Configure port C pins for RTS3, CTS3 and CD3 */
   immap->im_ioport.iop_pcpar |=  PC_HDLC_RTS;
   immap->im_ioport.iop_pcpar &= ~(PC_HDLC_CTS|PC_HDLC_CD);
   immap->im_ioport.iop_pcdir &= ~(PC_HDLC_RTS|PC_HDLC_CTS|PC_HDLC_CD);
#if defined(CONFIG_EHDCL_WITHCS)
   immap->im_ioport.iop_pcso  |=  PC_HDLC_CTS|PC_HDLC_CD;  /* CTS & CD = SCC3 */
#else
   immap->im_ioport.iop_pcso  &= ~(PC_HDLC_CTS|PC_HDLC_CD);/* CTS & CD = 0 */
#endif
#endif

#ifdef NEW_CLOCKS
   {
     //printk("going to map or3 to %x-%x\n", clocks_base, clocks_base+clocks_size); /* REMOVEME */
     /* 32kByte size, BIH asserted, SCY=16 wait state, 8 bits registers */
     immap->im_memctl.memc_or3 = 0xffff0930;
     immap->im_memctl.memc_br3 = clocks_base|0x0401;
     //immap->im_memctl.memc_or3 = 0xffff87f6;
     //immap->im_memctl.memc_br3 = clocks_base|0x0401;
     //printk("done mapping, now ioremap_nocache\n"); /* REMOVEME */

     if (!ioremap_nocache(clocks_base, clocks_size)) {
       printk(KERN_ERR "scit: unable to ioremap() memory %#010lx-%#010lx\n",
	      clocks_base, clocks_base+clocks_size);
       return -EINVAL;
     }
     //printk("done, clocks should be on\n"); /* REMOVEME */
   }
   set_clocks(32, 0, 0);
#else  /* NEW_CLOCKS */
#if !defined(CONFIG_EHDLC_ASYNC)
   if (tclk < 1 || tclk > 3) {
      printk(KERN_ERR "hdlc: invalid tclk parameter [%d]\n", tclk);
      scc_hdlc_exit();
      return -ENOMEM;
   }
   if (rclk < 1 || rclk > 4) {
      printk(KERN_ERR "hdlc: invalid rclk parameter [%d]\n", rclk);
      scc_hdlc_exit();
      return -ENOMEM;
   }
   cep->tclk = tclk;
   cep->rclk = rclk;

   if (tclk==3) {
     immap->im_cpm.cp_pbpar |=  0x8;
     immap->im_cpm.cp_pbdir &= ~0x8;
   }
   else {
     immap->im_ioport.iop_papar |= tclk2pa[tclk];
     immap->im_ioport.iop_padir |= tclk2pa[tclk];
   }

   immap->im_ioport.iop_papar |= rclk2pa[rclk];
   immap->im_ioport.iop_padir &= ~rclk2pa[rclk];

   //printk(KERN_INFO "hdlc: using BRG%d [%#06x]\n", tclk, tclk2pa[tclk]);
   //printk(KERN_INFO "hdlc: using CLK%d [%#06x]\n", rclk, rclk2pa[rclk]);
#endif

   /* Configure Serial Interface clock routing.
    * First, clear all SCC3 bits to zero, then set the ones we want */
   cp->cp_sicr &= ~0x00ff0000;   /* we reset GR3 and SC3 */

#if defined(CONFIG_EHDLC_ASYNC)
   cp->cp_sicr |= (tclk2sicr[tclk]<<16)|(tclk2sicr[tclk]<<19);
#else
   cp->cp_sicr |= (tclk2sicr[tclk]<<16)|(rclk2sicr[rclk]<<19);
#endif
   //printk("REMOVEME - %x %x \n", tclk2sicr[tclk], rclk2sicr[rclk]);
   //printk("REMOVEME - %x %x \n", tclk2sicr[tclk]<<16, rclk2sicr[rclk]<<19);
   //printk("REMOVEME - CP_PADIR %x\n", immap->im_ioport.iop_padir);
   //printk("REMOVEME - CP_PAPAR %x\n", immap->im_ioport.iop_papar);
   //printk("REMOVEME - CP_PBDIR %x\n", immap->im_cpm.cp_pbdir);
   //printk("REMOVEME - CP_PBPAR %x\n", immap->im_cpm.cp_pbpar);
   //printk("REMOVEME - CP_SICR %x\n", cp->cp_sicr);

#endif /* NEW_CLOCKS */

   /* The initialization of SDCR has already been done by the
    * communication processor initialization */

   /* Allocate space for the buffer descriptors in the DP ram.
    * These are relative offsets in the DP ram address space.
    * Initialize base addresses for the buffer descriptors.
    */

#ifdef DPRAM_850
   printk("size for rx bd: %d\n", sizeof(cbd_t)*RX_RING_SIZE);
   cep->rx_bd_base = (cbd_t *) &cp->res19[1024];
#else
   i = m8xx_cpm_dpalloc(sizeof(cbd_t)*RX_RING_SIZE);
   if (i == CPM_DP_NOSPACE) {
      printk(KERN_ERR "hdlc: unable to alloc into dualport RAM\n");
      scc_hdlc_exit();
      return -ENOMEM;
   }
   scc_hdlc_steps |= SCC_EHDLC_RBASE_DPALLOC;
   ep->sen_genscc.scc_rbase = i;
   cep->rx_bd_base = (cbd_t *)&cp->cp_dpmem[i];
#endif

#ifdef DPRAM_850
   printk("size for tx bd: %d\n", sizeof(cbd_t)*TX_RING_SIZE);
   cep->tx_bd_base = (cbd_t *) &cp->res19[2048];
#else
   i = m8xx_cpm_dpalloc(sizeof(cbd_t)*TX_RING_SIZE);
   if (i == CPM_DP_NOSPACE) {
      printk(KERN_ERR "hdlc: unable to alloc into dualport RAM\n");
      scc_hdlc_exit();
      return -ENOMEM;
   }
   scc_hdlc_steps |= SCC_EHDLC_TBASE_DPALLOC;
   ep->sen_genscc.scc_tbase = i;
   cep->tx_bd_base = (cbd_t *)&cp->cp_dpmem[i];
#endif

   cep->dirty_tx = cep->cur_tx = cep->tx_bd_base;
   cep->cur_rx = cep->rx_bd_base;

   /* FIXME... */
   /* Issue init Rx BD command for SCC.
    * Manual says to perform an Init Rx parameters here.  We have
    * to perform both Rx and Tx because the SCC may have been
    * already running.
    * In addition, we have to do it later because we don't yet have
    * all of the BD control/status set properly.
      cp->cp_cpcr = mk_cr_cmd(CPM_CR_HDLC, CPM_CR_INIT_RX) | CPM_CR_FLG;
      while (cp->cp_cpcr & CPM_CR_FLG);
   */

   /* Initialize function code registers for big-endian */
   ep->sen_genscc.scc_rfcr = SCC_EB;
   ep->sen_genscc.scc_tfcr = SCC_EB;

   /* Set maximum bytes per receive buffer.
    * This appears to be an HDLC frame size, not the buffer
    * fragment size.  It must be a multiple of four */
   ep->sen_genscc.scc_mrblr = PKT_MAXBLR_SIZE;

   /* Set CRC preset and mask to 16-bit */
   ep->sen_cpres = 0x0000ffff;
   ep->sen_cmask = 0x0000f0b8;

#if defined(CONFIG_EHDLC_ASYNC)
   /* Clear the Zero register */
   ep->sen_zero = 0;

   /* Init for PPP the BOF, EOF and the ESC characters */
   ep->sen_bof = 0x7e;
   ep->sen_eof = 0x7e;
   ep->sen_esc = 0x7d;
#else
   ep->sen_disfc = 0;   /* Discarded frame counter*/
   ep->sen_crcec = 0;   /* CRC Error counter */
   ep->sen_abtsc = 0;   /* Abort sequence counter */
   ep->sen_nmark = 0;   /* Nonmatching address receiver counter */
   ep->sen_retrc = 0;   /* Frame retrasmission counter */
   ep->sen_mflr = PKT_MAXBLR_SIZE;   /* Max frame length register */
#endif

   ep->sen_rfthr = 0x0001;		/* allow int. after each frame */
   
#if defined(CONFIG_EHDLC_ASYNC)
   /* FIXME: is that setting right? */
   ep->sen_txctl_tbl = 0;
   ep->sen_rxctl_tbl = 0;

   ep->sen_nof = 0;                     /* Use 1 opening flag */
#endif

   /* Now allocate the host memory pages and initialize the
    * buffer descriptors */
   bdp = cep->tx_bd_base;
   for (i = 0; i < TX_RING_SIZE; i++) {
      /* Initialize the BD for every fragment in the page */
      bdp->cbd_sc = 0;
      bdp->cbd_datlen = 0;
      bdp->cbd_bufaddr = 0;
      bdp++;
   }

   /* Set the last buffer to wrap */
   bdp--;
   bdp->cbd_sc |= BD_SC_WRAP;

   bdp = cep->rx_bd_base;
   for (i = 0; i < CPM_EHDLC_RX_PAGES; i++) {
      /* Allocate a page */
      mem_addr = __get_free_page(GFP_KERNEL);

      /* Make it uncached */
      pte = va_to_pte(mem_addr);
      pte_val(*pte) |= _PAGE_NO_CACHE;
      flush_tlb_page(init_mm.mmap, mem_addr);

      /* Initialize the BD for every fragment in the page */
      for (j = 0; j < CPM_EHDLC_RX_FRPPG; j++) {
         bdp->cbd_sc = BD_HDLC_RX_EMPTY|BD_HDLC_RX_INTR;
         bdp->cbd_bufaddr = __pa(mem_addr);
	 mem_addr += CPM_EHDLC_RX_FRSIZE;
	 bdp++;
      }
   }
   scc_hdlc_steps |= SCC_EHDLC_HDLC_PAGE_ALLOC;

   /* Set the last buffer to wrap */
   bdp--;
   bdp->cbd_sc |= BD_SC_WRAP;

#if !defined(CONFIG_EHDLC_ASYNC)
   /* hmask.... receive all, we hope */
   ep->sen_hmask = 0;
#endif

   /* Let's re-initialize the channel now.  We have to do it later
    * than the manual describes because we have just now finished
    * the BD initialization */
   cp->cp_cpcr = mk_cr_cmd(CPM_CR_EHDLC, CPM_CR_INIT_TRX)|CPM_CR_FLG;
   while (cp->cp_cpcr&CPM_CR_FLG);   /* FIXME: Brrr... */

   cep->skb_cur = cep->skb_dirty = 0;

   sccp->scc_scce = 0xffff;	/* Clear any pending events */

   /* Install our interrupt handler */
   cpm_install_handler(CPMVEC_EHDLC, scc_hdlc_interrupt, NULL);

   /* Enable interrupts for transmit error, complete frame
    * received, and any transmit buffer we have also set the
    * interrupt flag */
   sccp->scc_sccm = (SCCE_HDLC_TXE|SCCE_HDLC_RXF|SCCE_HDLC_TXB);

   /* Set GSMR_H to enable all normal operating modes.
    * Set GSMR_L to enable HDLC.
    */
   sccp->scc_gsmrh = 0;
#if defined(CONFIG_EHDLC_ASYNC)
   sccp->scc_gsmrl = SCC_GSMRL_MODE_AHDLC;
#else
   sccp->scc_gsmrl = SCC_GSMRL_MODE_HDLC;
#endif

   /* Set processing mode.  Use one opening and one closing flag, 16-bit
    * CCITT-CRC and prevent multiple frames in FIFO */
#if defined(CONFIG_EHDLC_ASYNC)
   sccp->scc_pmsr = SCU_PMSR_CL;   /* character len = 8 bits */
#else
   sccp->scc_pmsr = 0;
#if 0
#ifdef CRC32
   sccp->scc_pmsr |= 0x0800;
#endif
#endif
#endif

   /* Save some pointers into device structure */
   hdlc_to_dev(hdlc)->base_addr = (unsigned long)ep;
   hdlc_to_dev(hdlc)->irq = 0;   /* FIXME */
   hdlc_to_dev(hdlc)->init = NULL;   /* FIXME */
   hdlc_to_dev(hdlc)->open = scc_open;
   hdlc_to_dev(hdlc)->stop = scc_close;
   hdlc_to_dev(hdlc)->do_ioctl = scc_ioctl;

   /* The CPM HDLC specific entries in the device structure */
   hdlc->xmit = scc_hdlc_xmit;
   hdlc->attach = scc_hdlc_attach;
   hdlc->proto = IF_PROTO_HDLC;

   /* Now we are ready to register the HDLC device */
   if (register_hdlc_device(hdlc) < 0) {
      printk(KERN_ERR "hdlc: unable to register hdlc device\n");
      scc_hdlc_exit();
      return -ENODEV;
   }
   scc_hdlc_steps |= SCC_EHDLC_HDLC_DEV_REG;

   /* init the bhs */
   rx_bh.routine = scc_hdlc_interrupt_rx_bh;
   tx_bh.routine = scc_hdlc_interrupt_tx_bh;

   printk("%s: CPM HDLC Version 0.2 on SCC%d\n",
	  hdlc_to_name(hdlc), SCC_EHDLC+1);
#if defined(CONFIG_EHDLC_ODRAIN)
   printk("%s: TXD open drain mode\n", hdlc_to_name(hdlc));
#else
   printk("%s: TXD normal mode\n", hdlc_to_name(hdlc));
#endif
#if defined(CONFIG_EHDLC_WITHCS)
   printk("%s: CTS and CD input enabled\n", hdlc_to_name(hdlc));
#else
   printk("%s: CTS and CD always asserted\n", hdlc_to_name(hdlc));
#endif
#if defined(CONFIG_EHDLC_ASYNC)
   printk("a");
#endif
   printk("synchronous HDLC mode\n");

   return 0;
}

module_init(scc_hdlc_init);
module_exit(scc_hdlc_exit);

MODULE_AUTHOR("Rodolfo Giometti <giometti@ascensit.com>");
MODULE_DESCRIPTION("HDLC sync/async driver for Motorola MPC8xx");
MODULE_LICENSE("GPL");
EXPORT_NO_SYMBOLS;
