/*
 * frames.c
 *
 * Brad Parker, Heeltoe Consulting <brad@heeltoe.com>
 * $Id: frames.c,v 1.2 2004/05/11 19:23:20 brad Exp $
 */

#include <stdio.h>
#include <fcntl.h>

#include <sys/poll.h>
#include <sys/socket.h>

#include <linux/if.h>
#include <linux/sockios.h>
#include <linux/hdlc.h>

#define HDLC_DEV_NAME "/dev/hdlc/2"

int debug_flag;
int continuous_flag;
int loopback_flag;
int intclk_flag;
int read_flag;
int echo_flag;
int invert_flag;
int dump_flag;
int fd;

int tx_frame_num;
int rx_frame_num;
u_char buf[2048];

int
set_interface(void)
{
	struct ifreq ifr;
	sync_serial_settings settings;
	int ret;

	memset((char *)&ifr, 0, sizeof(ifr));
	memset((char *)&settings, 0, sizeof(settings));

	ifr.ifr_settings.type = IF_GET_IFACE;
	ifr.ifr_settings.size = sizeof(settings);
	ifr.ifr_settings.ifs_ifsu.sync = &settings;

	ret = ioctl(fd, SIOCWANDEV, &ifr);
	if (ret) {
		perror("IF_GET_IFACE");
		return -1;
	}

	if (intclk_flag) {
		settings.clock_type = CLOCK_TXINT;
	} else {
		settings.clock_type = CLOCK_EXT;
	}

	settings.clock_rate = 100000;

	if (loopback_flag) {
		settings.loopback |=  0x0001;
	} else {
		settings.loopback &= ~0x0001;
	}

	if (invert_flag) {
		settings.loopback |=  0x100;
	} else {
		settings.loopback &= ~0x100;
	}

	ifr.ifr_settings.type = IF_IFACE_SYNC_SERIAL;
	ifr.ifr_settings.size = sizeof(settings);
	ifr.ifr_settings.ifs_ifsu.sync = &settings;

	ret = ioctl(fd, SIOCWANDEV, &ifr);
	if (ret) {
		perror("IF_IFACE_SYNC_SERIAL");
		return -1;
	}

	return 0;
}

int
send_test_frame(int size)
{
	int i, ret;

	if (size > sizeof(buf))
		return -1;

	for (i = 0; i < size; i++)
		buf[i] = i;

	buf[0] = 0x1b;
	buf[1] = 0x01;
	buf[2] = (size-2) >> 8;
	buf[3] = (size-2);

	if (debug_flag) fprintf(stderr, "write(fd=%d,len=%d)\n", fd, size);

	ret = write(fd, buf, size);
	if (ret < 0) {
		perror("write");
		return -1;
	}

	tx_frame_num++;

	if (ret != size) {
		fprintf(stderr, "write returned %d with send of %d\n",
			ret, size);
		return -2;
	}

	return 0;
}

int
check_for_input(void)
{
	struct pollfd pfd;
	int ret;

	while (1) {
		pfd.fd = fd;
		pfd.events = POLLIN;
		pfd.revents = 0;
		
		ret = poll(&pfd, 1, 0);
		if (ret <= 0)
			break;

		ret = read(fd, buf, sizeof(buf));

		if (ret < 0) {
			perror("read error");
			return -1;
		}

		if (ret == 0) {
			printf("poll returned but read got zero!\n");
			return -1;
		}

		rx_frame_num++;

		if (dump_flag) {
			int i;

			printf("read %d (%d/%d)\n",
			       ret, tx_frame_num, rx_frame_num);

			for (i = 0; i < ret; i++)
				printf("%02x ", buf[i]);
			printf("\n");
		}

		if (echo_flag) {
			int len = ret;
			ret = write(fd, buf, len);
			if (ret != len) {
				printf("write returns %d on sending %d\n",
				       ret, len);
				return -1;
			}
		}
	}

	return 0;
}

void
usage(void)
{
	fprintf(stderr, "usage:\n");
	fprintf(stderr, "-c	run continuous, sending (forever)\n");
	fprintf(stderr, "-e	echo mode; echo frames recieved\n");
	fprintf(stderr, "-r	read mode; show frames recieved\n");
	fprintf(stderr, "\n");

	fprintf(stderr, "-C #	send # frames\n");
	fprintf(stderr, "-D #	delay # seconds before sending\n");
	fprintf(stderr, "-s #	send # byte in each frame\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "-i	use internal clock\n");
	fprintf(stderr, "-I	invert clock\n");
	fprintf(stderr, "-l	enable loopback\n");
}

void
show(void)
{
	printf("%d/%d      \r", tx_frame_num, rx_frame_num);
	fflush(stdout);
}

extern char *optarg;

main(int argc, char *argv[])
{
	int c;
	int count, delay, size;

	count = 1;
	delay = 0;
	size = 10;

	if (argc < 2) {
		usage();
		exit(1);
	}

	dump_flag = 1;

	while ((c = getopt(argc, argv, "cC:dD:eiIlrs:")) != -1) {
		switch (c) {
		case 'c':
			continuous_flag++;
			break;
		case 'C':
			count = atoi(optarg);
			break;
		case 'D':
			delay = atoi(optarg);
			break;
		case 'd':
			debug_flag++;
			break;
		case 'e':
			echo_flag++;
			read_flag++;
			dump_flag = 0;
			break;
		case 'i':
			intclk_flag++;
			break;
		case 'I':
			invert_flag++;
			break;
		case 'l':
			loopback_flag++;
			break;
		case 'r':
			read_flag++;
			break;
		case 's':
			size = atoi(optarg);
			dump_flag = 0;
			break;
		default:
			usage();
			exit(1);
		}
	}

	fd = open(HDLC_DEV_NAME, O_RDWR);
	if (fd < 0) {
		perror(HDLC_DEV_NAME);
		exit(1);
	}

	set_interface();

	if (delay) {
		sleep(delay);
	}

	while (continuous_flag) {
		show();
		send_test_frame(10);
		check_for_input();
	}

	if (read_flag) {
		if (echo_flag)
			printf("echoing...\n");
		else
			printf("reading...\n");

		while (1) {
			check_for_input();
		}
	}

	if (!continuous_flag) {
		int i;

		for (i = 0; i < count; i++) {
			show();
			send_test_frame(10);
			check_for_input();
		}

		printf("waiting\n");
		check_for_input();
		sleep(1);
		check_for_input();
		printf("done\n");
		show(); printf("\n");
	}

	exit(0);
}
