/*
 * arch/ppc/8xx_io/hdlc_m8xx.c
 *
 * HDLC (synchronous) frame driver for Motorola MPC8xx, 825x and 826x SCC.
 *
 * NOTE: This version is *not* a network driver.  It is a normal read/write
 * driver designed to be accessed by a user space program.
 * Brad Parker <brad@heeltoe.com> 2/04
 *
 * Modified to allow async as well as sync mode 4/04
 *
 * Originally from sources by Ricardo Scop, R SCOP Consult.
 * Copyright (c) 2002-2003 R SCOP Consult. (scop@plugin.com.br)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This driver is based on Daris Nevil's HDLC driver for MPC8xx SCC
 * and Dan Malek's Ethernet driver for MPC8260 SCC. It's also based on
 * some K. Halasa synchronous boards' drivers.
 *
 * ORIGINAL COPYRIGHT MESSAGES:
 * -------------------------------------------------------------------------
 * Copyright 2000 SNMC
 * Simple Network Magic Corporation
 * Copyright (c) 1999 Dan Malek (dmalek@jlc.net)
 * Copyright (C) 1998-2002 Krzysztof Halasa <khc@pm.waw.pl>
 * -------------------------------------------------------------------------
 *
 * Buffer descriptors are kept in the CPM dual port RAM, and the frame
 * buffers are in the host memory.
 *
 * At first, this was basically a work of updating the driver code to 2.4.x
 * kernels, and inserting conditional code to work with MPC82xx platforms
 * (826x and 825x, that is, not 824x).
 *
 * After that, some optimizations were made in the tx ring management code.
 *
 * TODO list:
 *  - Implement DCD handling
 *  - Implement DTR handling (where the hw permits)
 *  - Standardize board dependent stuff.
 *
 */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/netdevice.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/reboot.h>
#include <linux/spinlock.h>
#include <linux/poll.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/proc_fs.h>

#include <asm/uaccess.h>
#include <asm/processor.h>
#include <asm/system.h>
#include <asm/semaphore.h>
#include <asm/irq.h>

#ifdef CONFIG_8xx
# include <asm/8xx_immap.h>
# include <asm/mpc8xx.h>
#else  /* 8260 */
# include <asm/immap_8260.h>
# include <asm/mpc8260.h>
# include <asm/cpm_8260.h>
#endif

#include "cpm_hdlc.h"
#include "scchdlc.h"

#include "commproc_new.h"

#include <linux/hdlc.h>

/*
 *			Theory of Operation
 *
 * The MPC8xx and MPC82xx CPMs perform the HDLC processing on an SCC.
 * It can use an arbitrary number of buffers on byte boundaries, but 
 * must have at least two receive buffers to prevent constant overrun
 * conditions.
 *
 * The buffer descriptors are allocated from the CPM dual port memory
 * with the data buffers allocated from host memory, just like all other
 * serial communication protocols.  This creates a ring buffer
 * structure similar to the LANCE and other controllers.
 *
 * The driver runs as two independent, single-threaded flows of control.  One
 * is the send-packet routine, which enforces single-threaded use by the
 * port->tx_full flag.  The other thread is the interrupt handler + bottom half
 * tasks, which is single threaded by the hardware and other software.
 */

//#define CONFIG_SCC_HDLC_DEBUG
#ifdef CONFIG_SCC_HDLC_DEBUG
#define DEBUG(args...) printk(KERN_DEBUG args)
#else
#define DEBUG(args...)
#endif

#define HDLC_VERS	0x0004

/* The transmitter timeout
 */
#define TX_TIMEOUT	(2*HZ)

/* Tx and Rx ring sizes */
#define RX_RING_SIZE 8
#define TX_RING_SIZE 2 /* SCOP: testing at 64k only!!!! */

/* Maximum size for RX Buffers
 * Must be a multiple of 4 */
#define MAX_RX_BUFF_SIZE 1560 // 390*4

/* This value must be <= MAX_RX_BUFF_SIZE */
#define SCC_HDLC_MTU (1550+8)

#define SCC_PARAM_SIZE	0x100	/* SCC Parameter RAM size */

/* Defines number of RX Frames before an interrupt is generated */
#define RX_FRAME_THRESHOLD 1

/* HDLC Addresses and Masks */
#define HDLC_C_MASK	0xF0B8	/*	CRC Constant */
#define HDLC_C_PRES	0xFFFF	/* CRC Preset */
#define HDLC_ADDR_MASK	0x0000	/* Address Mask (0-bit: accept any address) */
#define HDLC_ZE_ADDR	0x0000

/*-------------------------------------------------------------------
 * Local structures */

/* The CPM buffer descriptors track the ring buffers.  The rx_bd_base and
 * tx_bd_base always point to the base of the buffer descriptors.  The
 * rx_tail and tx_head point to the currently available buffer.
 * The dirty_tx tracks the current buffer that is being sent by the
 * controller.  The tx_head and dirty_tx are equal under both completely
 * empty and completely full conditions.  The port->tx_full flag
 * determines the actual condition.
 */
typedef struct {

	int opened;
	int async;

	scc_t *base_addr;
	u_long mem_start;
	u_long mem_end;

	wait_queue_head_t	read_wait;
	struct sk_buff_head	readq;
	wait_queue_head_t	write_wait;
	struct fasync_struct    *fasync;

	char name[8];

	struct {
		u_long	rx_packets;
		u_long	rx_bytes;
		u_long	rx_packet_time;
		u_long	rx_dropped;

		u_long	rx_length_errors;	/* all of these go into */
		u_long	rx_frame_errors;	/* rx_errors... */
		u_long	rx_crc_errors;
		u_long	rx_over_errors;
		u_long	rx_missed_errors;

		u_long	rx_errors;		/* cumulative */

		u_long	tx_packets;
		u_long	tx_bytes;
		u_long	tx_packet_time;
		u_long	tx_start_time;
		u_long	tx_restart_time;
		u_long	tx_dropped;

		u_long	tx_fifo_errors;
		u_long	tx_carrier_errors;

		u_long	tx_errors;

		u_long	interrupts;
		u_long	interrupts_rx;
		u_long	interrupts_tx;

		u_long	full_sleeps;
		u_long	full_wakeups;
	} stats;

	/* Synchronous port parameters */

	sync_serial_settings settings;
	unsigned short encoding;
	unsigned short parity;
	
	/* SCC device number (1-4) */
	int	scc_num;
	int	scc_param_index;

	/* Buffer descriptors */
	/* TX Buffers and descriptors */
	volatile cbd_t	*tx_bd_base; /* Start of TX Buffer Descriptors */
	uint scc_tbase;
	int tx_head;
	volatile cbd_t	*tx_bd[TX_RING_SIZE];
	struct sk_buff	*tx_skbuff[TX_RING_SIZE];

	/* RX Buffers and descriptors */
	volatile cbd_t	*rx_bd_base; /* Start of RX Buffer Descriptors */
	uint scc_rbase;
	int rx_tail;
	volatile cbd_t	*rx_bd[RX_RING_SIZE];
	struct sk_buff	*rx_skbuff[RX_RING_SIZE];

	/* Bottom Half task queues */
	struct tq_struct rx_bh_tq;
	struct tq_struct tx_bh_tq;
	
	/* The tx ring entries to be free()ed. */
	int	dirty_tx;
		
	/* transmission ring full flag */
	volatile uint	tx_full; 

	/* mutual exclusion access control */
	spinlock_t lock; 

} hdlc_port_t;

typedef struct {
	ushort		scc_number;
	uint		pram_index;
	uint		chan_cmd; 	/* must have a typecast to ushort
						     when used with 8xx */
	ushort		irq_vector;
	uint		clock_route_mask; /* signicant bits must be set to 1 */
	uint		clock_route;
} SCC_Params_t;

/*-------------------------------------------------------------------
 * Function Prototypes */

#ifdef CONFIG_8xx
static void scc_hdlc_interrupt(void *dev_id , struct pt_regs * regs);
#else /* 8260 */
static void scc_hdlc_interrupt(int irq, void *dev_id, struct pt_regs * regs);
#endif

static int scc_hdlc_xmit(hdlc_port_t *port, struct sk_buff* skb);
static int scc_hdlc_ioctl(hdlc_port_t *port, struct ifreq* ifr, int cmd);
static int scc_hdlc_alloc_rx_ring(hdlc_port_t *port);
static struct sk_buff* scc_hdlc_alloc_skb(int mtu);
static void scc_hdlc_restart_rx(hdlc_port_t *port);
static void scc_hdlc_restart_tx(hdlc_port_t *port);
static void scc_hdlc_tx_bh(void *dev_id);
static void scc_hdlc_rx_bh(void *dev_id);
static void scc_set_iface(hdlc_port_t *port, int new_mode);

static void hdlc_dev_wakeup_tx(hdlc_port_t *port);
static int hdlc_dev_enqueue_rx_skb(hdlc_port_t *port, struct sk_buff *skb);

static void scc_hdlc_tx_wait_for_empty(hdlc_port_t *port);

/*-------------------------------------------------------------------
 * Global initializers */

#define MAX_HDLC_CHANNELS	4

static hdlc_port_t HdlcPorts[MAX_HDLC_CHANNELS];


#ifdef CONFIG_8xx

# define CPM_T 		cpm8xx_t
# define MK_CR_CMD(CH, CMD)	mk_cr_cmd((ushort) CH, CMD)
# define CPM_BDRAM_BASE		cp->cp_dpmem	/* SCC BD Ram base */
# define SCC_PARAM_BASE		cp->cp_dparam	/* SCC Parameter RAM base */

#define HDLC_SCC1_NUMBER	0
#define HDLC_SCC2_NUMBER	0
#define HDLC_SCC3_NUMBER	3
#define HDLC_SCC4_NUMBER	0

#define SICR_HDLC_MASK_SCC1	0x000000FF
#define SICR_HDLC_MASK_SCC2	0x0000FF00
#define SICR_HDLC_MASK_SCC3	0x00FF0000
#define SICR_HDLC_MASK_SCC4	0xFF000000

#define SICR_HDLC_CLKRT_SCC1	0x00000000	//undefined
#define SICR_HDLC_CLKRT_SCC2	0x00000000
#define SICR_HDLC_CLKRT_SCC3	0x00360000	/* CLK3 for tx & rx */
#define SICR_HDLC_CLKRT_SCC4	0x00000000

static SCC_Params_t SCC_Params[MAX_HDLC_CHANNELS] = {
	{HDLC_SCC1_NUMBER, PROFF_SCC1, CPM_CR_CH_SCC1, CPMVEC_SCC1,
		SICR_HDLC_MASK_SCC1, SICR_HDLC_CLKRT_SCC1},
	{HDLC_SCC2_NUMBER, PROFF_SCC2, CPM_CR_CH_SCC2, CPMVEC_SCC2,
		SICR_HDLC_MASK_SCC2, SICR_HDLC_CLKRT_SCC2},
	{HDLC_SCC3_NUMBER, PROFF_SCC3, CPM_CR_CH_SCC3, CPMVEC_SCC3,
		SICR_HDLC_MASK_SCC3, SICR_HDLC_CLKRT_SCC3},
	{HDLC_SCC4_NUMBER, PROFF_SCC4, CPM_CR_CH_SCC4, CPMVEC_SCC4,
		SICR_HDLC_MASK_SCC4, SICR_HDLC_CLKRT_SCC4},
};

#else /* 8260 */

# define CPM_T 		cpm8260_t
# define MK_CR_CMD(CH, CMD)	(CH | CMD)
# define CPM_BDRAM_BASE	immap->im_dprambase
# define SCC_PARAM_BASE	CPM_BDRAM_BASE

static SCC_Params_t SCC_Params[MAX_HDLC_CHANNELS] = {
	{HDLC_SCC1_NUMBER, PROFF_SCC1,
		mk_cr_cmd(CPM_CR_SCC1_PAGE, CPM_CR_SCC1_SBLOCK, 0, 0),
		SIU_INT_SCC1, CMX_HDLC_CLK_MASK_SCC1, CMX_HDLC_CLK_ROUTE_SCC1},
	{HDLC_SCC2_NUMBER, PROFF_SCC2,
		mk_cr_cmd(CPM_CR_SCC2_PAGE, CPM_CR_SCC2_SBLOCK, 0, 0), 
		SIU_INT_SCC2, CMX_HDLC_CLK_MASK_SCC2, CMX_HDLC_CLK_ROUTE_SCC2},
	{HDLC_SCC3_NUMBER, PROFF_SCC3,
		mk_cr_cmd(CPM_CR_SCC3_PAGE, CPM_CR_SCC3_SBLOCK, 0, 0), 
		SIU_INT_SCC3, CMX_HDLC_CLK_MASK_SCC3, CMX_HDLC_CLK_ROUTE_SCC3},
	{HDLC_SCC4_NUMBER, PROFF_SCC4,
		mk_cr_cmd(CPM_CR_SCC4_PAGE, CPM_CR_SCC4_SBLOCK, 0, 0), 
		SIU_INT_SCC4, CMX_HDLC_CLK_MASK_SCC4, CMX_HDLC_CLK_ROUTE_SCC4},
};

#endif /* CONFIG_8xx */



/*--------------------------------------------------------------
 * scc_alloc_cpm_dpram()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *
 * Allocate DPRAM for the device's BDs.
 */

static int scc_alloc_cpm_dpram(hdlc_port_t *port)
{
	uint i;
#ifdef CONFIG_8xx
	volatile CPM_T* cp = cpmp;
# define CPM_DPALLOC(X, Y)	m8xx_cpm_dpalloc(X)
#else /* 8260 */
	volatile immap_t *immap;
# define CPM_DPALLOC(X, Y)	m8260_cpm_dpalloc(X, Y)
	immap = (immap_t *)IMAP_ADDR;	/* Get pointer to internal registers */
#endif

	/*
	 * Allocate space for the buffer descriptors in the DP ram.
	 * These are relative offsets in the DP ram address space.
	 * Initialize base addresses for the buffer descriptors.
	 */
	
	/* Allocate RX Buffer descriptors */
	i = CPM_DPALLOC(sizeof(cbd_t) * RX_RING_SIZE, 8);
	if (i == CPM_DP_NOSPACE) {
		printk(KERN_WARNING "scc%d: No more CPM DPRAM for Rx\n",
		       port->scc_num);
		return -ENOMEM;
	}

	port->scc_rbase = i;
	port->rx_bd_base = (cbd_t*)&CPM_BDRAM_BASE[i];

	/* Allocate TX Buffer descriptors */
	i = CPM_DPALLOC(sizeof(cbd_t) * TX_RING_SIZE, 8);
	if (i == CPM_DP_NOSPACE) {
		printk(KERN_WARNING "scc%d: No more CPM DPRAM for Tx\n",
		       port->scc_num);
		return -ENOMEM;
	}

	port->scc_tbase = i;
	port->tx_bd_base = (cbd_t*)&CPM_BDRAM_BASE[i];
	
	return 0;

}

/*--------------------------------------------------------------
 * scc_hdlc_init_channels()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	None.
 *
 *  Initialize the SCC port for operation in HDLC mode
 */

#ifdef CONFIG_8xx

# define CLOCK_ROUTE_REG	cp->cp_sicr
# define FCR_BIG_ENDIAN		SCC_EB

#else /* 8260 */

# define CLOCK_ROUTE_REG	immap->im_cpmux.cmx_scr
# define FCR_BIG_ENDIAN		(CPMFCR_GBL | CPMFCR_EB)

#endif

/*
 *
 */
static int __init scc_hdlc_init_channels(void)
{
	int i;
	int result;
#ifdef CONFIG_8xx
	volatile CPM_T* cp = cpmp;
#else /* 8260 */
	volatile	immap_t		*immap;
#endif
	
#ifdef CONFIG_8xx
# define BASE_ADDR_SCC	cp->cp_scc 
#else /* 8260 */
# define BASE_ADDR_SCC	immap->im_scc
	immap = (immap_t *)IMAP_ADDR;	/* Get pointer to internal registers */
#endif

	for (i = 0; i < MAX_HDLC_CHANNELS; ++i) {
 		hdlc_port_t *port = &HdlcPorts[i];
		
		memset((char *)&HdlcPorts[i], 0, sizeof(HdlcPorts[0]));

		/* Initialize only hardcode-enabled SCCs */
		if (SCC_Params[i].scc_number == 0)
			continue;

		spin_lock_init(&HdlcPorts[i].lock);

		/* Save the SCC Channel number */
		port->scc_num = SCC_Params[i].scc_number;
		port->scc_param_index = i;

		sprintf(port->name, "%u", i);

		/* default to sync */
		port->async = 0;

		/*
		 * Get pointer to base of SCC control registers
		 * (index, not number)
		 */
		port->base_addr = (scc_t *)
			&BASE_ADDR_SCC[port->scc_num - 1];

		/* Get beginning of my SCC's Parameter Ram */ 
		port->mem_start = (unsigned long)
			&SCC_PARAM_BASE[SCC_Params[i].pram_index];

		/* Get end of my SCC's Parameter Ram */
		port->mem_end = port->mem_start + SCC_PARAM_SIZE - 1;

		/* default to external clock */
		port->settings.clock_type = CLOCK_EXT;

		/* Let's allocate BDs in DPRAM here, for now */
		result = scc_alloc_cpm_dpram(port);
		if (result)
			return result;

	} /* for */

	/* Success */
	return 0;
}

#define PA_DR1          ((ushort)0x4000)
#define PA_DR5          ((ushort)0x0400)
#define PA_DR7          ((ushort)0x0100)

#define PA_DR10		((ushort)0x0020)
#define PA_DR11		((ushort)0x0010)

int
setup_board_specific(void)
{
#ifdef CONFIG_RPXLITE
        volatile immap_t *immap = (immap_t *)IMAP_ADDR;
	int switches;

/* This bit added for DW boards */
#define BCSR0_BRG1TOPC15        ((uint)0x00000400)
#define BCSR0_ENPA5HDR          ((uint)0x00000800)

	/* set the configuration to enable PA5 on the header (CLK3) */
	*((volatile uint *)RPX_CSR_ADDR) |= BCSR0_ENPA5HDR;

	/* read switches */
	switches = *((volatile uint *)RPX_CSR_ADDR) & 0xf0;
	if (0) printk("switches %x\n", switches);

	/* output test clock on BRGO1 if sw3 = off */
	if (switches & 0x20) 
        {
                bd_t *bd = (bd_t *)__res;
                volatile cpm8xx_t *cp = cpmp;
		int count;

//#define BRG_RATE	1000	/* 1khz = 1ms */
//#define BRG_RATE	100000	/* 100khz = 10us */
//#define BRG_RATE	200000	/* 200khz = 5us */
#define BRG_RATE	355000	/* 355khz */

		printk("USING BRG1 TO GENERATE CLK3 FOR HDLC!\n");

                /* clock */              
		count = (bd->bi_intfreq) / (16 * BRG_RATE);

		printk("brg1, intfreq %d, count %d\n",
		       bd->bi_intfreq, count);

                /* enable, use internal clock synth */
                cp->cp_brgc1 = 0x00010000 | (count << 1) | 1;

		/* make BRG1 (PA7) an output */
		immap->im_ioport.iop_padir |= PA_DR7;
		immap->im_ioport.iop_papar |= PA_DR7;
		immap->im_ioport.iop_paodr &= ~PA_DR7;
        }

        /* make PA5 (CLK3) an input */
        immap->im_ioport.iop_padir &= ~PA_DR5;
        immap->im_ioport.iop_papar |= PA_DR5;

	immap->im_ioport.iop_padat &= ~PA_DR5;
	immap->im_ioport.iop_paodr &= ~PA_DR5;
#endif

#ifdef CONFIG_RPXCLASSIC
        volatile immap_t *immap = (immap_t *)IMAP_ADDR;

printk("RPX CLLF!\n");
        /* make PA1 (CLK7) an input - this is an 860 */
        immap->im_ioport.iop_padir &= ~PA_DR1;
        immap->im_ioport.iop_papar |= PA_DR1;

	immap->im_ioport.iop_padat &= ~PA_DR1;
	immap->im_ioport.iop_paodr &= ~PA_DR1;

	/* make PA10 an output */
        immap->im_ioport.iop_padir &= ~PA_DR10;
        immap->im_ioport.iop_papar |= PA_DR10;

	immap->im_ioport.iop_padat &= ~PA_DR10;
	immap->im_ioport.iop_paodr &= ~PA_DR10;

	/* make PA11 an input */
        immap->im_ioport.iop_padir &= ~PA_DR11;
        immap->im_ioport.iop_papar |= PA_DR11;

	immap->im_ioport.iop_padat &= ~PA_DR11;
	immap->im_ioport.iop_paodr &= ~PA_DR11;
#endif
	return 0;
}


static int
scc_hdlc_config_io_pins(hdlc_port_t *port)
{
#ifdef CONFIG_8xx
	volatile CPM_T* cp = cpmp;
#else /* 8260 */
	volatile	iop8260_t	*io = &immap->im_ioport;
#endif

	DEBUG("scc%d: scc_hdlc_config_io_pins()\n", port->scc_num);

	switch (port->scc_num) {
	case 1:
	case 2:
	case 4:
		break;

	case 3: /* SCC3 */
		DEBUG("scc%d: setup SCC3\n", port->scc_num);
		printk(KERN_INFO "scc%d: setup SCC3 pins\n", port->scc_num);

		/* make PB29=RXD3 & PB30=TXD3 */
		cp->cp_pbpar |=  0x0006;
		cp->cp_pbdir &= ~0x0006;
		cp->cp_pbodr &= ~0x0006;
		break;
	}

	return 0;
}



/*--------------------------------------------------------------
 * scc_open_hdlc_port()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) port - pointer to this driver's port structure
 *
 * Open the device
 */

static int scc_open_hdlc_port(hdlc_port_t *port)
{
	volatile scc_t *sccp = port->base_addr;
	volatile scc_hdlc_t *hp = (scc_hdlc_t *)port->mem_start;
	int result;
	
	DEBUG("scc%d: scc_open_hdlc_port()\n", port->scc_num);

	/* Make sure the parameter RAM area is not already allocated */
	if ((result = (check_region(port->mem_start, SCC_PARAM_SIZE)))) {
		printk(KERN_WARNING "scc%d: can't get region @ %08lx-%08lx\n",
		       port->scc_num,
		       port->mem_start, port->mem_start+SCC_PARAM_SIZE-1);
		return result;
	}

	/* Allocate the parameter RAM */
	request_region(port->mem_start, SCC_PARAM_SIZE, port->name);

	/* Disable rx & tx */
	sccp->scc_gsmrl &= ~(SCC_GSMRL_ENR | SCC_GSMRL_ENT);
	
	/* Configure I/O Pins */
	scc_hdlc_config_io_pins(port);

	/* configure board-specific pins */
	setup_board_specific();

	/* Set the GSMR_H and GSMR_L registers */
	/*
	 * ATTENTION!!! These two lines must be here, before the following
	 * scc_set_iface call, which modifies some bits of scc_gsmrl
	 */
	if (port->async) {
		printk("scc%d: async mode\n", port->scc_num);
		sccp->scc_gsmrh = 0;
		sccp->scc_gsmrl = SCC_GSMRL_MODE_UART |
				  SCC_GSMRL_TDCR_16 | SCC_GSMRL_RDCR_16;
	} else {
		/* sync */
		printk("scc%d: sync mode\n", port->scc_num);
		sccp->scc_gsmrh = 0 | SCC_GSMRH_RTSM;
		sccp->scc_gsmrl = SCC_GSMRL_MODE_HDLC;
	}

	/* Configure clocks and BRGs */
	scc_set_iface(port, 0);
	
	/* Initialize Tx and Rx rings' bases */
	hp->genscc.scc_rbase = port->scc_rbase;
	hp->genscc.scc_tbase = port->scc_tbase;
	
#if 0
	/*
	 * FIXME: Here we would have to put DPRAM allocation when we can
	 * free it dynamically...
	 */
	if (result = scc_alloc_cpm_dpram(port)) {
		return result;
	}
	...
#endif
	
	/* Init Function Code registers (Big Endian) */
	hp->genscc.scc_rfcr = hp->genscc.scc_tfcr = FCR_BIG_ENDIAN; 

	/* Set the RX Buffer maximum size 
	 * This appears to be a frame size, not the buffer
	 * fragment size.  It must be a multiple of four.
	 */
	hp->genscc.scc_mrblr = MAX_RX_BUFF_SIZE;

	/* Set the MTU */
	hp->mflr = SCC_HDLC_MTU;

	/* Set the CRC Constant and Preset registers */
	hp->c_mask = HDLC_C_MASK;
	hp->c_pres = HDLC_C_PRES;

	/*
	 * Set the all addresses to zero; since mask is zero we accept anything
	 */
	hp->hmask = HDLC_ADDR_MASK;
	hp->haddr1 = HDLC_ZE_ADDR;
	hp->haddr2 = HDLC_ZE_ADDR;
	hp->haddr3 = HDLC_ZE_ADDR;
	hp->haddr4 = HDLC_ZE_ADDR;
	
	/* Clear the pm counters */
	hp->disfc = 0;
	hp->crcec = 0;
	hp->abtsc = 0;
	hp->nmarc = 0;
	hp->retrc = 0;

	/* Set the RX Frame Threshold */
	hp->rfthr = RX_FRAME_THRESHOLD;

	/* Set the Protocol Specific Mode Register */
	if (port->async) {
		/* async 8 bits, 1 stop, no parity */
		sccp->scc_pmsr = 0x3000;
	} else {
		/* sync
		 * NOF = 0 (1 flag between adjacent frames)
		 * CRC = 16-bit
		 * No Retransmission
		 * FSE = Normal
		 * DRT = Normal
		 * No HDLC Bus mode
		 * BRM = Ignored
		 * MFF = Normal */
		sccp->scc_pmsr = 0x0000;
	}

	/* Disable All Interrupts for this SCC */
	sccp->scc_sccm = 0x0000;

	/* Clear pending events */
	sccp->scc_scce = 0xFFFF;

	/* Install my interrupt handler */
#ifdef CONFIG_8xx
	cpm_install_handler(SCC_Params[port->scc_param_index].irq_vector,
			    scc_hdlc_interrupt, port);
#else /* 8260 */
	request_8xxirq(SCC_Params[port->scc_param_index].irq_vector,
		       scc_hdlc_interrupt, 0, "hdlc", port);
#endif

	/*
	 * Enable Interrupts for Transmit Error, Rx Received, and TXB
	 * Let's enable BSY dectection, too.
	 */
	sccp->scc_sccm = (SCCM_TXE | SCCM_RX | SCCM_RXF | SCCM_TX | SCCM_BSY);

	/*
	 * Set the DSR register, just it case it has been modified
	 * by another driver since reset
	 */
	sccp->scc_dsr = 0x7E7E;

	/* Allocate buffers for all RX Buf Descr */
	if ((result = scc_hdlc_alloc_rx_ring(port)))
		goto fail_buffer;

	/* Initialize and start the Receiver and Transmitter */
	scc_hdlc_restart_rx(port);
	scc_hdlc_restart_tx(port);

	DEBUG("scc%d: scc_open_hdlc_port() --> 001\n", port->scc_num);

	goto done;

fail_buffer:
	printk("scc%d: fail_buffer\n", port->scc_num);

	release_region(port->mem_start, SCC_PARAM_SIZE);

done:
	/* Done */
	return result;
}

/*--------------------------------------------------------------
 * scc_close_hdlc_port()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *
 * Close the device and free all resources
 */

static void scc_close_hdlc_port(hdlc_port_t *port)
{
	volatile scc_t *sccp = port->base_addr;
	volatile CPM_T *cp = cpmp;
	int i;

	DEBUG("scc%d: scc_close_hdlc_port()\n", port->scc_num);

	/* Send a Stop Transmit command to the CPM */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
				CPM_CR_STOP_TX) | CPM_CR_FLG;

	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG, CPM_CR_FLG,
		      scc_close_hdlc_port());

	/* Disable rx & tx */
	sccp->scc_gsmrl &= ~(SCC_GSMRL_ENR | SCC_GSMRL_ENT);

	/* Free the socket buffers associated with the RX buffer rings */
	for (i = 0; i < RX_RING_SIZE; ++i) {
		/* Free any allocated socket buffers */
		if (port->rx_skbuff[i])
			dev_kfree_skb(port->rx_skbuff[i]);

		port->rx_skbuff[i] = 0;
	}

	/* Free the socket buffers associated with the TX buffer rings */
	for (i = 0; i < TX_RING_SIZE; ++i) {
		/* Free any allocated socket buffers */
		if (port->tx_skbuff[i])
			dev_kfree_skb(port->tx_skbuff[i]);

		port->tx_skbuff[i] = 0;
	}

	/* Release the Parameter Ram associated with this channel */
	release_region(port->mem_start, SCC_PARAM_SIZE);

	/* FIXME: Release RX & TX Buffer descriptors */
	/* m8xx_cpm_dpfree(hp->genscc.scc_rbase); */
	/* m8xx_cpm_dpfree(hp->genscc.scc_tbase); */

	/* Disable All Interrupts for this SCC */
	sccp->scc_sccm = 0x0000;

	/* Clear pending events */
	sccp->scc_scce = 0xFFFF;

	/* Clear statistics & counters */
	memset((char *)&port->stats, 0, sizeof(port->stats));

	/* Uninstall the interrupt handler */
	/* Just calling a request with a null handler does the trick */
#ifdef CONFIG_8xx
	cpm_free_handler(SCC_Params[port->scc_param_index].irq_vector);
#else /* 8260 */
	request_8xxirq(SCC_Params[port->scc_param_index].irq_vector,
		       0, 0, "hdlc", port);
#endif

	/* Done */
}

static void scc_encoding_setting(hdlc_port_t *port)
{
   /* Nothing to do here */

}

static void scc_async_setting(hdlc_port_t *port)
{
	volatile CPM_T* cp = cpmp;
	int index, brg;

	index = port->scc_param_index;

	/* scc3/brg3 specific */
	brg = 2;
	CLOCK_ROUTE_REG &= ~SCC_Params[index].clock_route_mask;
	CLOCK_ROUTE_REG |= (((brg<<3)|(brg<<0))<<16);

	printk("scc%d: async baud rate %d\n",
	       port->scc_num, port->settings.clock_rate);

	if (0) printk("index %d, mask %x, value %x\n",
		      index,
		      ~SCC_Params[index].clock_route_mask,
		      (((brg<<3)|(brg<<0))<<16));

	m8xx_cpm_setbrg(brg, port->settings.clock_rate);
}


static void scc_clock_setting(hdlc_port_t *port)
{
#ifdef CONFIG_8xx
	volatile CPM_T* cp = cpmp;
#else /* 8260 */
	volatile immap_t* immap = (immap_t *)IMAP_ADDR;
#endif
	int index;

	DEBUG("scc%d: scc_clock_setting()\n", port->scc_num);

	index = port->scc_param_index;

	/* ignore if async mode */
	if (port->async) {
		scc_async_setting(port);
		return;
	}

	switch (port->settings.clock_type) {
	case CLOCK_TXINT:
		DEBUG("scc%d: CLOCK_TXINT\n", port->scc_num);
		switch (port->scc_num) {
		case 1:
		case 2:
		case 4:
			break;

		case 3:
			CLOCK_ROUTE_REG &= ~SCC_Params[index].clock_route_mask;
			CLOCK_ROUTE_REG |= 0; /* SCC3 = BRG1 */

			printk(KERN_INFO "scc%d: internal clock, BRG1\n",
			       port->scc_num);
			break;

		default:
			printk(KERN_WARNING
			       "scc%d: Invalid port number\n", port->scc_num);
			return;
		}
		break;

	default:
	case CLOCK_EXT:
		DEBUG("scc%d: CLOCK_EXT\n", port->scc_num);
		CLOCK_ROUTE_REG &= ~SCC_Params[index].clock_route_mask;
		CLOCK_ROUTE_REG |= SCC_Params[index].clock_route;

		printk(KERN_INFO "scc%d: external clock\n",
		       port->scc_num);

		return;
	}
}

static void scc_loopback_setting(hdlc_port_t *port)
{
	volatile scc_t *sccp = port->base_addr;
	
	if (port->settings.loopback & 0x0001) {
		sccp->scc_gsmrl |= SCC_GSMRL_DIAG_LOOP | SCC_GSMRL_DIAG_ECHO;

		printk(KERN_INFO "scc%d: loopback on\n", port->scc_num);
	}
	else {
		sccp->scc_gsmrl &= ~(SCC_GSMRL_DIAG_LOOP | SCC_GSMRL_DIAG_ECHO);

		printk(KERN_INFO "scc%d: loopback off\n", port->scc_num);
	}

}

static void scc_txinv_setting(hdlc_port_t *port)
{
	volatile scc_t *sccp = port->base_addr;
	int txinv;

	txinv = port->settings.loopback & 0x0100;
	
	if (txinv) {
		sccp->scc_gsmrl |= SCC_GSMRL_TCI;

		printk(KERN_INFO "scc%d: tx clock invertion on\n",
		       port->scc_num);
	}
	else {
		sccp->scc_gsmrl &= ~SCC_GSMRL_TCI;

		printk(KERN_INFO "scc%d: tx clock invertion off\n",
		       port->scc_num);
	}
}

static void scc_crc_setting(hdlc_port_t *port)
{
	volatile scc_hdlc_t *hp = (scc_hdlc_t *)port->mem_start;
	int parity;

	/* ignore if async mode */
	if (port->async)
		return;

	parity = (port->settings.loopback & 0xf000) >> 12;

	switch (parity) {
	case PARITY_CRC16_PR0_CCITT:
	case PARITY_CRC16_PR1_CCITT:
	default:
		hp->c_pres = 0x0000ffff;
		hp->c_mask = 0x0000f0b8;
		printk(KERN_INFO "scc%d: using 16-bit CRC-CCITT\n",
		       port->scc_num);
		break;

	case PARITY_CRC32_PR0_CCITT:
	case PARITY_CRC32_PR1_CCITT:
		hp->c_pres = 0xffffffff;
		hp->c_mask = 0xdebb20e3;
		printk(KERN_INFO "scc%d: using 32-bit CRC-CCITT\n",
		       port->scc_num);
		break;
	}
}

/* ----------------------------------------- */

static void 
hdlc_orderly_stop(hdlc_port_t *port)
{
	volatile CPM_T *cp = cpmp;
	volatile scc_t *sccp = port->base_addr;

	/* Disable rx */
	sccp->scc_gsmrl &= ~SCC_GSMRL_ENR;

	/* wait for tx to drain */
	scc_hdlc_tx_wait_for_empty(port);

	/* Send a Stop Transmit command to the CPM */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
				CPM_CR_STOP_TX) | CPM_CR_FLG;

	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG, CPM_CR_FLG,
		      scc_close_hdlc_port());

	/* Disable tx */
	sccp->scc_gsmrl &= ~SCC_GSMRL_ENR;
}

static int scc_change_hdlc_port(hdlc_port_t *port)
{
	volatile scc_t *sccp = port->base_addr;
	volatile scc_hdlc_t *hp = (scc_hdlc_t *)port->mem_start;
	int result = 0;
	
	DEBUG("scc%d: scc_change_hdlc_port()\n", port->scc_num);

	/* Disable rx & tx */
	sccp->scc_gsmrl &= ~(SCC_GSMRL_ENR | SCC_GSMRL_ENT);
	
	/* Set the GSMR_H and GSMR_L registers */
	if (port->async) {
		printk("scc%d: async mode\n", port->scc_num);
		sccp->scc_gsmrh = 0;
		sccp->scc_gsmrl = SCC_GSMRL_MODE_UART |
				  SCC_GSMRL_TDCR_16 | SCC_GSMRL_RDCR_16;
	} else {
		/* sync */
		printk("scc%d: sync mode\n", port->scc_num);
		sccp->scc_gsmrh = 0 | SCC_GSMRH_RTSM;
		sccp->scc_gsmrl = SCC_GSMRL_MODE_HDLC;
	}

	/* Clear the pm counters */
	hp->disfc = 0;
	hp->crcec = 0;
	hp->abtsc = 0;
	hp->nmarc = 0;
	hp->retrc = 0;

	/* Set the Protocol Specific Mode Register */
	if (port->async) {
		/* async 8 bits, 1 stop, no parity */
		sccp->scc_pmsr = 0x3000;
	} else {
		/* sync
		 * NOF = 0 (1 flag between adjacent frames)
		 * CRC = 16-bit
		 * No Retransmission
		 * FSE = Normal
		 * DRT = Normal
		 * No HDLC Bus mode
		 * BRM = Ignored
		 * MFF = Normal */
		sccp->scc_pmsr = 0x0000;
	}

	/* disable interrupts, clear pending, enable */
	sccp->scc_sccm = 0x0000;
	sccp->scc_scce = 0xFFFF;
	sccp->scc_sccm = (SCCM_TXE | SCCM_RX | SCCM_RXF | SCCM_TX | SCCM_BSY);

	/* Initialize and start the Receiver and Transmitter */
	scc_hdlc_restart_rx(port);
	scc_hdlc_restart_tx(port);

	DEBUG("scc%d: scc_change_hdlc_port() done\n", port->scc_num);

	/* Done */
	return result;
}

/* change sync or async mode */
static void scc_change_xsync_mode(hdlc_port_t *port, int new_mode)
{
	DEBUG("scc%d: scc_change_xsync_mode(new_mode=%d)\n",
	      port->scc_num, new_mode);

	if (new_mode == 0)
		return;

	/* check if we need to do a mode change */
	switch (new_mode) {
	case 1: /* sync */
		if (port->async) {
			DEBUG("stopping, was async; now sync\n");
			hdlc_orderly_stop(port);
			port->async = 0;
			scc_change_hdlc_port(port);
		}
		break;
	case 2: /* async */
		if (!port->async) {
			DEBUG("stopping, was sync; now async\n");
			hdlc_orderly_stop(port);
			port->async = 1;
			scc_change_hdlc_port(port);
		}
		break;
	}
}

static void scc_set_iface(hdlc_port_t *port, int new_mode)
{
   struct {
      void (*action)(hdlc_port_t *);
   } *p, do_setting[] = {
      { scc_encoding_setting },
      { scc_clock_setting },
      { scc_loopback_setting },
      { scc_txinv_setting },
      { scc_crc_setting },
      { NULL }
   };

   DEBUG("scc%d: scc_set_iface(new_mode=%d)\n", port->scc_num, new_mode);

   scc_change_xsync_mode(port, new_mode);

   for (p = do_setting; p->action; p++)
	   p->action(port);

   DEBUG("scc%d: scc_set_iface() done!\n", port->scc_num);
}


/*--------------------------------------------------------------
 * scc_hdlc_ioctl()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) dev - pointer to this driver's device structure
 *	(IO) ifr - pointer to a structure for passing arguments
 *	(I) cmd - ioctl() function specifier
 *
 * Perform special device control functions
 */

static int scc_hdlc_ioctl(hdlc_port_t *port, struct ifreq *ifr, int cmd)
{
	const size_t size = sizeof(sync_serial_settings);
	sync_serial_settings new_line, *line = ifr->ifr_settings.ifs_ifsu.sync;

	DEBUG("scc%d: scc_hdlc_ioctl()\n", port->scc_num);

	if (cmd != SIOCWANDEV)
		return -EINVAL;

#define IF_IFACE_ASYNC_SERIAL	0x0

	switch (ifr->ifr_settings.type) {
	case IF_GET_IFACE:
		ifr->ifr_settings.type = IF_IFACE_SYNC_SERIAL;

		if (ifr->ifr_settings.size < size) {
			ifr->ifr_settings.size = size; /* data size wanted */
			return -ENOBUFS;
		}
		if (copy_to_user(line, &port->settings, size))
 			return -EFAULT;
		return 0;

	case IF_IFACE_SYNC_SERIAL:
		if (copy_from_user(&new_line, line, size))
			return -EFAULT;

		if (1) printk("hdlc: set sync %d\n", new_line.clock_rate);

		if ((new_line.clock_type != CLOCK_EXT &&
		     new_line.clock_type != CLOCK_TXINT ) ||
		    (new_line.clock_type == CLOCK_TXINT &&
		     (new_line.clock_rate < 1000 ||
		      new_line.clock_rate > 5500000)))
		{
			return -EINVAL;
		}

		/* Update settings */
		memcpy(&port->settings, &new_line, size);
		scc_set_iface(port, 1);
		break;

	case IF_IFACE_ASYNC_SERIAL:
		if (copy_from_user(&new_line, line, size))
			return -EFAULT;

		if (1) printk("hdlc: set async %d\n", new_line.clock_rate);

		if (new_line.clock_rate < 300 ||
		    new_line.clock_rate > 250000)
		{
			printk("scc%d: "
			       "IF_IFACE_ASYNC_SERIAL bad baud rate %d\n",
			       port->scc_num, new_line.clock_rate);

			return -EINVAL;
		}

		/* Update settings */
		memcpy(&port->settings, &new_line, size);
		scc_set_iface(port, 2);
		break;

	default:
		return -EINVAL;
	}
	
	/* Success */
	return(0);
}


/*--------------------------------------------------------------
 * scc_hdlc_alloc_rx_ring()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) port - pointer to this driver's device structure
 *
 * Allocate buffers for each buffer descriptor in the RX ring.
 * The receiver must be restarted after calling this function.
 */

static int scc_hdlc_alloc_rx_ring(hdlc_port_t *port)
{
	int i;
	volatile cbd_t *bdp;
	struct sk_buff *newskb;
	volatile scc_hdlc_t *hp = (scc_hdlc_t *)port->mem_start;

	/* Initialize the RX Buffer Descriptor Ring */
	bdp = port->rx_bd_base;
	for (i = 0; i < RX_RING_SIZE; ++i, ++bdp) {
		newskb = scc_hdlc_alloc_skb(hp->genscc.scc_mrblr);
		if (!newskb) {
			printk(KERN_WARNING
			       "scc%d: cannot alloc socket buffer\n",
			       port->scc_num);
			goto fail_rx_buffer;
		}

		/* Assign physical memory to each RX Buffer Descriptor */
		bdp->cbd_bufaddr = __pa(newskb->data);
		port->rx_skbuff[i] = newskb;
	}

	/* Success */
	return(0);

fail_rx_buffer:
	for (i = 0; i < RX_RING_SIZE; ++i) {
		/* Free any allocated socket buffers */
		if (port->rx_skbuff[i])
			dev_kfree_skb(port->rx_skbuff[i]);

		port->rx_skbuff[i] = 0;
	}

	return -ENOMEM;
}


/*--------------------------------------------------------------
 * scc_hdlc_alloc_skb()
 *
 * Return:
 *	Pointer to a new socket buffer.
 *
 * Input Parameters:
 *	(I) mtu - specifies the size in bytes of the current MTU
 *
 * Allocates a new socket buffer large enough to receive a complete
 * frame of the current MTU size.  For the MPC8xx, the buffer is 
 * pre-flush from the cache so that data received from the SCC is
 * not corrupted later.
 */

static struct sk_buff *scc_hdlc_alloc_skb(int mtu)
{
	struct sk_buff *newskb;

	/* Allocate a socket buffer */
	newskb = dev_alloc_skb(mtu);

#ifdef CONFIG_8xx
	if (newskb) {
		/*
		 * Push the data cache so updates are made now from cache
		 * (we don't want data clobbered later)
		 */
		flush_dcache_range((ulong)newskb->data,
				   (ulong)(newskb->data + mtu));
	}
#endif

	return(newskb);
}

/*--------------------------------------------------------------
 * scc_hdlc_restart_rx()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) port - pointer to this driver's device structure
 *
 * Restarts (or starts for the first time) the Receiver of the
 * specified SCC.
 */

static void scc_hdlc_restart_rx(hdlc_port_t *port)
{
	volatile CPM_T *cp = cpmp;
	volatile scc_t *sccp = port->base_addr;
	volatile cbd_t *bdp;
	int i;

	/* Disable the receiver */
	sccp->scc_gsmrl &= ~SCC_GSMRL_ENR;

	/*
	 * Re-INIT RX
	 * This command lets the CPM know it must reinitialize the SCC
	 * with the new parameter RAM values.
	 * This intializes the rbptr register
	 */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
				CPM_CR_INIT_RX) | CPM_CR_FLG;
	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG, CPM_CR_FLG,
		      scc_hdlc_restart_rx());

	/* Synchronize the local state machine with the CPM */
	port->rx_tail = 0;

	/* Initialize the RX Buffer Descriptor Ring */
	bdp = port->rx_bd_base;
	for (i = 0; i < RX_RING_SIZE; ++i, ++bdp) {
		/* Save the indexed pointer to the bd */
		port->rx_bd[i] = bdp;

		if (i == (RX_RING_SIZE - 1))
			bdp->cbd_sc = BD_HDLC_RX_EMPTY | BD_HDLC_RX_INTRPT |
				BD_HDLC_RX_WRAP;
		else
			bdp->cbd_sc = BD_HDLC_RX_EMPTY | BD_HDLC_RX_INTRPT;
	}

	/*
	 * Enable Receive processing
	 * Hunt Mode is automatically enetered
	 */
	sccp->scc_gsmrl |= SCC_GSMRL_ENR;
}

/*--------------------------------------------------------------
 * scc_hdlc_restart_tx()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) port - pointer to this driver's device structure
 *
 * Restarts (or starts for the first time) the Transmitter of the
 * specified SCC.
 */

static void scc_hdlc_restart_tx(hdlc_port_t *port)
{
	volatile CPM_T *cp = cpmp;
	volatile scc_t *sccp = port->base_addr;
	volatile cbd_t *bdp;
	int i;

	DEBUG("scc%d: scc_hdlc_restart_tx()\n", port->scc_num);

	/* Send the Stop Transmission command to the CPM for this SCC */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
				CPM_CR_STOP_TX) | CPM_CR_FLG;

	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG, CPM_CR_FLG,
		      scc_hdlc_restart_tx());

	/* Disable transmission */
	sccp->scc_gsmrl &= ~SCC_GSMRL_ENT;

	/* Re-INIT TX command
	 * This command lets the CPM know it must reinitialize the SCC
	 * with the new parameter RAM values.
	 * This intializes the tbptr register */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
				CPM_CR_INIT_TX) | CPM_CR_FLG;

	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG, CPM_CR_FLG,
		      scc_hdlc_restart_tx());

	/* Synchronize the local state machine with the CPM */
	port->tx_head = port->dirty_tx = 0;
	port->tx_full = 0;

	/* Initialize the TX Buffer Descriptor Ring */
	bdp = port->tx_bd_base;

	for (i = 0; i < TX_RING_SIZE; ++i, ++bdp) {
		/* Save the indexed pointer to the bd */
		port->tx_bd[i] = bdp;
#if 1
		if (i == (TX_RING_SIZE - 1)) {
			bdp->cbd_sc = BD_HDLC_TX_WRAP;
		} else {
			bdp->cbd_sc = 0;
		}
#else
		bdp->cbd_sc = 0;
#endif
		/* No pointers yet */
		bdp->cbd_bufaddr = 0;

		/* Free any allocated socket buffers */
		if (port->tx_skbuff[i]) {
			dev_kfree_skb(port->tx_skbuff[i]);
			port->stats.tx_dropped++;
		}

		port->tx_skbuff[i] = 0;
	}

#if 0
	/* After a HW reset, tx seems not to be starting automatically... */
	cp->cp_cpcr = MK_CR_CMD(SCC_Params[port->scc_param_index].chan_cmd,
				CPM_CR_RESTART_TX) | CPM_CR_FLG;

	WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG, CPM_CR_FLG,
		      scc_hdlc_restart_tx());
#endif
    
	/* Enable Transmit processing */
	sccp->scc_gsmrl |= SCC_GSMRL_ENT;

	/* Record start time of transmission */
	port->stats.tx_restart_time = jiffies;

	hdlc_dev_wakeup_tx(port);
}

/*--------------------------------------------------------------
 * scc_hdlc_xmit()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	(I) port - pointer to this driver's port structure
 *	(I) skb - pointer to a buffer to transmit
 *
 * Start transmission of a new socket buffer
 */

static int scc_hdlc_xmit(hdlc_port_t *port, struct sk_buff *skb)
{
	volatile cbd_t *bdp;
	int i;

	DEBUG("scc%d: scc_hdlc_xmit()\n", port->scc_num);

	/* Get the head of the queue */
	i = port->tx_head;
	bdp = port->tx_bd[i];

	/* paranoid: shouldn't happen */
	if (bdp->cbd_sc & BD_HDLC_TX_READY) {
		printk("scc%d: no tx bd's available\n", port->scc_num);

		/* We loose this packet */
		dev_kfree_skb(skb);
		port->stats.tx_dropped++;

		return -ENOMEM;
	}

#ifdef CONFIG_HDLC_DEBUG_PKT
	printk(KERN_DEBUG "scc%d TX(%i):", port->scc_num, skb->len);
	debug_frame(skb);
#endif

	/* Set buffer length and pointer */
	port->tx_skbuff[i] = skb;
	bdp->cbd_datlen = skb->len;

	/* get the physical address */
	bdp->cbd_bufaddr = __pa(skb->data);

#ifdef CONFIG_8xx
	/* Push the data cache so the CPM does not get stale data */
	flush_dcache_range((ulong)skb->data, (ulong)(skb->data + skb->len));
#endif

	/* Clear status flags */
	bdp->cbd_sc &= ~BD_HDLC_TX_STATS;

	spin_lock_irq(&port->lock);

	/* We found a bd we can use, bump the head index */
	++port->tx_head;
	if (port->tx_head >= TX_RING_SIZE)
		port->tx_head = 0;

	/* remember when we started */
	port->stats.tx_start_time = jiffies;

	/* Send the packet */
#if 0
	if (i == (TX_RING_SIZE - 1))
		bdp->cbd_sc |= BD_HDLC_TX_READY | BD_HDLC_TX_INTRPT |
 			BD_HDLC_TX_LAST | BD_HDLC_TX_CRC | BD_HDLC_TX_WRAP;
	else
#endif
		bdp->cbd_sc |= BD_HDLC_TX_READY | BD_HDLC_TX_INTRPT |
			BD_HDLC_TX_LAST | BD_HDLC_TX_CRC;

	if (port->tx_head == port->dirty_tx)  {
		port->tx_full = 1;
	}

	spin_unlock_irq(&port->lock);

	/* Update statistics */
	port->stats.tx_bytes += skb->len;
	port->stats.tx_packets++;
	port->stats.tx_packet_time = jiffies;

	/* Done */
	return(0);
}

/*--------------------------------------------------------------
 * scc_hdlc_interrupt()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) irq    - not used
 *	(I) dev_id - pointer to this driver's device structure
 * 	(I) regs   - not used
 *
 * Interrupt service routine, deals with Tx & Rx asynchronous
 * events and errors.
 */

#ifdef CONFIG_8xx
static void scc_hdlc_interrupt(void *dev_id, struct pt_regs * regs)
#else /* 8260 */
static void scc_hdlc_interrupt(int irq, void *dev_id, struct pt_regs * regs)
#endif
{
	hdlc_port_t *port = (hdlc_port_t *)dev_id;
	volatile scc_t *sccp = port->base_addr;
	ushort int_events;

	DEBUG("scc%d: Entering ISR\n", port->scc_num);

	/* Get the interrupt events */
	int_events = sccp->scc_scce;

	/* DEBUG("scc%d: sccp->scc_scce=0x%x\n", port->scc_num, int_events); */

	/* Clear off these event */
	sccp->scc_scce = int_events;

	port->stats.interrupts++;

	if (int_events & SCCM_BSY) {
		port->stats.rx_errors++;
		port->stats.rx_dropped++;
		DEBUG("scc%d: SCCM_BSY=%ld\n",
		      port->scc_num, port->stats.rx_errors);
	}

	if (int_events & (SCCM_RX|SCCM_RXF)) {
		/* Queue the task into the immediate bottom half queue */
		port->rx_bh_tq.routine = scc_hdlc_rx_bh;
		port->rx_bh_tq.data = (void *)port;
		queue_task(&port->rx_bh_tq, &tq_immediate);
		mark_bh(IMMEDIATE_BH);
		port->stats.interrupts_rx++;
	}

	if (int_events & SCCM_TXE) {
		port->stats.tx_errors++;
		DEBUG("scc%d: SCCM_TXE=%ld\n",
		      port->scc_num, port->stats.tx_errors);
	}

	if (int_events & (SCCM_TX | SCCM_TXE)) {
		/* Queue the task into the immediate bottom half queue */
		port->tx_bh_tq.routine = scc_hdlc_tx_bh;
		port->tx_bh_tq.data = (void*)port;
		queue_task(&port->tx_bh_tq, &tq_immediate);
		mark_bh(IMMEDIATE_BH);
		port->stats.interrupts_tx++;
	}
}


static void scc_hdlc_tx_wait_for_empty(hdlc_port_t *port)
{
	int busy = 1;

	while (busy) {
		spin_lock(&port->lock);

		/* Tx ring is empty */
		if (port->dirty_tx == port->tx_head && port->tx_full == 0) {
			busy = 0;
		}

		spin_unlock(&port->lock);
	}
}

/*--------------------------------------------------------------
 * scc_hdlc_tx_bh()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) dev_id - pointer to this driver's device structure
 *
 * This is one of the bottom half handlers for scc_hdlc_interrupt().
 * It completes the service of TX buffers that have been transmitted.
 */

static void scc_hdlc_tx_bh(void *dev_id)
{
	hdlc_port_t *port = (hdlc_port_t *)dev_id;
	volatile scc_hdlc_t *hp = (scc_hdlc_t *)port->mem_start;
	volatile cbd_t *bdp;
	int i, must_restart;

	DEBUG("scc%d: scc_hdlc_tx_bh()\n", port->scc_num);

	/* Get some CPM error counters */
	if ((i = hp->disfc + hp->crcec + hp->nmarc) != 0) {
		DEBUG("scc%d: scc_hdlc_tx_bh(): rxerr=0x%x\n",
		      port->scc_num, i);
		port->stats.rx_errors += i;
		port->stats.rx_dropped += i;
		/* FIXME: non-atomic! we may be loosing some counting here,
		         but better count some then none... -- Scop*/
		hp->disfc = hp->crcec =	hp->nmarc = 0;
	}
	
	spin_lock(&port->lock);	/* FIXME: should it be spin_lock_irq??? */
	
	must_restart = 0;
	
	/* Scan TX bd's and cleanup */
	while (1)  {
		/* Get the tail of the queue */
		i = port->dirty_tx;
		bdp = port->tx_bd[i];

		/* Don't touch buffers waiting to be transmitted */
		if (bdp->cbd_sc & BD_HDLC_TX_READY)  {
			break;
		}

		/* Tx ring is empty */
		if ((i==port->tx_head) && (port->tx_full == 0))	{
			break;
		}

		/* We found a bd with data, bump the tail index */
		if (++port->dirty_tx >= TX_RING_SIZE)
			port->dirty_tx = 0;

		/* Unused bd if its pointer is null; paranoid */
		if (port->tx_skbuff[i])	{
			/* Free the socket buffer */
			dev_kfree_skb(port->tx_skbuff[i]);
			port->tx_skbuff[i] = 0;
		}

		/* We have at least one TX Buff Descr available now */
		if (port->tx_full) {
			port->tx_full = 0;
			hdlc_dev_wakeup_tx(port);
		}

		/* Accumulate TX error statistics */
		if (bdp->cbd_sc & BD_HDLC_TX_UN) {
			must_restart = 1;
			DEBUG("scc%d: BD_HDLC_TX_UN\n", port->scc_num);
			port->stats.tx_fifo_errors++;
		}

		if (bdp->cbd_sc & BD_HDLC_TX_CL) {
			must_restart = 1;
			DEBUG("scc%d: BD_HDLC_TX_CL\n", port->scc_num);
			port->stats.tx_carrier_errors++;
		}
	}
	
	if (must_restart) {
		volatile CPM_T *cp = cpmp;

		/*
		 * Some transmit errors cause the transmitter to shut
		 * down.  We now issue a restart transmit.  Since the
		 * errors close the BD and update the pointers, the restart
		 * _should_ pick up without having to reset any of our
		 * pointers either.
		*/
		cp->cp_cpcr = MK_CR_CMD(
			SCC_Params[port->scc_param_index].chan_cmd,
			CPM_CR_RESTART_TX) | CPM_CR_FLG;
		WHILE_CPM_NEW(cp->cp_cpcr & CPM_CR_FLG, CPM_CR_FLG,
			      scc_hdlc_tx_bh());
	}
	
	spin_unlock(&port->lock); /* FIXME: should it be spin_unlock_irq??? */
}


/*--------------------------------------------------------------
 * scc_hdlc_rx_bh()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	(I) dev_id - pointer to this driver's device structure
 *
 * This is one of the bottom half handlers for scc_hdlc_interrupt().
 * It completes the service of RX buffers that have been received.
 */

static void scc_hdlc_rx_bh(void *dev_id)
{
	hdlc_port_t *port = (hdlc_port_t *)dev_id;
	volatile scc_hdlc_t *hp = (scc_hdlc_t *)port->mem_start;
	volatile cbd_t *bdp;
	struct sk_buff *skb, *newskb;
	ushort pkt_len, rx_stat;
	int rxerr, i;

	DEBUG("scc_hdlc_rx_bh(%p)\n", port);

	while (1)  {
		/* Get the tail of the queue */
		i = port->rx_tail;
		bdp = port->rx_bd[i];

		/* done? */
		if (bdp->cbd_sc & BD_HDLC_RX_EMPTY)
			break;

		/* We found a bd with data, bump the tail index */
		if (++port->rx_tail >= RX_RING_SIZE)
			port->rx_tail = 0;

		/* Get the status of the RX packet */
		rx_stat = bdp->cbd_sc;

		/* Update error statistics */
		rxerr = 0;

		if (rx_stat & BD_HDLC_RX_DE) {
			rxerr = 2;
			DEBUG("scc%d: scc_hdlc_rx_bh(): RX_DE\n",
			      port->scc_num);
		}

		if (rx_stat & BD_HDLC_RX_LG) {
			port->stats.rx_length_errors++;
			rxerr = 2;
			DEBUG("scc%d: scc_hdlc_rx_bh(): RX_LG\n",
			      port->scc_num);

		}

		if (rx_stat & BD_HDLC_RX_NO) {
			port->stats.rx_frame_errors++;
			rxerr = 2;
			DEBUG("scc%d: scc_hdlc_rx_bh(): RX_NO\n",
			      port->scc_num);
		}

		if (rx_stat & BD_HDLC_RX_AB) {
			rxerr = 2;
			DEBUG("scc%d: scc_hdlc_rx_bh(): RX_AB\n",
			      port->scc_num);
		}

		if (rx_stat & BD_HDLC_RX_CRC) {
			port->stats.rx_crc_errors++;
			rxerr = 2;
			DEBUG("scc%d: scc_hdlc_rx_bh(): RX_CRC\n",
			      port->scc_num);
		}

		if (rx_stat & BD_HDLC_RX_OV) {
			port->stats.rx_over_errors++;
			rxerr = 1;
			DEBUG("scc%d: scc_hdlc_rx_bh(): RX_OV\n",
			      port->scc_num);
		}

		if (rx_stat & BD_HDLC_RX_CD) {
			port->stats.rx_missed_errors++;
			rxerr = 1;
			DEBUG("scc%d: scc_hdlc_rx_bh(): RX_CD\n",
			      port->scc_num);
		}

		if (rxerr) {
			DEBUG("scc%d: scc_hdlc_rx_bh(): rxerr=0x%x\n",
				port->scc_num, rxerr);
			port->stats.rx_errors++;
		}

		if (rxerr <= 1) {
			/* Get a skb to replace the one just consumed */
			newskb = scc_hdlc_alloc_skb(hp->genscc.scc_mrblr);
			if (!newskb) {
				printk("scc%d: no rx buffers, "
				       "dropping packet\n", port->scc_num);
				port->stats.rx_dropped++;
			} else {
				/* Get the socket data buffer */
				skb = port->rx_skbuff[i];
				port->rx_skbuff[i] = newskb;
				pkt_len = bdp->cbd_datlen;
    	
				/* Set the phys addr of the new skb */
				bdp->cbd_bufaddr = __pa(newskb->data);
			}
		} else
			newskb = NULL; /* drop packets with critical errors */
		
		/* Clear status flags */
		bdp->cbd_sc &= ~BD_HDLC_RX_STATS;

		/* Mark ready to receive data */
		bdp->cbd_sc |= BD_HDLC_RX_EMPTY;

		if (newskb)  {
			if (!port->async) {
				/* Set the length of the received data - CRC */
				skb_put(skb, pkt_len-2);
			} else {
				skb_put(skb, pkt_len);
			}

#ifdef CONFIG_HDLC_DEBUG_PKT
			printk(KERN_DEBUG "scc%d RX(%i):",
			       port->scc_num, skb->len);
			debug_frame(skb);
#endif

			/* Update statistics */
			port->stats.rx_packets++;
			port->stats.rx_bytes += pkt_len;
			port->stats.rx_packet_time = jiffies;

			/* Point to the mac address (first HDLC byte) */
			skb->mac.raw = skb->data;

			/* Send it to the upper layer */
			hdlc_dev_enqueue_rx_skb(port, skb);
		}
	}
}

/* ------------------ linux driver i/o  routines ------------------- */


/* called by bh code - post an skb to the read queue */
static int
hdlc_dev_enqueue_rx_skb(hdlc_port_t *port, struct sk_buff *skb)
{
	/* add skb to read queue */
	skb_queue_tail(&port->readq, skb);

	/* notify and wake up reader process */
	if (port->fasync)
		kill_fasync(&port->fasync, SIGIO, POLL_IN);

	wake_up_interruptible(&port->read_wait);

	return 0;
}

static void
hdlc_dev_wakeup_tx(hdlc_port_t *port)
{
#if 0
	/* notify and wake up writer process */
	if (port->fasync)
		kill_fasync(&port->fasync, SIGIO, POLL_OUT);
#endif

	port->stats.full_wakeups++;

	wake_up_interruptible(&port->write_wait);
}


static int
hdlc_dev_enqueue_tx_skb(hdlc_port_t *port, struct sk_buff *skb)
{
	int ret;

	if ((ret= scc_hdlc_xmit(port, skb)))
		return ret;

	return 0;
}

/*
 * check if transmit side is full;
 * if so cause the userland thread to block...
 */
static int
hdlc_dev_wait_for_tx_space(hdlc_port_t *port, struct file *file)
{
	DECLARE_WAITQUEUE(wait, current);
	int ret = 0;
	int first_time = 1;

	/* if tx side is full then wait... */
	while (1) {
		add_wait_queue(&port->write_wait, &wait);
		set_current_state(TASK_INTERRUPTIBLE);

		/* not full? ok, move on... */
		if (!port->tx_full) {
			remove_wait_queue(&port->write_wait, &wait);
			break;
		}

		/* if non-blocking fd, return error */
		if (file->f_flags & O_NONBLOCK) {
			ret = -EAGAIN;
			break;
		}

		if (first_time) {
			first_time = 0;
			port->stats.full_sleeps++;
		}

		schedule();

		remove_wait_queue(&port->write_wait, &wait);

		if (signal_pending(current)) {
			ret = -ERESTARTSYS;
			break;
		}
	}

	set_current_state(TASK_RUNNING);

	return ret;
}

/* writev support */
static ssize_t
hdlc_dev_writev(struct file *file, const struct iovec *iv,
		unsigned long count, loff_t *pos)
{
	hdlc_port_t *port = (hdlc_port_t *)file->private_data;
	struct sk_buff *skb;
	unsigned long i;
	size_t len;
	int ret;

	if (!port)
		return -EBADFD;

	DEBUG("scc%d: hdlc_writev %ld\n", port->scc_num, count);

	/* verify vector */
	for (i = 0, len = 0; i < count; i++) {
		if (verify_area(VERIFY_READ, iv[i].iov_base, iv[i].iov_len))
			return -EFAULT;
		len += iv[i].iov_len;
	}

	/* if we're full then wait... */
	if ((ret = hdlc_dev_wait_for_tx_space(port, file)))
		return ret;

	/* allocate skb & copy data into it */
	if (!(skb = alloc_skb(len + 2, GFP_KERNEL))) {
		port->stats.tx_dropped++;
		return -ENOMEM;
	}

	skb_reserve(skb, 2);
	memcpy_fromiovec(skb_put(skb, len), (struct iovec *)iv, len);

	/* enqueue skb & start transmit */
	if ((ret = hdlc_dev_enqueue_tx_skb(port, skb))) {
		return ret;
	}

	return len;
}

static unsigned int
hdlc_dev_poll(struct file *file, poll_table *wait)
{
	hdlc_port_t *port = (hdlc_port_t *)file->private_data;
	unsigned int mask = POLLOUT | POLLWRNORM;

	poll_wait(file, &port->read_wait, wait);

	if (skb_queue_len(&port->readq))
		mask |= POLLIN | POLLRDNORM;

	return mask;
}

/* readv support */
static ssize_t
hdlc_dev_readv(struct file *file, const struct iovec *iv,
	       unsigned long count, loff_t *pos)
{
	hdlc_port_t *port = (hdlc_port_t *)file->private_data;
	DECLARE_WAITQUEUE(wait, current);
	struct sk_buff *skb;
	ssize_t len, ret = 0;
	unsigned long i;

	if (!port)
		return -EBADFD;

	DEBUG("scc%d: hdlc_readv\n", port->scc_num);

	/* verify vector handed from user space */
	for (i = 0, len = 0; i < count; i++) {
		if (verify_area(VERIFY_WRITE, iv[i].iov_base, iv[i].iov_len))
			return -EFAULT;
		len += iv[i].iov_len;
	}

	add_wait_queue(&port->read_wait, &wait);
	while (len) {
		current->state = TASK_INTERRUPTIBLE;

		/* read frames from the queue */
		if (!(skb = skb_dequeue(&port->readq))) {
			if (file->f_flags & O_NONBLOCK) {
				ret = -EAGAIN;
				break;
			}

			if (signal_pending(current)) {
				ret = -ERESTARTSYS;
				break;
			}

			/* Nothing to read, let's sleep */
			schedule();
			continue;
		}

#define MIN(a,b) ((a) < (b) ? (a) : (b))

		/* copy to user vector */
		len = MIN(skb->len, len);

		skb_copy_datagram_iovec(skb, 0, (struct iovec *)iv, len);
		ret = len;

		kfree_skb(skb);
		break;
	}

	current->state = TASK_RUNNING;
	remove_wait_queue(&port->read_wait, &wait);

	return ret;
}

static ssize_t
hdlc_dev_read(struct file *file, char *buf, size_t count, loff_t *pos)
{
	struct iovec iv = { (void *)buf, count };
	return hdlc_dev_readv(file, &iv, 1, pos);
}

static ssize_t
hdlc_dev_write(struct file *file, const char *buf, size_t count, loff_t *pos)
{
	struct iovec iv = { (void *)buf, count };
	return hdlc_dev_writev(file, &iv, 1, pos);
}

/* support poll() on open hdlc channel file */
static int
hdlc_dev_fasync(int fd, struct file *file, int mode)
{
	hdlc_port_t *port = (hdlc_port_t *)file->private_data;
	int ret;

	if (!port)
		return -EBADFD;

	DEBUG("scc%d: hdlc_dev_fasync %d\n", port->scc_num, mode);

	if ((ret = fasync_helper(fd, file, mode, &port->fasync)) < 0)
		return ret; 
 
	if (mode) {
		if (!file->f_owner.pid) {
			file->f_owner.pid  = current->pid;
			file->f_owner.uid  = current->uid;
			file->f_owner.euid = current->euid;
		}
	} else 
		port->fasync = 0;

	return 0;

}

/* called when /dev/hdlc/ device node is opened */
static int
hdlc_dev_open(struct inode *inode, struct file *file)
{
	int result, minor;
	hdlc_port_t *port;

	/* use minor # as port descriminator */
	minor = MINOR(inode->i_rdev);
	if (minor < 0 || minor >= MAX_HDLC_CHANNELS)
		return -ENODEV;

	port = &HdlcPorts[minor];
	DEBUG("scc%d: hdlc_dev_open()\n", port->scc_num);

	/* channel valid? */
	if (port->scc_num == 0)
		return -ENODEV;

	/* already open? */
	if (port->opened)
		return -EBUSY;

	file->private_data = port;

	/* init rx & tx queues */
	skb_queue_head_init(&port->readq);
	init_waitqueue_head(&port->read_wait);
	init_waitqueue_head(&port->write_wait);

	/* turn on h/w port */
	result = scc_open_hdlc_port(port);
	if (result) {
		file->private_data = NULL;
		return result;
	}

	if (0) printk("scc%d: admin status: UP\n", port->scc_num);

	port->opened++;

	MOD_INC_USE_COUNT;

	return 0;
}


/* called when /dev/hdlc/ device node is closed */
static int
hdlc_dev_release(struct inode *inode, struct file *file)
{
	hdlc_port_t *port = (hdlc_port_t *)file->private_data;

	if (!port)
		return 0;

	DEBUG("scc%d: hdlc_dev_release()\n", port->scc_num);

	if (port->opened == 0)
		return -ENODEV;

	/* shut down h/w */
	scc_close_hdlc_port(port);

	if (0) printk("scc%d: admin status: DOWN\n", port->scc_num);

	/* release any waiters on an open hdlc channel */
	hdlc_dev_fasync(-1, file, 0);

	/* mark closed */
	port->opened--;
	file->private_data = NULL;
	port->fasync = 0;

	/* drop read & write queues */
	skb_queue_purge(&port->readq);

	MOD_DEC_USE_COUNT;

	return 0;
}

static int
hdlc_dev_ioctl(struct inode *inode, struct file *file,
	       unsigned int cmd, unsigned long arg)
{
	hdlc_port_t *port = (hdlc_port_t *)file->private_data;
	struct ifreq ifr;

	if (copy_from_user(&ifr, (void *)arg, sizeof(struct ifreq)))
		return -EFAULT;

	ifr.ifr_name[IFNAMSIZ-1] = 0;

	return scc_hdlc_ioctl(port, &ifr, cmd);
}

static struct file_operations hdlc_fops = {
	owner:		THIS_MODULE,
	read:           hdlc_dev_read,
	write:          hdlc_dev_write,
	poll:           hdlc_dev_poll,
	ioctl:		hdlc_dev_ioctl,
	fasync:         hdlc_dev_fasync,
	open:		hdlc_dev_open,
	release:	hdlc_dev_release
};

static devfs_handle_t hdlc_devfs_handle;
static int hdlc_major;

/* register active channels in devfs tree under /dev/hdlc/... */
int
scc_hdlc_init_devs(void)
{
#if 1
	devfs_register_series(hdlc_devfs_handle, "%u", MAX_HDLC_CHANNELS,
			       DEVFS_FL_DEFAULT,
			       hdlc_major, 0,
			       S_IFCHR | S_IRUSR | S_IWUSR,
			       &hdlc_fops, NULL);
#else
	int i, minor;

	for (i = 0; i < MAX_HDLC_CHANNELS; ++i) {
		char name[32];
		minor = i;
		sprintf(name, "%d", minor);
		port->devfs = devfs_register(hdlc_devfs_handle,
					     name, DEVFS_FL_DEFAULT,
					     hdlc_major, minor,
					     S_IFBLK | S_IRUSR | S_IWUSR,
					     &hdlc_fops, NULL);
		if (port->devfs == NULL) {
			printk(KERN_WARNING "SCC_HDLC: unable to register "
			       "hdlc channel%d\n", i);
			SCC_Params[i].scc_number = 0;
			continue;
		}
	}
#endif

	return 0;
}

#ifdef CONFIG_PROC_FS
/* This macro frees the machine specific function from bounds checking and
 * this like that... */
#define PRINT_PROC(fmt,args...)                                 \
        do {                                                    \
                *len += sprintf( buffer+*len, fmt, ##args );    \
                if (*begin + *len > offset + size)              \
                        return( 0 );                            \
                if (*begin + *len < offset) {                   \
                        *begin += *len;                         \
                        *len = 0;                               \
                }                                               \
        } while(0)

static int
scc_hdlc_proc_infos(char *buffer, int *len, off_t *begin,
		    off_t offset, int size)
{
	int i;

        PRINT_PROC("SCC HDLC frame driver v%d.%d\n",
		   HDLC_VERS >> 8, HDLC_VERS & 0xff);

	for (i = 0; i < MAX_HDLC_CHANNELS; ++i) {
 		hdlc_port_t *port = &HdlcPorts[i];

		if (!port->opened)
			continue;

		PRINT_PROC("scc%d: open %d; mode %s\n",
			   port->scc_num, port->opened,
			   port->async ? "async" : "sync");
#if 1
		PRINT_PROC("txring: head %d, dirty %d, full %d\n",
			   port->tx_head, port->dirty_tx, port->tx_full);

		PRINT_PROC("full: sleeps %ld, wakeups %ld\n",
			   port->stats.full_sleeps,
			   port->stats.full_wakeups);

		{
			volatile scc_t *sccp = port->base_addr;
			volatile cbd_t *bdp;

			PRINT_PROC("sccm %04x, scce %04x, sccs %04x\n",
				   sccp->scc_sccm,
				   sccp->scc_scce,
				   sccp->scc_sccs);

			for (i = 0; i < TX_RING_SIZE; i++) {
				bdp = port->tx_bd[i];
				PRINT_PROC("tx_bd[%d] %p: sc %04x %04x/%08x\n",
					   i, bdp,
					   bdp->cbd_sc, bdp->cbd_datlen,
					   bdp->cbd_bufaddr);
			}
		}
#endif

		PRINT_PROC("ints: %lu, rx %lu, tx %lu\n",
			   port->stats.interrupts,
			   port->stats.interrupts_rx,
			   port->stats.interrupts_tx);

		PRINT_PROC("rx: packets %lu, errors %lu, dropped %lu\n",
			   port->stats.rx_packets,
			   port->stats.rx_errors,
			   port->stats.rx_dropped);

		PRINT_PROC("tx: packets %lu, errors %lu, dropped %lu\n",
			   port->stats.tx_packets,
			   port->stats.tx_errors,
			   port->stats.tx_dropped);

		PRINT_PROC("-----\n");

		PRINT_PROC("rx: bytes %lu, time %lu\n",
			   port->stats.rx_bytes,
			   port->stats.rx_packet_time);

		PRINT_PROC("rxe: length %lu, frame %lu, crc %lu\n",
			   port->stats.rx_length_errors,
			   port->stats.rx_frame_errors,
			   port->stats.rx_crc_errors);

		PRINT_PROC("rxe: over %lu, missed %lu\n",
			   port->stats.rx_over_errors,
			   port->stats.rx_missed_errors);

		PRINT_PROC("tx: bytes %lu, time %lu\n",
			   port->stats.tx_bytes,
			   port->stats.tx_packet_time);

		PRINT_PROC("txe: over %lu, carrier %lu\n",
			   port->stats.tx_fifo_errors,
			   port->stats.tx_carrier_errors);
	}

        return 1;
}

static int
scc_hdlc_read_proc(char *buffer, char **start, off_t offset,
		   int size, int *eof, void *data)
{
        int len = 0;
        off_t begin = 0;

        *eof = scc_hdlc_proc_infos(buffer, &len, &begin, offset, size);

        if (offset >= begin + len)
                return 0;

        *start = buffer + (offset - begin);

        return size < begin + len - offset ? size : begin + len - offset;
}
#endif

/* register node under /proc/drivers/... */
int
scc_hdlc_init_proc(void)
{
#ifdef CONFIG_PROC_FS
        if (!create_proc_read_entry("driver/hdlc",
                                    0, 0, scc_hdlc_read_proc, NULL))
        {
                printk(KERN_WARNING "hdlc: can't create /proc/driver/hdlc\n");
        }
#endif
	return 0;
}

/*--------------------------------------------------------------
 * scc_hdlc_init()
 *
 * Return:
 *	0 on success, negative integer on failure
 *
 * Input Parameters:
 *	None.
 *
 *  Initialize the SCC HDLC driver
 */

static int __init scc_hdlc_init(void)
{
	int result;

	DEBUG("SCC_HDLC: scc_hdlc_init()\n");

	printk(KERN_INFO "SCC HDLC sync/async frame driver v%d.%d\n",
	       HDLC_VERS >> 8, HDLC_VERS & 0xff);

	hdlc_major = devfs_register_chrdev(0, "hdlc", &hdlc_fops);
	if (hdlc_major == 0) {
		return -ENODEV;
	}

        hdlc_devfs_handle = devfs_mk_dir (NULL, "hdlc", NULL);
	if (hdlc_devfs_handle == NULL) {
		return -ENODEV;
	}

	if ((result = scc_hdlc_init_channels()))
		return result;

	if ((result = scc_hdlc_init_devs()))
		return result;

	if ((result = scc_hdlc_init_proc()))
		return result;


	return 0;
}

/*--------------------------------------------------------------
 * scc_hdlc_exit()
 *
 * Return:
 *	None.
 *
 * Input Parameters:
 *	None.
 *
 * Remove the device from the list of network devices.
 * Return allocated resources back to the system pool.
 */

static void __exit scc_hdlc_exit(void)
{
#if 0
	int i;

	DEBUG("SCC_HDLC: scc_hdlc_exit()\n");

	for (i = 0; i < MAX_HDLC_CHANNELS; ++i) {
		/* Cleanup only still enabled SCCs */
		if (SCC_Params[i].scc_number == 0)
			continue;

		devfs_unregister(&HdlcPorts[i].devfs);
	}
#endif

	DEBUG("SCC_HDLC: scc_hdlc_exit()\n");

        devfs_unregister (hdlc_devfs_handle);
}


module_init(scc_hdlc_init);
module_exit(scc_hdlc_exit);

/* Module Configuration Parameters */
MODULE_AUTHOR("Brad Parker <brad@heeltoe.com>");
MODULE_DESCRIPTION("SCC HDLC Frame Driver for MPC8xx and 82xx");
EXPORT_NO_SYMBOLS;
MODULE_LICENSE("GPL");

