/*
 * decode.c
 * decode Symbolics world files
 * an experiment, really...
 *
 * Copyright (C) 2004 Brad Parker <brad@heeltoe.com>
 * $Id: decode.c,v 1.3 2004/06/09 15:14:00 brad Exp $
 */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>

int debug;
int flag_dump_world_header;
int flag_dump_world_sparse;
int flag_dump_world_wired;
int flag_dump_world_unwired;

#define NEW_WORLD_HEADER_COOKIE 014322444510

typedef struct {
	int bits;
	int holding;
	int fd;
	u_char b[20];
} stream;

int
stream_set(stream *s, int fd, int bits)
{
	s->fd = fd;
	s->bits = bits;
	s->holding = 0;

	lseek(fd, 0, SEEK_SET);
	return 0;
}

off_t
stream_pos(stream *s)
{
	off_t o;

	o = lseek(s->fd, (off_t)0, SEEK_CUR);

	return o;
}

int
stream_seek(stream *s, off_t off)
{
	if (debug) printf("stream_seek() off 0x%lx\n", off);
	lseek(s->fd, off, SEEK_SET);
	s->holding = 0;
	return 0;
}


char tohex(char b)
{
    b = b & 0xf;
    if (b < 10) return '0' + b;
    return 'a' + (b - 10);
}

void
dumpmem(char *ptr, int len, int offset)
{
    char line[80], chars[80], *p, b, *c, *end;
    int i, j;

    end = ptr + len;
    while (ptr < end) {

	p = line;
	c = chars;
	printf("%04x ", offset);

	*p++ = ' ';
	for (j = 0; j < 16; j++) {
		if (ptr < end) {
			b = *ptr++;
			*p++ = tohex(b >> 4);
			*p++ = tohex(b);
			*p++ = ' ';
			*c++ = ' ' <= b && b <= '~' ? b : '.';
		} else {
			*p++ = 'x';
			*p++ = 'x';
			*p++ = ' ';
			*c++ = 'x';
		}
	}
	*p = 0;
	*c = 0;
        printf("%s %s\n",
	       line, chars);
	offset += 16;
    }
}


u_long
get32(int fd, u_long *pl)
{
	u_char b[4];
	int ret;

	ret = read(fd, b, 4);
	if (ret < 4) return -1;

	if (0) printf("get32: %02x %02x %02x %02x\n", b[0], b[1], b[2], b[3]);

	*pl = (b[3] << 24) | (b[2] << 16) | (b[1] << 8) | b[0];
	return 0;
}

typedef struct {
	u_char b[5];
} w40;

int
read40(stream *s, w40 *p)
{
	int ret;

	switch (s->holding) {
	case 0:
		ret = read(s->fd, s->b, 20);
		if (ret != 20)
			return -1;
		s->holding = 4;
		/* fall through */
	case 4:
		p->b[0] = s->b[0];
		p->b[1] = s->b[4];
		p->b[2] = s->b[5];
		p->b[3] = s->b[6];
		p->b[4] = s->b[7];
		s->holding = 3;
		break;
	case 3:
		p->b[0] = s->b[1];
		p->b[1] = s->b[8];
		p->b[2] = s->b[9];
		p->b[3] = s->b[10];
		p->b[4] = s->b[11];
		s->holding = 2;
		break;
	case 2:
		p->b[0] = s->b[2];
		p->b[1] = s->b[12];
		p->b[2] = s->b[13];
		p->b[3] = s->b[14];
		p->b[4] = s->b[15];
		s->holding = 1;
		break;
	case 1:
		p->b[0] = s->b[3];
		p->b[1] = s->b[16];
		p->b[2] = s->b[17];
		p->b[3] = s->b[18];
		p->b[4] = s->b[19];
		s->holding = 0;
		break;
	}

	return 0;
}


int
dump40(char *s, w40 *p)
{
	if (s) {
		printf("%s: %02x%02x%02x%02x%02x ",
		       s, p->b[0],p->b[1],p->b[2],p->b[3],p->b[4]);
	} else {
		printf("%02x%02x%02x%02x%02x ",
		       p->b[0],p->b[1],p->b[2],p->b[3],p->b[4]);
	}
	return 0;
}

u_long
ldb40(w40 *p, int bits, int shift)
{
	int mask;
	u_long v;

	mask = 0xffffffff >> (32 - bits);
	v = (p->b[4] << 24) | (p->b[3] << 16) | (p->b[2] << 8) | p->b[1];

	return (v >> shift) & mask;
}

long long
lldb40(w40 *p, int bits, int shift)
{
	long long mask;
	long long v;

	mask = (long long)-1 >> (64 - bits);
	v = ((long long)p->b[0] << 32) | (p->b[4] << 24) | (p->b[3] << 16) | (p->b[2] << 8) | p->b[1];

	return (v >> shift) & mask;
}

int
show_page_40(stream *s, int pageno)
{
	off_t offset;
	w40 page[256];
	int i;

	offset = pageno * 1280;
	stream_seek(s, offset);

	for (i = 0; i < 256; i++) {
		read40(s, &page[i]);
	}

	for (i = 0; i < 256; i++) {
		w40 *p;
		char cc[21], *pc;
		int j;

		p = &page[i];
		pc = cc;

		for (j = 0; j < 4; j++) {
			printf("%02x%02x%02x%02x%02x ",
			       p->b[0], p->b[4], p->b[3], p->b[2], p->b[1]);
			*pc++=p->b[0]; *pc++=p->b[4]; *pc++=p->b[3]; *pc++=p->b[2]; *pc++=p->b[1];
			p++;
		}

		*pc = 0;
		for (j = 0; j < 20; j++) {
			if (cc[j] < ' ' || cc[j] > '~') cc[j] = '.';
		}

		printf(" %s", cc);
		printf("\n");
		i += 3;
	}
}

int
parse_40bit_world_header(int fd)
{
	stream s;
	w40 h1, h2, h3, h4;
	w40 c1, c2, c3, c4;
	int new_world_header_format_version, new_world_header_target_architecture;
	int wired_cmds, unwired_cmds;
	int i;

	stream_set(&s, fd, 40);

	if (read40(&s, &h1))
		return -1;
	if (read40(&s, &h2))
		return -1;
	if (read40(&s, &h3))
		return -1;
	if (read40(&s, &h4))
		return -1;

	dump40("h1 ", &h1);
	dump40("h2 ", &h2);
	dump40("h3 ", &h3);
	dump40("h4 ", &h4);
	printf("\n");

	new_world_header_format_version = ldb40(&h1, 16, 0);
	new_world_header_target_architecture =ldb40(&h1, 16, 16);

	printf("format version %o, target architecture %o\n",
	       new_world_header_format_version, new_world_header_target_architecture);

#define CURRENT_NEW_WORLD_FORMAT_VERSION 0100

#define NEW_WORLD_TARGET_ARCHITECTURE_IVORY_0 0100
#define NEW_WORLD_TARGET_ARCHITECTURE_IVORY_1 0101

	printf("world format: ");
	switch (new_world_header_format_version) {
	case CURRENT_NEW_WORLD_FORMAT_VERSION:
		printf("current world format\n");
		break;
	default:
		printf("bad world header format version\n");
		return -1;
	}

	printf("architecture: ");
	switch (new_world_header_target_architecture) {
	case NEW_WORLD_TARGET_ARCHITECTURE_IVORY_0:
		printf("ivory 0\n");
		break;
	case NEW_WORLD_TARGET_ARCHITECTURE_IVORY_1:
		printf("ivory 1\n");
		break;
	default:
		printf("bad world header target architecture\n");
	}

	wired_cmds = ldb40(&h2, 32, 0);
	unwired_cmds = ldb40(&h3, 32, 0);

	/* checksums */
	read40(&s, &c1);
	read40(&s, &c2);
	read40(&s, &c3);
	read40(&s, &c4);

	printf("wired_cmds 0x%x %d\n", wired_cmds, wired_cmds);
	printf("unwired_cmds 0x%x %d\n", unwired_cmds, unwired_cmds);

	printf("checksums %x %x %x %x\n",
	       ldb40(&c1, 32, 0),
	       ldb40(&c2, 32, 0),
	       ldb40(&c3, 32, 0),
	       ldb40(&c4, 32, 0));

#define NEW_WORLD_OPCODE_DATA_PAGES 		0
#define NEW_WORLD_OPCODE_CONSTANT		1
#define NEW_WORLD_OPCODE_CONSTANT_INCREMENTED	2
#define NEW_WORLD_OPCODE_COPY			3

	printf("wired:\n");
	for (i = 0; i < wired_cmds; i++) {
		w40 vma, op, data;
		int words, opcode;

		read40(&s, &vma);
		read40(&s, &op);
		read40(&s, &data);

		printf("%d: ", i);
		if (0) {
			dump40("vma", &vma);
			dump40("opcode", &op);
			dump40("data", &data);
		}

		words = ldb40(&op, 24, 0);
		opcode = ldb40(&op, 8, 24);

		switch (opcode) {
		case NEW_WORLD_OPCODE_DATA_PAGES:
			printf("page %x, words %d, pages %d; ",
			       ldb40(&data, 32, 0), words, words / 256);
			break;
		case NEW_WORLD_OPCODE_CONSTANT:
			printf("constant %llx; ", lldb40(&data, 40, 0));
			break;
		case NEW_WORLD_OPCODE_CONSTANT_INCREMENTED:
			printf("constant++ %llx; ", lldb40(&data, 40, 0));
			break;
		case NEW_WORLD_OPCODE_COPY:
			printf("copy %08x; ", ldb40(&data, 32, 0));
			break;
		}

		printf("vma %08x opcode %08x data %08x\n",
		       ldb40(&vma, 32, 0),
		       ldb40(&op, 32, 0),
		       ldb40(&data, 32, 0));
	}

	printf("unwired:\n");
	for (i = 0; i < unwired_cmds; i++) {
		w40 vma, op, data;
		int words, opcode;

		read40(&s, &vma);
		read40(&s, &op);
		read40(&s, &data);

		printf("%d: ", i);
		if (0) {
			dump40("vma", &vma);
			dump40("op", &op);
			dump40("data", &data);
		}

		words = ldb40(&op, 24, 0);
		opcode = ldb40(&op, 8, 24);

		switch (opcode) {
		case NEW_WORLD_OPCODE_DATA_PAGES:
			printf("page %x, words %d, pages %d; ",
			       ldb40(&data, 32, 0), words, words / 256);
			break;
		case NEW_WORLD_OPCODE_CONSTANT:
			printf("constant %llx; ", lldb40(&data, 40, 0));
			break;
		case NEW_WORLD_OPCODE_CONSTANT_INCREMENTED:
			printf("constant++ %llx; ", lldb40(&data, 40, 0));
			break;
		case NEW_WORLD_OPCODE_COPY:
			printf("copy %08x; ", ldb40(&data, 32, 0));
			break;
		}

		printf("vma %08x opcode %08x data %08x\n",
		       ldb40(&vma, 32, 0),
		       ldb40(&op, 32, 0),
		       ldb40(&data, 32, 0));
	}

	printf("file offset 0x%x\n", stream_pos(&s));

	show_page_40(&s, 2);
	return 0;
}

typedef struct {
	u_char b[5];
} w36;

int
read36(stream *s, w36 *p)
{
	int ret;

	switch (s->holding) {
	case 0:
		ret = read(s->fd, s->b, 9);
		if (ret != 9)
			return -1;
		s->holding = 2;
		if (0) printf("read %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
			      s->b[0], s->b[1], s->b[2], s->b[3], 
			      s->b[4], s->b[5], s->b[6], s->b[7], s->b[8]);
		/* fall through */
	case 2:
		p->b[0] = s->b[4] & 0xf;
		p->b[1] = s->b[0];
		p->b[2] = s->b[1];
		p->b[3] = s->b[2];
		p->b[4] = s->b[3];
		s->holding = 1;
		break;
	case 1:
		p->b[0] = (s->b[8] >> 4) & 0xf;
		p->b[1] = ((s->b[5] & 0xf) << 4) | (s->b[4] >> 4);
		p->b[2] = ((s->b[6] & 0xf) << 4) | (s->b[5] >> 4);
		p->b[3] = ((s->b[7] & 0xf) << 4) | (s->b[6] >> 4);
		p->b[4] = ((s->b[8] & 0xf) << 4) | (s->b[7] >> 4);
		s->holding = 0;
		break;
	}

	return 0;
}

int
dump36(char *s, w36 *p)
{
#if 0
	printf("%s: %02x%02x%02x%02x%02x ", s, p->b[0],p->b[4],p->b[3],p->b[2],p->b[1]);
#else
	printf("%s: %02x%02x%02x%02x%02x ", s, p->b[0],p->b[1],p->b[2],p->b[3],p->b[4]);
#endif
	return 0;
}

u_long
ldb36(w36 *p, int bits, int shift)
{
	int mask;
	u_long v;

	mask = 0xffffffff >> (32 - bits);
	v = (p->b[4] << 24) | (p->b[3] << 16) | (p->b[2] << 8) | p->b[1];

	return (v >> shift) & mask;
}


u_long
ldb36_nbs(w36 *p, int bits, int shift)
{
	int mask;
	u_long v;

	mask = 0xffffffff >> (32 - bits);
	v = (p->b[1] << 24) | (p->b[2] << 16) | (p->b[3] << 8) | p->b[4];

	return (v >> shift) & mask;
}

int
oddi_36(w36 *p)
{
	return ldb36(p, 16, 16) | ((p->b[0] & 0x8) ? 0x10000 : 0);
}

int
eveni_36(w36 *p)
{
	return ldb36(p, 16, 0) | ((p->b[0] & 0x4) ? 0x10000 : 0);
}

u_long amem[4096];
u_char amem_tag[4096];

void
write_amem(int addr, int v)
{
	amem[addr & 0x0fff] = v;
}

int
read_amem(int addr)
{
	return amem[addr & 0x0fff];
}

extern char *find_sym_by_val(int mem, int v);

char *
find_amem(int v)
{
	int i;
	for (i = 0; i < 4096; i++)
		if ((amem[i] & 0x0ffffff) == (v & 0x0fffffff))
			return find_sym_by_val(1, i);

	return 0;
}

int
read36_page(stream *s, int offset, u_long *mem, u_char *tag)
{
	int i;

	stream_seek(s, offset);

	for (i = 0; i < 256; i++) {
		w36 w;
		read36(s, &w);

//		tag[i] = w.b[0];
		tag[i] = (w.b[0] << 4) | (w.b[4] >> 4);
		mem[i] = (w.b[4] << 24) | (w.b[3] << 16) |
			(w.b[2] << 8) | w.b[1];
	}

	return 0;
}

int
show_page_36(stream *s, unsigned long addr, int fpn)
{
	off_t offset;
	w36 page[256];
	u_char p[256*4];
	int i, j, wordnum;
	int show_ascii, show_36, show_macro;

	wordnum = addr;

	offset = fpn * 1152;  /* (256/2)*9 */
	stream_seek(s, offset);

	for (i = 0; i < 256; i++) {
		read36(s, &page[i]);

		p[i*4+0] = page[i].b[1];
		p[i*4+1] = page[i].b[2];
		p[i*4+2] = page[i].b[3];
		p[i*4+3] = page[i].b[4];
	}

	show_ascii = 1;
	show_36 = 1;
	show_macro = 1;

	if (show_ascii) {
		dumpmem(p, 1024, offset);
		printf("\n");
	}

	if (show_36) {
		for (i = 0; i < 256; i += 4) {
			printf("%4o ", wordnum + i);
			for (j = 0; j < 4; j++) {
//				dump36("", &page[i+j]);
				printf("%02x:%08x ",
//				       page[i+j].b[0],
				       (page[i+j].b[0] << 4) | (page[i+j].b[4] >> 4),
				       ldb36(&page[i+j], 32, 0));
			}
			printf("\n");
		}
	}

	if (show_macro) {
		for (i = 0; i < 256; i += 4) {
			char *sym;

			printf("%4o ", wordnum + i);
			for (j = 0; j < 4; j++) {
				int e, o;

				e = eveni_36(&page[i+j]);
				o = oddi_36(&page[i+j]);
				printf("%05x %05x  ", e, o);
			}
			printf("\n");
			for (j = 0; j < 4; j++) {
				int e,o;

				sym = find_amem(wordnum+i+j);
				if (sym) {
					printf("%s:\n", sym);
				}

				e = eveni_36(&page[i+j]);
				o = oddi_36(&page[i+j]);
				printf("%4o ", wordnum + i + j);
				disassemble36(e);
				printf("%4o ", wordnum + i + j);
				disassemble36(o);

				if (0) printf("inst %x %x, tag %x\n",
					      ldb36(&page[i+j], 16, 16), e, page[i+j].b[0]);
			}
			printf("\n");
		}
	}

}

struct vma_entry_s {
	int mtype;
	unsigned long vma;
	int words;
	int fpn;
};

int vma_entry_count;
struct vma_entry_s vma_entrys[256];

int
add_vma_map_entry(int mtype, unsigned long vma, int words, int fpn)
{
	int i;

	if (vma_entry_count < 256) {
		struct vma_entry_s *v = &vma_entrys[vma_entry_count++];
		v->mtype = mtype;
		v->vma = vma;
		v->words = words;
		v->fpn = fpn;
	}

	for (i = 0; i < words / 256; i++) {
		paging_add_vma_entry(vma + i*256, fpn + i);
	}

	return 0;
}

int
find_vma_map_fpn(unsigned long vma)
{
	int i, offset;
	struct vma_entry_s *v = vma_entrys;

	for (i = 0; i < vma_entry_count; i++, v++) {
		unsigned long start, end;
		start = v->vma;
		end = v->vma + v->words - 1;
		if (start <= vma && vma <= end) {
			offset = (vma - start) / 256;
			return v->fpn + offset;
		}
	}

	return -1;
}

int
get_amem_sym_value(char *sym)
{
	int v, a;

	find_sym(1, sym, &v);
	a = read_amem(v);

	return a;
}

void
show_sym_code36(stream *s, char *sym)
{
	int v, a, fpn;

#define a28(a)		((a) & 0x0fffffff)
//#define addrtopage(a)	(4 + (a28(a) / 256))

	a = get_amem_sym_value(sym);

//	p = addrtopage(a);
	fpn = find_vma_map_fpn(a28(a));

	a = a28(a) & 0xffffff00;

	printf("%s = %o -> %o, page 0x%x\n", sym, v, a, fpn);
	show_page_36(s, a, fpn);
}

void
find_string_code36(stream *s, char *str)
{
	off_t offset;
	w36 page[256];
	u_char p[256*4];
	int fpn, len, i, j, hit;


	printf("searching for '%s'\n", str);

	len = strlen(str);

	for (fpn = 0; 1; fpn++) {
		offset = fpn * 1152;  /* (256/2)*9 */
		stream_seek(s, offset);

		for (i = 0; i < 256; i++) {
			read36(s, &page[i]);

			p[i*4+0] = page[i].b[1];
			p[i*4+1] = page[i].b[2];
			p[i*4+2] = page[i].b[3];
			p[i*4+3] = page[i].b[4];
		}

		for (i = 0; i < 1024; i++) {
			if (toupper(p[i]) == toupper(str[0])) {
				hit = 0;
				for (j = 0; j < len && (i+j) < 1024; j++)
					if (toupper(p[i+j]) == toupper(str[j]))
						hit++;

				if (hit == len) {
					printf("hit @ fpn %d, offset %x\n",
					       fpn, offset + i);

					dumpmem(p, 1024, offset);

					printf("\n");
					for (i = 0; i < 256; i += 4) {
//						printf("%4o ", offset + i);
printf("%4o %4x ", fpn*256 + i, fpn*256 + i);
printf("%4x ", offset + (i/4)*16);
						for (j = 0; j < 4; j++) {
//							dump36("", &page[i+j]);
							printf("%02x:%08x ",
							       page[i+j].b[0],
							       ldb36(&page[i+j], 32, 0));
						}
						printf("\n");
					}
//					return;
					break;
				}
			}
		}
	}

	printf("search done\n");
}

/*
 * read old style 36-bit world
 */

int
parse_36bit_world_header(int fd)
{
	stream s;
	w36 h1, h2, h3, h4, h5;
	int mic_version, wlf_version, sparse, wired, unwired;
	int i, j;

	stream_set(&s, fd, 36);

	if (read36(&s, &h1))
		return -1;
	if (read36(&s, &h2))
		return -1;
	if (read36(&s, &h3))
		return -1;
	if (read36(&s, &h4))
		return -1;
	if (read36(&s, &h5))
		return -1;

	if (flag_dump_world_header) {
		dump36("mic_version", &h1);
		dump36("wlf_version", &h2);
		dump36("sparse", &h3);
		dump36("wired", &h4);
		dump36("unwired", &h5);
		printf("\n");
	}

	mic_version = ldb36(&h1, 32, 0);
	wlf_version = ldb36(&h2, 32, 0);
	sparse = ldb36(&h3, 32, 0);
	wired = ldb36(&h4, 32, 0);
	unwired = ldb36(&h5, 32, 0);

	printf("mic_version %x %d, ", mic_version, mic_version);
	printf("wlf_version %x %d, ", wlf_version, wlf_version);
	printf("sparse %x %d, ", sparse, sparse);
	printf("wired %x %d, ", wired, wired);
	printf("unwired %x %d\n", unwired, unwired);

	if (flag_dump_world_sparse)
		printf("sparse vm entries:\n");

	for (i = 0; i < sparse; i++) {
		w36 vma, datum;
		int amem_addr;

		if (read36(&s, &vma))
			return -1;
		if (read36(&s, &datum))
			return -1;

		if (0) {
			printf("%d: ", i);
			dump36("vma", &vma);
			dump36("datum", &datum);
			printf("\n");
		}

		if (flag_dump_world_sparse) {
			printf("%d: vma %08x, datum %08x (0%011o) ",
			       i, ldb36(&vma, 32, 0),
			       ldb36(&datum, 32, 0), ldb36(&datum, 32, 0));
				//dump36("", &datum);
		}

		amem_addr = ldb36(&vma, 32, 0) & 0x0fff;

		write_amem(amem_addr, ldb36(&datum, 32, 0));

		if (flag_dump_world_sparse) {
			char *name;

			name = find_sym_by_val(1, amem_addr);
			if (name) {
				printf("%s", name);
			}

			printf("\n");
		}
	}

	if (debug)
	{
		int words, bytes;

		words = 5 + sparse*2;
		if (words & 1) words++;
		bytes = (words / 2) * 9;

		printf("words %d, bytes %x\n", words, bytes);
	}

	if (debug) printf("file offset 0x%x\n", stream_pos(&s));

	if (flag_dump_world_wired)
		printf("wired vm entries:\n");

	for (i = 0; i < wired; i++) {
		w36 vma, words, fpn;

		if (read36(&s, &vma))
			return -1;
		if (read36(&s, &words))
			return -1;
		if (read36(&s, &fpn))
			return -1;

		if (0) {
			printf("%d: ", i);
			dump36("vma", &vma);
			dump36("words", &words);
			dump36("fpn", &fpn);
			printf("\n");
		}

		if (flag_dump_world_wired) {
			printf("%d: vma %08x, words %08x, fpn %08x\n",
			       i,
			       ldb36(&vma, 32, 0),
			       ldb36(&words, 32, 0),
			       ldb36(&fpn, 32, 0));
		}

		add_vma_map_entry(1,
				  ldb36(&vma, 32, 0),
				  ldb36(&words, 32, 0),
				  ldb36(&fpn, 32, 0));
	}

	if (flag_dump_world_unwired)
		printf("unwired vm entries:\n");

	for (i = 0; i < unwired; i++) {
		w36 vma, words, fpn;

		if (read36(&s, &vma))
			return -1;
		if (read36(&s, &words))
			return -1;
		if (read36(&s, &fpn))
			return -1;

		if (0) {
			printf("%d: ", i);
			dump36("vma", &vma);
			dump36("words", &words);
			dump36("fpn", &fpn);
			printf("\n");
		}

		if (flag_dump_world_unwired) {
			printf("%d: vma %08x, words %08x, fpn %08x\n",
			       i,
			       ldb36(&vma, 32, 0),
			       ldb36(&words, 32, 0),
			       ldb36(&fpn, 32, 0));
		}

		add_vma_map_entry(2,
				  ldb36(&vma, 32, 0),
				  ldb36(&words, 32, 0),
				  ldb36(&fpn, 32, 0));
	}

	if (debug) printf("file offset 0x%x\n", stream_pos(&s));

	{
		int a, fpn;

#if 0		
		a = 0;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);

		a = 0x376a00;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);

		a = 0x516000;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);
#endif

#if 0
		a = 0x100ae00;
		printf("0x%x %o\n", 0x100ae3d, 0x100ae3d);
		printf("0x%x %o\n", 0x100ae51, 0x100ae51);
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);
#endif

		a = 0;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);

		a = 061370 & ~0xff;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);

		a = 061767 & ~0xff;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);
	}

	show_sym_code36(&s, "FUNCTION-SYSTEM-STARTUP");
//	show_sym_code36(&s, "PULL-LEXPR-ARGS-ESCAPE-PC");
//	show_sym_code36(&s, "CALL-FUNNY-METHOD-ESCAPE-PC");

//	show_sym_code36(&s, "FUNCTION-CELL-LOCATION");


#if 0
	{
		int a, fpn;
		a = 061373 & ~0xff;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);

		a = 061437 & ~0xff;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);

	}
#endif

#if 0
	find_string_code36(&s, ".LISP");
	{
		int a, fpn;
		a = 0x3300;
//		fpn = find_vma_map_fpn(a28(a));
//		show_page_36(&s, a, fpn);
		a = 0x0d00;
//		fpn = find_vma_map_fpn(a28(a));
//		show_page_36(&s, a, fpn);

//6600ab5
//6600abe 630005276
		a = 0x6600a00;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);

//630005274 00:466c58bd 04:466c25b9 00:2039585a 01:80000001 
//630005300 00:466c58bd 04:466c59e4 00:2039c722 01:80000001 
//38585a 16254132

//looking for 53f271 24771161

// 630005276 = 38585a 16254132
// 016254132 -> ascii name

//16254130             00:7100ae3d 03:6053f271 00:0039585a 
//16254134 03:2ffff545 00:4035f9be 00:7100ae3d

		a = 0x0395800;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);

		a = 0x53f200;
		fpn = find_vma_map_fpn(a28(a));
		show_page_36(&s, a, fpn);
	}

	return 0;
	find_string_code36(&s, "SYSTEM-STARTUP");
#endif

	printf("\n");

	start_sim(&s);

	return 0;
}

/*
 * look at the world file header,  try and figure out if it's a new style
 * or old style world
 *
 * if the cookie matches, parse the world
 */
int
parse_world_header(int fd)
{
	u_long header_cookie;

	if (get32(fd, &header_cookie))
		return -1;

	if (debug) {
		printf("header cookie %x %o\n",
		       header_cookie, header_cookie);
		printf("want %x %o\n",
		       NEW_WORLD_HEADER_COOKIE, NEW_WORLD_HEADER_COOKIE);
	}

	if (header_cookie == NEW_WORLD_HEADER_COOKIE) {
		printf("40-bit world\n");
		parse_40bit_world_header(fd);
	} else {
		printf("36-bit world\n");
		parse_36bit_world_header(fd);
	}

	return 0;
}

int
disassemble_init(void)
{
	disassemble36_setup();
	return 0;
}

int
symbol_init(void)
{
	symbol36_setup();
}


void
usage(void)
{
	exit(1);
}

extern char *optarg;
extern int optind;

main(int argc, char *argv[])
{
	int c, fd;

	flag_dump_world_sparse = 0;
	flag_dump_world_header = 0;
	flag_dump_world_wired = 0;
	flag_dump_world_unwired = 0;

	while ((c = getopt(argc, argv, "dDm:")) != -1) {
		switch (c) {
		case 'd':
			debug++;
			break;
		case 'D':
			flag_dump_world_sparse = 1;
			flag_dump_world_wired = 1;
			flag_dump_world_unwired = 1;
			break;
		case 'm':
			symbol36_mic_sym_filename(optarg);
			break;
		default:
			usage();
		}
	}

	disassemble_init();
	symbol_init();
	paging_init();

	if (0) printf("optind %d, argc %d, argv[optind] %s\n",
		      optind, argc, argv[optind]);

	if (optind >= argc)
		usage();

	fd = open(argv[optind], O_RDONLY);

	parse_world_header(fd);

	exit(0);
}
