#!/bin/bash

#
# This script restores the contents of flash for WR Linux
# on Altera boards.
#

#
# Script Variables.
#####################
#
# Flash base address.
#
BASE_ADDR=0x10000000
#
# Offsets in flash.
#
DEFAULT_MMU_OFST=0x3500000
OPTION_BITS_OFST=0x3fe0000
KERNEL_OFST=0x2900000
FS_OFST=0x0
UBOOT_OFST=0x2800000
UBOOT_ENV_OFST=0x3FF0000
#
# SREC filenames.
#
SREC_DIR=${PWD}/srecs
FS_FN=${SREC_DIR}/filesystem.jffs2.srec
UBOOT_FN=${SREC_DIR}/u-boot.srec
UBOOT_ENV_FN=${SREC_DIR}/setup_default_uboot_env.srec
KERNEL_FN=${SREC_DIR}/uImage0.img.srec
OPTION_BITS_FN=${SREC_DIR}/option_bits.srec
DEFAULT_MMU_FN=${SREC_DIR}/DEFAULT_MMU_QUAL.rbf.srec
#
# Usage statement function.
#
usage() {
  echo ""
  echo "You must pass at least one argument to this script."
  echo ""
  echo "USAGE: ${0} [--prog_kernel|--prog_fs|--prog_default_mmu|--prog_option_bits|--prog_uboot_env|--prog_all]"
  echo "                                   NOTE: Multiple arguments accepted." 
  echo ""
}

#
# Option variables.
#
prog_kernel=
prog_fs=
prog_uboot=
prog_uboot_env=
prog_default_mmu=
prog_option_bits=

#
# Parse Options.
#

if [ $# == 0 ] ; then
  usage
  exit 0
fi

while test $# != 0 
do

  case "$1" in
  -a|--a|--all|--prog_all)
    prog_kernel=1
    prog_fs=1
    prog_uboot=1
    prog_uboot_env=1
    prog_default_mmu=1
    prog_option_bits=1
    shift
    ;;
  -k|--k|-prog_kernel|--prog_kernel)
    prog_kernel=1
    shift
    ;;
  -fs|--fs|-prog_fs|--prog_fs)
    prog_fs=1
    shift
    ;;
  -u|--u|-prog_uboot|--prog_uboot)
    prog_uboot=1
    shift
    ;;
  -ue|--ue|-prog_uboot_env|--prog_uboot_env)
    prog_uboot_env=1
    shift
    ;;
  -d|--d|-prog_default_mmu|--prog_default_mmu)
    prog_default_mmu=1
    shift
    ;;
  -o|--o|-prog_option_bits|--prog_option_bits)
    prog_option_bits=1
    shift
    ;;
  -*)
    usage
    exit 0
    ;;
  esac
done

#
# Configure the SOF to enable flash programming.
#
echo ""
echo "Configuring SOF..."
echo ""
echo nios2-configure-sof
echo ""

#
# Programming Flash
#

echo ""
echo "Programming Flash..."
echo ""
echo ""
if [ ${prog_kernel} ] ; then
  echo ""
  echo "Programming the kernel..."
  echo nios2-flash-programmer --base=${BASE_ADDR} ${KERNEL_FN}
  echo ""
fi
if [ ${prog_fs} ] ; then
  echo ""
  echo "Programming the JFFS2 filesystem..."
  echo nios2-flash-programmer --base=${BASE_ADDR} ${FS_FN}
  echo ""
fi 
if [ ${prog_uboot} ] ; then
  echo ""
  echo "Programming Uboot..."
  echo nios2-flash-programmer --base=${BASE_ADDR} ${UBOOT_FN}
  echo ""
fi
if [ ${prog_uboot_env} ] ; then
  echo ""
  echo "Programming Uboot Env..."
  echo nios2-flash-programmer --base=${BASE_ADDR} ${UBOOT_ENV_FN}
  echo ""
fi
if [ ${prog_default_mmu} ] ; then
  echo ""
  echo "Programming DEFAULT MMU SOF."
  echo nios2-flash-programmer  --base=${BASE_ADDR} ${DEFAULT_MMU_FN}
  echo ""
fi
if [ ${prog_option_bits} ] ; then
  echo ""
  echo "Programming Option Bits..."
  echo nios2-flash-programmer --base=${BASE_ADDR} ${OPTION_BITS_FN}
  echo ""
fi
echo ""
echo ""
echo "Done..."
echo ""
echo ""
