/*
 * isn.c
 *
 * simple NIOS-II simulator instruction decode
 * Brad Parker <brad@heeltoe.com>
 * 10/23/07
 *
 * Copyright (C) 2007 Brad Parker
 */

#include <stdio.h>
#include <stdlib.h>

#include "sim.h"
#include "ops.h"

u32 pc;
#define ra reg[31]
#define ba reg[30]
#define ea reg[29]
u32 reg[32];

u32 ctl_reg[32];
#define status ctl_reg[0]
#define estatus ctl_reg[1]
#define bstatus ctl_reg[2]
#define ienable ctl_reg[3]
#define ipending ctl_reg[4]
#define cpuid ctl_reg[5]

u32 irq_latch;

#define PIE	0x01
#define U	0x02

u32 exception_handler;
u32 break_handler;

u32 isn;
u32 op;
u32 opx;
u32 opx2;
int carry;

static int local_show;
int trace_int;

void
show_regs(void)
{
    int i, j;

    for (i = 0; i < 32; i += 8) {
        printf("R%02d: ", i);
        for (j = 0; j < 8; j++) {
            printf("%08x ", reg[i+j]);
        }
        printf("\n");
    }

    printf("ctl ");
    for (j = 0; j < 6; j++) {
        printf("%08x ", ctl_reg[j]);
    }
    printf("\n");
    printf("irq %08x\n", irq_latch);

}

void
show_state(int verbose)
{
    char *name;
    int offset;

    if (verbose) {
        printf("\n");
        printf("PC:  %08x, RA %08x\n", pc, ra);
        disas(pc, isn);

        if (find_func_loc(pc, &name, &offset) == 0) {
            printf("%s+0x%x\n", name, offset);
        }

        show_regs();
    } else {
        int do_regs;

        if (find_func_loc(pc, &name, &offset) == 0) {
            printf("%s+0x%x\n", name, offset);
            do_regs = 1;
        } else
            do_regs = 0;

        printf("RA %08x, PC ", ra);
        disas(pc, isn);

        if (do_regs)
            show_regs();
    }

}

static inline u32
rotate_left(u32 value, int bitstorotate)
{
    unsigned int tmp;
    int mask;

    /* determine which bits will be impacted by the rotate */
    if (bitstorotate == 0)
        mask = 0;
    else
        mask = (int)0x80000000 >> bitstorotate;
		
    /* save off the affected bits */
    tmp = (value & mask) >> (32 - bitstorotate);
		
    /* perform the actual rotate */
    /* add the rotated bits back in (in the proper location) */
    return (value << bitstorotate) | tmp;
}

static inline u32
rotate_right(u32 value, int bitstorotate)
{
    return rotate_left(value, 32 - bitstorotate);
}

static inline void
inst_loop(int show)
{
    u32 a, b, c;
    u32 imm26, addr;
    s16 imm16;
    u8 imm5;
    long long llv;

    timer_update();

//    if (pc == 0x804ba8) { show = 1; local_show = 1; }
//    if (pc == 0x97da84) { show = 1; local_show = 1; }
//    if ((pc & 0xffffff00) == 0x9d5700) { show = 1; local_show = 1; }
//    if (pc == 0x00800d80) { show = 1; local_show = 1; }

    if (ipending && (status & PIE)) {
        estatus = status | PIE;
        status &= ~(U | PIE);
        if (trace_int)
            printf("interrupt: pc %08x, new pc %08x\n", pc, exception_handler);
        ea = pc + 4;
        pc = exception_handler;
    }

    isn = mem_read_i(pc);
    op = isn & 0x3f;

    if (show) {
        show_state(0);
    }

    switch (op) {
    case inst_call:	/* J type */
        imm26 = (isn >> 6) & 0x03ffffff;
        ra = pc + 4;
        pc = (pc & 0xf0000000) | (imm26 << 2);
        break;

    case inst_jmpi:
        imm26 = (isn >> 6) & 0x03ffffff;
        pc = pc + (imm26 << 2);
        break;

    case inst_ldbuio:
    case inst_ldbu:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        reg[b] = mem_read_d8(addr);
        pc += 4;
        break;

    case inst_addi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] + imm16;
        pc += 4;
        break;

    case inst_stbio:
    case inst_stb:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        mem_write_d8(addr, reg[b]);
        pc += 4;
        break;

    case inst_br:
        imm16 = (isn >> 6) & 0xffff;
        pc = (pc + 4) + imm16;
        break;
        
    case inst_ldbio:
    case inst_ldb:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        reg[b] = (s8)mem_read_d8(addr);
        pc += 4;
        break;

    case inst_cmpgei:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] >= imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_ldhuio:
    case inst_ldhu:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        reg[b] = mem_read_d16(addr);
        pc += 4;
        break;

    case inst_andi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] & (u16)imm16;
        pc += 4;
        break;

    case inst_sthio:
    case inst_sth:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        mem_write_d16(addr, reg[b]);
        pc += 4;
        break;

    case inst_bge:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] >= (s32)reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_ldh:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        reg[b] = (s16)mem_read_d16(addr);
        pc += 4;
        break;

    case inst_cmplti:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] < imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_ori:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] | (u16)imm16;
        pc += 4;
        break;

    case inst_stwio:
    case inst_stw:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        mem_write_d32(addr, reg[b]);
        pc += 4;
        break;

    case inst_blt:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] < (s32)reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_ldwio:
    case inst_ldw:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        reg[b] = mem_read_d32(addr);
        pc += 4;
        break;

    case inst_cmpnei:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] != imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_xori	:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] ^ (u16)imm16;
        pc += 4;
        break;

    case inst_bne:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] != reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_cmpeqi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] == imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_muli:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] * imm16;
        pc += 4;
        break;

    case inst_beq:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] == reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_cmpgeui:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] >= (u16)imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_andhi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] & (imm16 << 16);
        pc += 4;
        break;


    case inst_bgeu:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((u32)reg[a] >= (u32)reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_ldhio:
    case inst_cmpltui:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] < (u16)imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

//    case inst_custom:
    case inst_initd:
    case inst_flushda:
    case inst_flushd:
        pc += 4;
        break;

    case inst_orhi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] | (imm16 << 16);
        pc += 4;
        break;

    case inst_bltu:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] < reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_R_type: /* R-type */
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        c = (isn >> 17) & 0x1f;
        opx = (isn >> 6) & 0x7ff;
        opx2 = (isn >> 11) & 0x3f;
        imm5 = (isn >> 6) & 0x1f;
        switch (opx2) {
        case 0x01: /* eret */
            if (trace_int)
                printf("eret; ea %08x, status %08x -> %08x\n",
                       ea, status, estatus);
            status = estatus;
            pc = ea;
            break;
        case 0x02: /* roli */
            reg[c] = rotate_left(reg[a], imm5);
            pc += 4;
            break;
        case 0x03: /* rol */
            reg[c] = rotate_left(reg[a], reg[b] & 0x1f);
            pc += 4;
            break;
        case 0x04: /* flushp */
            pc += 4;
            break;
        case 0x05: /* ret */
            pc = ra;
local_show = 0;
            break;
        case 0x06: /* nor */
            reg[c] = ~(reg[a] | reg[b]);
            pc += 4;
            break;
        case 0x07: /* mulxuu */
            llv = (long long)(u32)reg[a] * reg[b];
            reg[c] = llv >> 32;
            pc += 4;
            break;
        case 0x08: /* cmpge */
            if ((s32)reg[a] >= (s32)reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x09: /* bret */
            status = bstatus;
            pc = ba;
            break;
        case 0x0b: /* ror */
            reg[c] = rotate_right(reg[a], reg[b] & 0x1f);
            pc += 4;
            break;
        case 0x0c: /* flushi */
            pc += 4;
            break;
        case 0x0d: /* jmp */
            pc = reg[a];
            break;
        case 0x0e: /* and */
            reg[c] = reg[a] & reg[b];
            pc += 4;
            break;
        case 0x10: /* cmplt */
            if ((s32)reg[a] < (s32)reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x12: /* slli */
            reg[c] = reg[a] << imm5;
            pc += 4;
            break;
        case 0x13: /* sll */
            reg[c] = reg[a] << (reg[b] & 0x1f);
            pc += 4;
            break;
        case 0x16: /* or */
            reg[c] = reg[a] | reg[b];
            pc += 4;
            break;
        case 0x17: /* mulxsu */
            llv = (long long)(s32)reg[a] * reg[b];
            reg[c] = llv >> 32;
            pc += 4;
            break;
        case 0x18: /* cmpne */
            if (reg[a] != reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x1a: /* srli */
            reg[c] = reg[a] >> imm5;
            pc += 4;
            break;
        case 0x1b: /* srl */
            reg[c] = reg[a] >> (reg[b] & 0x1f);
            pc += 4;
            break;
        case 0x1c: /* nextpc */
            reg[c] = pc + 4;
            pc += 4;
            break;
        case 0x1d: /* callr */
            ra = pc + 4;
            pc = reg[a];
            break;
        case 0x1e: /* xor */
            reg[c] = reg[a] ^ reg[b];
            pc += 4;
            break;
        case 0x1f: /* mulxss */
            llv = (long long)(s32)reg[a] * (s32)reg[b];
            reg[c] = llv >> 32;
            pc += 4;
            break;
        case 0x20: /* cmpeq */
            if (reg[a] == reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x24: /* divu */
            reg[c] = reg[a] / reg[b];
            pc += 4;
            break;
        case 0x25: /* div */
            reg[c] = (s32)reg[a] / (s32)reg[b];
            pc += 4;
            break;
        case 0x26: /* rdctl */
            reg[c] = ctl_reg[imm5];
            pc += 4;
            break;
        case 0x27: /* mul */
            reg[c] = reg[a] * reg[b];
            pc += 4;
            break;
        case 0x28: /* cmpgeu */
            if (reg[a] >= reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x29: /* initi */
            pc += 4;
            break;
        case 0x2d: /* trap */
            estatus = status;
            status &= ~U;
            ea = pc + 4;
            pc = exception_handler;
            break;
        case 0x2e: /* wrctl */
            ctl_reg[imm5] = reg[a];
            if (trace_int && imm5) printf("wrctl[%d] <- %08x\n", imm5, reg[a]);

            if (imm5 == 0 || imm5 == 1 || imm5 == 2)
                ctl_reg[imm5] &= 1;

            if (imm5 == 3)
                ipending = ienable & irq_latch;

            pc += 4;
            break;
        case 0x30: /* cmpltu */
            if (reg[a] < reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x31: /* add */
            llv = reg[a] + reg[b];
            carry = llv >> 32 ? 1 : 0;
            reg[c] = llv;
            pc += 4;
            break;
        case 0x34: /* break */
            bstatus = status;
            status &= ~U;
            ba = pc + 4;
            pc = break_handler;
            break;
        case 0x36: /* sync */
            pc += 4;
            break;
        case 0x39: /* sub */
            reg[c] = reg[a] - reg[b];
            pc += 4;
            break;
        case 0x3a: /* srai */
            reg[c] = (s32)reg[a] >> imm5;
            pc += 4;
            break;
        case 0x3b: /* sra */
            reg[c] = (s32)reg[a] >> (reg[b] & 0x1f);
            pc += 4;
            break;
        default:
            printf("unknown r-type opcode %x isn %08x @ %08x\n",
                   opx2, isn, pc);
            exit(1);
        }
        break;

    case inst_xorhi :
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] ^ (imm16 << 16);
        pc += 4;
        break;

    default:
        printf("unknown opcode 0x%x isn %08x @ %08x\n", op, isn, pc);
        exit(1);
    }
}

static inline void
inst_poll(void)
{
    jtag_uart_poll();
}

static u32 initial_pc;
static int initial_pc_set;

static u32 initial_reg[32];
static int initial_reg_set[32];

void
inst_set_pc(u32 pc)
{
    initial_pc = pc;
    initial_pc_set = 1;
}

void
inst_set_regs(int first, int last, u32 *regs)
{
    int i;

    for (i = first; i <= last; i++) {
        initial_reg_set[i] = 1;
        initial_reg[i] = regs[i-first];
    }
}

void
inst_run(int max_cycles, int show)
{
    int cycles, pollcount;
    int max_poll = 2000000;
    int i;

    if (initial_pc_set)
        pc = initial_pc;

    for (i = 0; i < 32; i++) {
        if (initial_reg_set[i]) {
            reg[i] = initial_reg[i];
            printf("initial r%d = %08x\n", i, reg[i]);
        }
    }

    cycles = 0;
    pollcount = 0;
    while (1) {
        inst_loop(show || local_show);

        if (max_cycles && ++cycles > max_cycles)
            break;

        if (++pollcount > max_poll) {
            pollcount = 0;
            inst_poll();
        }
    }

    if (show) {
        dump_mem();
    }
}

void
inst_final_dump(void)
{
    printf("\npc %08x\n", pc);
    show_regs();
}

void
inst_init(void)
{
    int i;

    pc = 0x2f000000;
    exception_handler = 0x800020;
    status = 0;

    for (i = 0; i < 32; i++)
        reg[i] = 0;

#if 0
    {
        int i;
        u32 addr;

        addr = 0x009db940;
        for (i = 0; i < 16; i++) {
            printf("%08x %08x\n", addr, mem_read_d32(addr));
            addr += 4;
        }
    }
#endif
}

/*
8    ~ISP1362_avalon_slave_1_irq_n_from_sa,
7    ~ISP1362_avalon_slave_0_irq_n_from_sa,
6    DM9000A_avalon_slave_0_irq_from_sa,
5    button_pio_s1_irq_from_sa,
4    timer_1_s1_irq_from_sa,
3    timer_0_s1_irq_from_sa,
2    uart_0_s1_irq_from_sa,
1    jtag_uart_0_avalon_jtag_slave_irq_from_sa,
0    epcs_controller_epcs_control_port_irq_from_sa
*/

void set_local_show(void) { local_show = 1; }

void
interrupt_generate(int irq)
{
    if (trace_int) printf("interrupt_generate %d\n", irq);

    irq_latch |= (1 << irq);
    ipending = ienable & irq_latch;
}

void
interrupt_ack(int irq)
{
    if (trace_int) printf("interrupt_ack %d\n", irq);

    irq_latch &= ~(1 << irq);
    ipending = ienable & irq_latch;
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
