/**************************************************************************** 
                                  elfread.c                                   
                              ------------------                              
    date            : May 2003                                               
    copyright       : Institut de Recherche en Informatique de Toulouse       
    author          : Marc Finet                                              
    email           : finet@irit.fr, sainrat@irit.fr                        
 ****************************************************************************/
                                                                              
/**************************************************************************** 
 *                                                                          * 
 *   This program is free software; you can redistribute it and/or modify   * 
 *   it under the terms of the GNU General Public License as published by   * 
 *   the Free Software Foundation; either version 2 of the License, or      * 
 *   (at your option) any later version.                                    * 
 *                                                                          * 
 ****************************************************************************/

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <stdarg.h>
#include "elf.h"
#include "elfread.h"
#include "elfconvert.h"
#include "elfmisc.h"

int Is_Elf_Little = 0;

Elf_Tables Tables = {0,NULL,-1,NULL,
                     -1,NULL,NULL,
                     -1,NULL,NULL,
                     -1,NULL,NULL,
                     0,NULL};

struct text_info Text;
struct data_info Data;
Elf32_Ehdr Ehdr;

static int valid_elf_read;

void elf_error(char *s,...){
    va_list argp;
    fprintf(stderr, "Emulator Error: ");
    va_start(argp,s);
    vfprintf(stderr, s, argp);
    va_end(argp);
    exit(21);
}

void ElfReadHeader(int fd, Elf32_Ehdr *Ehdr);
void ElfReadSecHdrTbl(int fd, const Elf32_Ehdr *Ehdr);
void ElfReadPgmHdrTbl(int fd, const Elf32_Ehdr *Ehdr);
void ElfReadSecNameTbl(int fd, const Elf32_Ehdr *Ehdr);
void ElfReadSymTbl(int fd, const Elf32_Ehdr *Ehdr);
void ElfReadDySymTbl(int fd, const Elf32_Ehdr *Ehdr);
void ElfReadTextSecs(int fd, const Elf32_Ehdr *Ehdr);
void ElfReadDataSecs(int fd, const Elf32_Ehdr *Ehdr);
void ElfInsertDataSec(const Elf32_Shdr *hdr,int fd);
void ElfCheckExec(const Elf32_Ehdr *Ehdr);
        
void ElfRead(int elf){
     valid_elf_read = 0;
     ElfReadHeader(elf,&Ehdr);
     ElfCheckExec(&Ehdr);
     ElfReadPgmHdrTbl(elf,&Ehdr);
     ElfReadSecHdrTbl(elf,&Ehdr);
     ElfReadSecNameTbl(elf,&Ehdr);
     ElfReadSymTbl(elf,&Ehdr);
     ElfReadTextSecs(elf,&Ehdr);
     ElfReadDataSecs(elf,&Ehdr);
     valid_elf_read = 1;
}


void ElfReadHeader(int fd, Elf32_Ehdr *Ehdr){
    int foffset;
    foffset = lseek(fd,0,SEEK_CUR);
    lseek(fd,0,SEEK_SET);
    if(read(fd,Ehdr,sizeof(Elf32_Ehdr)) != sizeof(Elf32_Ehdr))
        elf_error("ReadElfHeader: Could not read Elf Header properly.\n");
    if(bcmp(Ehdr->e_ident,ELFMAG,4))
        elf_error("ReadElfHeader: Not a valid ELF File.\n");
    if(Ehdr->e_ident[EI_DATA] == 1)
	Is_Elf_Little = 1;
    else if(Ehdr->e_ident[EI_DATA] == 2)
	Is_Elf_Little = 0;
    else{
	elf_error("ReadElfHeader: Invalid Data Encoding of ELF file.\n");
    }
    if(Ehdr->e_ident[EI_CLASS] == ELFCLASS64)
	elf_error("ReadElfHeader: ELF File contains 64-bit objects which are not supported.\n");
    lseek(fd,foffset,SEEK_SET);
    if (is_host_little() != Is_Elf_Little) 
        ConvertElfHeader(Ehdr);
    return;
}

void ElfCheckExec(const Elf32_Ehdr *Ehdr){
    if(Ehdr->e_type != ET_EXEC)
        elf_error("ElfCheckExec: File is not an executable !\n");
    return;
}

void ElfReadPgmHdrTbl(int fd,const Elf32_Ehdr *Ehdr){
    int32_t i;
    if(Ehdr->e_phoff == 0)
        elf_error("ElfReadSecHdrTbl: No Sections !\n");   
    lseek(fd,Ehdr->e_phoff,SEEK_SET);    
    Tables.pgm_hdr_tbl_size = Ehdr->e_phnum;
    Tables.pgm_header_tbl = (Elf32_Phdr *)malloc(Ehdr->e_phnum * sizeof(Elf32_Phdr));
    if(Tables.pgm_header_tbl == NULL)
	elf_error("ElfReadSecHdrTbl: Could not allocate memory for program header table.\n");
    if(read(fd,Tables.pgm_header_tbl,(Ehdr->e_phnum*sizeof(Elf32_Phdr))) 
           != (Ehdr->e_phnum*sizeof(Elf32_Phdr)))
	elf_error("ElfReadSecHdrTbl: Could not read program header table properly.\n");
    for(i=0;i<Ehdr->e_phnum;++i){
        if (is_host_little() != Is_Elf_Little) 
            ConvertPgmHeader(&Tables.pgm_header_tbl[i]);
    }
   return;
}

void ElfReadSecHdrTbl(int fd, const Elf32_Ehdr *Ehdr){
    int32_t i,foffset;
    if(Ehdr->e_shoff == 0)
        elf_error("ElfReadSecHdrTbl: No Sections !\n");   
     foffset = lseek(fd,0,SEEK_CUR);
     lseek(fd,Ehdr->e_shoff,SEEK_SET);
     Tables.sechdr_tbl_size = Ehdr->e_shnum;
     Tables.sec_header_tbl = (Elf32_Shdr *)malloc(Ehdr->e_shnum * sizeof(Elf32_Shdr));
    if(Tables.sec_header_tbl == NULL)
	elf_error("ElfReadSecHdrTbl: Could not allocate memory for section header table.\n");
    if(read(fd,Tables.sec_header_tbl,(Ehdr->e_shnum*sizeof(Elf32_Shdr))) 
           != (Ehdr->e_shnum*sizeof(Elf32_Shdr)))
	elf_error("ElfReadSecHdrTbl: Could not read section header table properly.\n");
    for(i=0;i<Ehdr->e_shnum;++i){
        if (is_host_little() != Is_Elf_Little) 
            ConvertSecHeader(&Tables.sec_header_tbl[i]);
    }
    lseek(fd,foffset,SEEK_SET);
    return;
}

void ElfReadSecNameTbl(int fd, const Elf32_Ehdr *Ehdr){
    int foffset;
    Elf32_Shdr Eshdr;
    if(Ehdr->e_shoff == 0)
        elf_error("Undefined error\n");   
    if(Tables.secnmtbl_ndx == 0)
        elf_error("ElfReadSecNameTbl: Sec_name_tbl_ndx = 0\n");
    else if(Tables.secnmtbl_ndx > 0)
	return; /* Already red the table */
    Tables.secnmtbl_ndx = Ehdr->e_shstrndx;
    foffset = lseek(fd,0,SEEK_CUR);
    lseek(fd,Ehdr->e_shoff+Ehdr->e_shstrndx*Ehdr->e_shentsize,SEEK_SET);
    if(read(fd,&Eshdr,sizeof(Eshdr)) != sizeof(Eshdr))
	elf_error("ReadSecNameTbl: Could not read string table header properly.\n");
    
    if (is_host_little() != Is_Elf_Little)
        ConvertSecHeader(&Eshdr);
    Tables.sec_name_tbl = (char *)malloc(Eshdr.sh_size);	

    if(Tables.sec_name_tbl == NULL)
	elf_error("ReadSecNameTbl: Colud not allocate memory for string table.\n");
    lseek(fd,Eshdr.sh_offset,SEEK_SET);

    if(read(fd,Tables.sec_name_tbl,Eshdr.sh_size) != Eshdr.sh_size)
	elf_error("ReadSecNameTbl: Colud not read string table properly.\n");

    lseek(fd,foffset,SEEK_SET);
    return;
}


void ElfReadSymTbl(int fd, const Elf32_Ehdr *Ehdr){
    int32_t i,j,foffset;
    if(Ehdr->e_shoff == 0)
        elf_error("ElfReadSymTbl: Undefined error\n");   
    if(Tables.symtbl_ndx == 0)
	elf_error("ElfReadSymTbl: Undefined error\n");
    else if(Tables.symtbl_ndx > 0)
	return; /* already done */

    foffset = lseek(fd,0,SEEK_CUR);
    lseek(fd,Ehdr->e_shoff,SEEK_SET);
    for(i=0;i<Ehdr->e_shnum;++i){
	if(Tables.sec_header_tbl[i].sh_type == SHT_SYMTAB)
	break;
    }
    if(Ehdr->e_shnum == i)
	elf_error("ElfReadSymTbl: No Symbol Table present in the binary.\n");
    Tables.symtbl_ndx = i;
    lseek(fd,Tables.sec_header_tbl[i].sh_offset,SEEK_SET);
    Tables.sym_tbl = (Elf32_Sym *)malloc(Tables.sec_header_tbl[i].sh_size);
    if(Tables.sym_tbl == NULL) 
	elf_error("ReadSymTbl: Could not allocate memory for symbol table.\n");
    if(read(fd,Tables.sym_tbl,Tables.sec_header_tbl[i].sh_size) != Tables.sec_header_tbl[i].sh_size)
	elf_error("ReadSymTbl: Could not read symbol table properly.\n");
    if(is_host_little() != Is_Elf_Little){
	for(j=0;j<(Tables.sec_header_tbl[i].sh_size/Tables.sec_header_tbl[i].sh_entsize);++j)
            ConvertSymTblEnt(&Tables.sym_tbl[j]);		
    }
    /* Got Symbol table now reading string table for it */
    i = Tables.sec_header_tbl[i].sh_link;
    Tables.symstr_tbl = (char *)malloc(Tables.sec_header_tbl[i].sh_size);
    if(Tables.symstr_tbl == NULL)
	elf_error("ReadSymTbl: Could not allocate memory for string table.\n");	

    lseek(fd,Tables.sec_header_tbl[i].sh_offset,SEEK_SET);
    if(read(fd,(char *)Tables.symstr_tbl,Tables.sec_header_tbl[i].sh_size) != 
           Tables.sec_header_tbl[i].sh_size)
	elf_error("ReadSymTbl: Could not read string table properly.\n");

    lseek(fd,foffset,SEEK_SET);
    return;
}

void ElfReadDySymTbl(int fd, const Elf32_Ehdr *Ehdr){
    //Elf32_Sym Esym;
    //Elf32_Shdr Eshdr;
    int i,foffset,j;
    if(Tables.dysymtbl_ndx == 0)
	elf_error("ElfReadDySymTbl: Undefined error\n");
    else if(Tables.dysymtbl_ndx > 0)
	return; /*already done*/
    
    foffset = lseek(fd,0,SEEK_CUR);
    lseek(fd,Ehdr->e_shoff,SEEK_SET);
    for(i=0;i<Ehdr->e_shnum;++i){
	if(Tables.sec_header_tbl[i].sh_type == SHT_DYNSYM)
	break;
    }
    if(Ehdr->e_shnum == i)
        elf_error("ElfReadDySymTbl: Cannot find dynamic Symbol\n");
    Tables.dysymtbl_ndx = i;
    lseek(fd,Tables.sec_header_tbl[i].sh_offset,SEEK_SET);
    Tables.dysym_tbl = (Elf32_Sym *)malloc(Tables.sec_header_tbl[i].sh_size);
    if(Tables.dysym_tbl == NULL)
	elf_error("ReadDySymTbl: Could not allocate memory for dynamic symbol table.\n");
    if(read(fd,Tables.dysym_tbl,Tables.sec_header_tbl[i].sh_size) != Tables.sec_header_tbl[i].sh_size)
	elf_error("ReadDySymTbl: Could not read dynamic symbol table properly.\n");
    if(is_host_little() != Is_Elf_Little){
	for(j=0;j<(Tables.sec_header_tbl[i].sh_size/Tables.sec_header_tbl[i].sh_entsize);++j)
		ConvertSymTblEnt(&Tables.dysym_tbl[j]);
    }
    /* Got Symbol table now reading string table for it */
    i = Tables.sec_header_tbl[i].sh_link;
    Tables.dystr_tbl = (char *)malloc(Tables.sec_header_tbl[i].sh_size);
    if(Tables.dystr_tbl == NULL)
        elf_error("ReadDySymTbl: Could not allocate memory for stirng table.\n");	
    lseek(fd,Tables.sec_header_tbl[i].sh_offset,SEEK_SET);
    if(read(fd,Tables.dystr_tbl,Tables.sec_header_tbl[i].sh_size) != Tables.sec_header_tbl[i].sh_size)
	elf_error("ReadDySymTbl: Could not read stirng table properly.\n");
    lseek(fd,foffset,SEEK_SET);
    return;
}

void ElfReadTextSecs(int fd, const Elf32_Ehdr *Ehdr){
    int32_t i,foffset;
    struct text_secs *txt_sec,**ptr,*ptr1;
    if(Ehdr->e_shoff == 0){
	elf_error("Undefined error\n");
    }
    foffset = lseek(fd,0,SEEK_CUR);
    lseek(fd,Ehdr->e_shoff,SEEK_SET);
    for(i=0;i<Ehdr->e_shnum;++i){
//---
//printf("%s\n", &Tables.sec_name_tbl[Tables.sec_header_tbl[i].sh_name]);
//printf("sh_type %x, want %x; sh_flags %x, want %x\n",
//       Tables.sec_header_tbl[i].sh_type, SHT_PROGBITS,
//       Tables.sec_header_tbl[i].sh_flags, SHF_ALLOC | SHF_EXECINSTR);
//--
	if((Tables.sec_header_tbl[i].sh_type == SHT_PROGBITS) && 
	   ((Tables.sec_header_tbl[i].sh_flags & ~SHF_WRITE) == (SHF_ALLOC | SHF_EXECINSTR))){
	    txt_sec = (struct text_secs *)malloc(sizeof(struct text_secs));
	    if(txt_sec == NULL)
	        elf_error("ElfReadTextSecs: Could't allocate memory for tables.\n");
            strcpy(txt_sec->name,&Tables.sec_name_tbl[Tables.sec_header_tbl[i].sh_name]);
//printf("name %s\n", txt_sec->name);
            if(!strcmp(txt_sec->name,".text"))
                Text.txt_index = i;
            txt_sec->offset = Tables.sec_header_tbl[i].sh_offset;
            txt_sec->address = Tables.sec_header_tbl[i].sh_addr;
            txt_sec->size = Tables.sec_header_tbl[i].sh_size;
            txt_sec->next = NULL;
            txt_sec->bytes= (uint8_t *)malloc(txt_sec->size *sizeof(uint8_t));
            if (txt_sec->bytes==NULL)
                elf_error("ElfReadTextSecs: Could't allocate memory for text section.\n");
            lseek(fd,txt_sec->offset,SEEK_SET);
            if(read(fd,txt_sec->bytes,txt_sec->size) != txt_sec->size)
                elf_error("ElfReadTextSecs: Could not read \"%s\" section properly.\n",txt_sec->name);            
            /* set next ptr */
            ptr = &Text.secs;
            while(*ptr != NULL){
                if((*ptr)->address > txt_sec->address){
                    txt_sec->next = *ptr;
                    *ptr = txt_sec;
                    break;
                }
                ptr = &((*ptr)->next);
            }
            if(*ptr == NULL)
                *ptr = txt_sec;
        }
    }
    if(Text.secs == NULL)
	elf_error("ElfReadTextSecs: No Text sections found in ELF.\n");
    /* ??? */
    Text.address = Text.secs->address;
    ptr1 = Text.secs;
    while(ptr1->next != NULL){
        ptr1 = ptr1->next;
    }
    Text.size = ptr1->address + ptr1->size - Text.address;
    Text.bytes = (uint8_t *)malloc(Text.size); 
    if(Text.bytes == NULL)
        elf_error("ElfReadTextSecs: Could not allocate memory for instructions in the elf.\n");
    memset(Text.bytes,0,Text.size);
    ptr1 = Text.secs;
    while(ptr1 != NULL){
        if(!strcmp(ptr1->name,".text")){
            Text.txt_addr = ptr1->address;
            Text.txt_size = ptr1->size;
        }
        lseek(fd,ptr1->offset,SEEK_SET);
        if(read(fd,&Text.bytes[ptr1->address-Text.address],ptr1->size) != ptr1->size)
            elf_error("ElfReadTextSecs: Could not read \"%s\" section properly.\n",ptr1->name);
        ptr1 = ptr1->next;
    }
    lseek(fd,foffset,SEEK_SET);
    return;
}

void ElfReadDataSecs(int fd, const Elf32_Ehdr *Ehdr){
    int32_t i,foffset;
    //struct data_secs *ptr;
    //Elf32_Shdr txt_hdr;
    foffset = lseek(fd,0,SEEK_CUR);
    for(i=0;i<Ehdr->e_shnum;++i){
        if(Tables.sec_header_tbl[i].sh_type == SHT_PROGBITS){ 
            if(Tables.sec_header_tbl[i].sh_flags == (SHF_ALLOC | SHF_WRITE)){
                ElfInsertDataSec(&Tables.sec_header_tbl[i],fd);
            }
            else if(Tables.sec_header_tbl[i].sh_flags == (SHF_ALLOC)){
                ElfInsertDataSec(&Tables.sec_header_tbl[i],fd);
            }
        }
        else if(Tables.sec_header_tbl[i].sh_type == SHT_NOBITS){ 
            if(Tables.sec_header_tbl[i].sh_flags == (SHF_ALLOC | SHF_WRITE)){
                ElfInsertDataSec(&Tables.sec_header_tbl[i],fd);
            }
        }
    }
    Data.address = Data.secs->address;
    lseek(fd,foffset,SEEK_SET);    
    return;
}

void ElfInsertDataSec(const Elf32_Shdr *hdr,int fd){
    struct data_secs *data_sec,**ptr;
    data_sec = (struct data_secs *)malloc(sizeof(struct data_secs));
    if(data_sec == NULL)
        elf_error("ElfInsertDataSec: Cannot allocate memory\n");
    strcpy(data_sec->name,&Tables.sec_name_tbl[hdr->sh_name]);
    data_sec->offset = hdr->sh_offset;
    data_sec->address = hdr->sh_addr;
    data_sec->size = hdr->sh_size;
    data_sec->type = hdr->sh_type;
    data_sec->flags = hdr->sh_flags;
    data_sec->next = NULL;
    data_sec->bytes=(int8_t *)malloc(data_sec->size);
    if (data_sec->bytes==NULL)
        elf_error("ElfInsertDataSec: Cannot allocat memory !\n");
    if (strcmp(data_sec->name,".bss")!=0 && strcmp(data_sec->name,".sbss")){
        lseek(fd,data_sec->offset,SEEK_SET);
        if(read(fd,data_sec->bytes,data_sec->size)!=data_sec->size)
            elf_error("ElfInsertDataSec: Cannot allocat memory !\n");
    } else {
        memset(data_sec->bytes,0,data_sec->size);
    }
    ptr = &Data.secs;
    while(*ptr != NULL){
        if((*ptr)->address > data_sec->address){
            data_sec->next = *ptr;
            *ptr = data_sec;
            break;
        }
        ptr = &((*ptr)->next);
    }
    if(*ptr == NULL)
        *ptr = data_sec;
    return;
}

int ElfReadHashTbl(int fd, Elf32_Ehdr *Ehdr)
{
    Elf32_Shdr Eshdr;
    int i,foffset;

    if(Ehdr->e_shoff == 0){
        return(-1);
    }

    if(Tables.hashtbl_ndx == 0)
        return(-1);
    else if(Tables.hashtbl_ndx > 0)
        return(0); /* Already red the table */

    if(Ehdr->e_type != ET_EXEC && Ehdr->e_type != ET_DYN){
        Tables.hashtbl_ndx = 0;
        return(-1);
    }

    foffset = lseek(fd,0,SEEK_CUR);

    lseek(fd,Ehdr->e_shoff,SEEK_SET);

    /* Finding Hash Table */

    for(i=0;i<Ehdr->e_shnum;++i){

        if(read(fd,&Eshdr,sizeof(Eshdr)) != sizeof(Eshdr)){
            fprintf(stderr,"ReadHashTbl: Could not read section header for hash table properly.\n");
            lseek(fd,foffset,SEEK_SET);
            return(-1);
        }

        if (is_host_little() != Is_Elf_Little)
            ConvertSecHeader(&Eshdr);

        if(Eshdr.sh_type == SHT_HASH)
            break;

    }

    if(i == Ehdr->e_shnum){
        lseek(fd,foffset,SEEK_SET);
        Tables.hashtbl_ndx = 0;
        return(-1);
    }

    /* Got it reading Hash Table */

    Tables.hashtbl_ndx = i;

    lseek(fd,Eshdr.sh_offset,SEEK_SET);

    if(Eshdr.sh_entsize != sizeof(Elf32_Word)){
        fprintf(stderr,"ReadHashTbl: Could not read hash table unexpected Hash objects size.\n");
        Tables.hashtbl_ndx = 0;
        return(-1);
    }

    Tables.hash_tbl = (Elf32_Word *)malloc(Eshdr.sh_size);

    if(Tables.hash_tbl == NULL){
        fprintf(stderr,"ReadHashTbl: Could not allocate memory for hash table.\n");	
        Tables.hashtbl_ndx = -1;
        return(-1);
    }

    if(read(fd,Tables.hash_tbl,Eshdr.sh_size) != Eshdr.sh_size){
        fprintf(stderr,"ReadHashTbl: Could not read hash table properly.\n");
        free(Tables.hash_tbl);
        lseek(fd,foffset,SEEK_SET);
        Tables.hashtbl_ndx = 0;
        return(-1);
    }

    if(is_host_little() != Is_Elf_Little){
        for(i=0;i<(Eshdr.sh_size/Eshdr.sh_entsize);++i)
            Tables.hash_tbl[i] = ConvertByte4(Tables.hash_tbl[i]);
    }

    /* Reading Symbol table for this Hash Table */

    lseek(fd,Ehdr->e_shoff+Eshdr.sh_link*Ehdr->e_shentsize,SEEK_SET);

    if(read(fd,&Eshdr,sizeof(Eshdr)) != sizeof(Eshdr)){
        fprintf(stderr,"ReadHashTbl: Could not read section header for hash symbol table properly.\n");
        Tables.hashtbl_ndx = 0;
        free(Tables.hash_tbl);
        lseek(fd,foffset,SEEK_SET);
        return(-1);
    }

    if (is_host_little() != Is_Elf_Little)
        ConvertSecHeader(&Eshdr);

    lseek(fd,Eshdr.sh_offset,SEEK_SET);

    Tables.hashsym_tbl = (Elf32_Sym *)malloc(Eshdr.sh_size);

    if(Tables.hashsym_tbl == NULL){
        fprintf(stderr,"ReadHashTbl: Could not allocate memory for hash symbol table.\n");	
        Tables.hashtbl_ndx = -1;
        free(Tables.hash_tbl);
        return(-1);
    }

    if(read(fd,Tables.hashsym_tbl,Eshdr.sh_size) != Eshdr.sh_size){
        fprintf(stderr,"ReadHashTbl: Could not read hash symbol table properly.\n");
        Tables.hashtbl_ndx = 0;
        free(Tables.hash_tbl);
        free(Tables.hashsym_tbl);
        lseek(fd,foffset,SEEK_SET);
        return(-1);
    }

    if(is_host_little() != Is_Elf_Little){
        for(i=0;i<(Eshdr.sh_size/Eshdr.sh_entsize);++i)
            ConvertSymTblEnt(&Tables.hashsym_tbl[i]);
    }

    lseek(fd,foffset,SEEK_SET);

    return(0);

}


void ElfFreeTables()
{
    if(Tables.sechdr_tbl_size > 0)
        free(Tables.sec_header_tbl);

    if(Tables.secnmtbl_ndx > 0)
        free(Tables.sec_name_tbl);

    if(Tables.symtbl_ndx > 0){
        free(Tables.sym_tbl);
        free(Tables.symstr_tbl);
    }

    if(Tables.dysymtbl_ndx > 0){
        free(Tables.dysym_tbl);
        free(Tables.dystr_tbl);
    }

    if(Tables.hashtbl_ndx > 0){
        free(Tables.hash_tbl);
        free(Tables.hashsym_tbl);
    }

    Tables.sec_header_tbl = NULL;
    Tables.sec_name_tbl = NULL;
    Tables.sym_tbl = NULL;
    Tables.symstr_tbl = NULL;
    Tables.dysym_tbl = NULL;
    Tables.dystr_tbl = NULL;
    Tables.hash_tbl = NULL;
    Tables.hashsym_tbl = NULL;

    Tables.sechdr_tbl_size = 0;
    Tables.secnmtbl_ndx = -1;
    Tables.symtbl_ndx = -1;
    Tables.dysymtbl_ndx = -1;
    Tables.hashtbl_ndx = -1;

}

#if 0
void ElfFreeText(){
     if(Text.bytes != NULL)
         free(Text.bytes);
     if(Text.secs != NULL)
         ElfFreeTextSecs(Text.secs);    
     Text.secs = NULL;
     Text.bytes = NULL;     
}

void ElfFreeTextSecs(struct text_secs *ptr){
     if(ptr == NULL)
         return;
     else
         ElfFreeTextSecs(ptr->next);     
     free(ptr);     
     return;
}

void ElfFreeData(){
    if(Data.secs != NULL)
	ElfFreeDataSecs(Data.secs);
	Data.secs = NULL;
    }

void ElfFreeDataSecs(struct data_secs *ptr)
{
    if(ptr == NULL)
        return;
    else
        ElfFreeDataSecs(ptr->next);

    free(ptr);

    return;
}
		

#if 0
void ElfListTextSecs(int fd, const Elf32_Ehdr *Ehdr)
{
    Elf32_Sym Esym;
    Elf32_Shdr Eshdr;
    int i,foffset;
    char *text;

    if(Elf_Error)
        return;

    if(Ehdr->e_shoff == 0){
        return;
    }

    for(i=0;i<Tables.sechdr_tbl_size;++i){

        if((Tables.sec_header_tbl[i].sh_type == SHT_PROGBITS) && 
           (Tables.sec_header_tbl[i].sh_flags == (SHF_ALLOC | SHF_EXECINSTR))){
#if 0
            printf("Name: %s\n",
                   &Tables.sec_name_tbl[Tables.sec_header_tbl[i].sh_name]);
            printf("Offset: 0x%d\n",Tables.sec_header_tbl[i].sh_offset);
            printf("Size: %x\n",Tables.sec_header_tbl[i].sh_size);
            printf("addr: 0x%x\n",Tables.sec_header_tbl[i].sh_addr);
            printf("end: 0x%x\n",
            Tables.sec_header_tbl[i].sh_addr+Tables.sec_header_tbl[i].sh_size);
#endif
        }

    }

    return;

}


void ElfListFuncs()
{
    Elf32_Shdr *Eshdr;
    int32_t records,i;

    if(Tables.symtbl_ndx == 0)
        return;

    Eshdr = &Tables.sec_header_tbl[Tables.symtbl_ndx];

    records = Eshdr->sh_size/Eshdr->sh_entsize;

    for(i=0;i<records;++i){

        if(ELF32_ST_TYPE(Tables.sym_tbl[i].st_info) == 2){
#if 0
            printf("Name: %s\n",&Tables.symstr_tbl[Tables.sym_tbl[i].st_name]);
            printf("Addr: %x\n",Tables.sym_tbl[i].st_value);
            printf("Size: %x\n",Tables.sym_tbl[i].st_size);
            if(Tables.sym_tbl[i].st_value >= Text.address &&
               Tables.sym_tbl[i].st_value < (Text.address + Text.size))
                printf("Text sec: Yes\n");
            else
                printf("Text sec: No\n");
#endif
        }
    }
}
#endif

int16_t ElfReadInstrBytes( uint8_t *buf, int16_t no_bytes )
{
    static int32_t index = 0;
    int16_t i = 0;

    for(i=0;(i<no_bytes)&&(index<Text.size);++i){
        buf[i] = Text.bytes[index];
        ++index;
    }

    return(i);
}

/*
 * pour plus de simplicite: acces direct
 */ 
int16_t ElfReadInstrBytesFrom( uint8_t *buf, uint32_t start, int16_t no_bytes)
{
    int32_t index=start;
    uint32_t end_wanted=start+(uint32_t)no_bytes;
    if (start>Text.size)
        return(-1);
    no_bytes=(end_wanted>Text.size) ? Text.size-start : no_bytes;
    memcpy(buf,Text.bytes+index,no_bytes);
    return(no_bytes);   
}


void ElfListDataSecs(const Elf32_Ehdr *Ehdr)
{
    int i;
    if(Ehdr->e_shoff == 0){
        return;
    }

    for(i=0;i<Tables.sechdr_tbl_size;++i){

        if(Tables.sec_header_tbl[i].sh_type == SHT_PROGBITS){  
            if(Tables.sec_header_tbl[i].sh_flags == (SHF_ALLOC | SHF_WRITE)){
#if 0
                printf("Name: %s\n",
                       &Tables.sec_name_tbl[Tables.sec_header_tbl[i].sh_name]);
                printf("Offset: 0x%d\n",Tables.sec_header_tbl[i].sh_offset);
                printf("Size: %x\n",Tables.sec_header_tbl[i].sh_size);
                printf("addr: 0x%x\n",Tables.sec_header_tbl[i].sh_addr);
                printf("end: 0x%x\n",
            Tables.sec_header_tbl[i].sh_addr+Tables.sec_header_tbl[i].sh_size);
#endif
            }
            else if(Tables.sec_header_tbl[i].sh_flags == (SHF_ALLOC)){
#if 0
                printf("aName: %s\n",
                       &Tables.sec_name_tbl[Tables.sec_header_tbl[i].sh_name]);
                printf("Offset: 0x%d\n",Tables.sec_header_tbl[i].sh_offset);
                printf("Size: %x\n",Tables.sec_header_tbl[i].sh_size);
                printf("addr: 0x%x\n",Tables.sec_header_tbl[i].sh_addr);
                printf("end: 0x%x\n",
            Tables.sec_header_tbl[i].sh_addr+Tables.sec_header_tbl[i].sh_size);
#endif
            }

        }
        else if(Tables.sec_header_tbl[i].sh_type == SHT_NOBITS){  
            if(Tables.sec_header_tbl[i].sh_flags == (SHF_ALLOC | SHF_WRITE)){
#if 0
                printf("bName: %s\n",
                       &Tables.sec_name_tbl[Tables.sec_header_tbl[i].sh_name]);
                printf("Offset: 0x%d\n",Tables.sec_header_tbl[i].sh_offset);
                printf("Size: %x\n",Tables.sec_header_tbl[i].sh_size);
                printf("addr: 0x%x\n",Tables.sec_header_tbl[i].sh_addr);
                printf("end: 0x%x\n",
            Tables.sec_header_tbl[i].sh_addr+Tables.sec_header_tbl[i].sh_size);
#endif
            }
        }

    }

    return;

}



uint32_t GetMainAddr()
{
    Elf32_Shdr *Eshdr;
    int32_t records,i;

    if(Tables.symtbl_ndx == 0){
        fprintf(stderr,"GetMainAddre: No symbol table present.\n");
        return 0;
    }

    Eshdr = &Tables.sec_header_tbl[Tables.symtbl_ndx];

    records = Eshdr->sh_size/Eshdr->sh_entsize;

		
    for(i=0;i<records;++i){

        if(!strcmp(&Tables.symstr_tbl[Tables.sym_tbl[i].st_name],"main"))
            return(Tables.sym_tbl[i].st_value);
    }
		

    fprintf(stderr,"GetMainAddre: Warning - Could't get main's address.\n"
            "Using text section start address.\n");

    return (Text.txt_addr);
}

uint32_t GetCodeBaseAddr(){
    return (Text.txt_addr);
}

uint32_t GetCodeSize(void){
    return (Text.txt_size);
}

/* Renvoie d'adresse des donnees */
uint32_t GetDataAddr(void){
    uint32_t addr=0;
    int i;
    for(i=0;i<Tables.pgm_hdr_tbl_size;i++){
        if ( (Tables.pgm_header_tbl[i].p_type == PT_LOAD)       /* Loadable  and  */
             && ((Tables.pgm_header_tbl[i].p_flags & PF_X)==0) )/* not executable */
            return Tables.pgm_header_tbl[i].p_vaddr;            /* not clear why 1st */
    }
    return addr;                
}

/* Renvoie la taille des donnees */
uint32_t GetDataSize(void){
    uint32_t size=0;
    int i;
    for(i=0;i<Tables.pgm_hdr_tbl_size;i++){
        if ( (Tables.pgm_header_tbl[i].p_type == PT_LOAD)       /* Loadable  and  */
             && ((Tables.pgm_header_tbl[i].p_flags & PF_X)==0) )/* not executable */
            return Tables.pgm_header_tbl[i].p_memsz;            /* not clear why 1st */
    }
    return size;                
}
#endif

int
find_func_loc(uint32_t pc, char **pname, int *poffset)
{
    Elf32_Shdr *Eshdr;
    int32_t records, i;

    if (!valid_elf_read)
        return -1;

    if(Tables.symtbl_ndx == 0){
	    printf("No symbol table present.\n");
	    return -1;
    }

    Eshdr = &Tables.sec_header_tbl[Tables.symtbl_ndx];
    
    records = Eshdr->sh_size/Eshdr->sh_entsize;
		
    for (i = 0; i < records; ++i) {
	    if (Tables.sym_tbl[i].st_value == pc) {
		    *pname = &Tables.symstr_tbl[Tables.sym_tbl[i].st_name];
		    *poffset = 0;
		    return 0;
	    }
    }
		
    return -1;
}



/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
