/*
 * isn.c
 *
 * simple NIOS-II simulator instruction decode
 * Brad Parker <brad@heeltoe.com>
 * 10/23/07
 *
 * added MMU/TLB
 * 6/30/10
 *
 * Copyright (C) 2007,2010 Brad Parker
 */

#include <stdio.h>
#include <stdlib.h>

#include "sim.h"
#include "ops.h"

u32 pc;
#define ra reg[31]
#define ba reg[30]
#define ea reg[29]
u32 reg[32];

u32 ctl_reg[32];
#define status ctl_reg[0]
#define estatus ctl_reg[1]
#define bstatus ctl_reg[2]
#define ienable ctl_reg[3]
#define ipending ctl_reg[4]
#define cpuid ctl_reg[5]

#define exception ctl_reg[7]
#define pteaddr ctl_reg[8]
#define tlbacc ctl_reg[9]
#define tlbmisc ctl_reg[10]
#define badaddr ctl_reg[12]
#define config ctl_reg[13]
#define mpubase ctl_reg[14]
#define mpuacc ctl_reg[15]

#define WE (1<<18)

u32 irq_latch;

#define PIE		(1 << 0)
#define U		(1 << 1)
#define EH		(1 << 2)
#define IH		(1 << 3)
#define IL_MASK		(0x3f << 4)
#define CRS_MASK	(0x3f << 10)
#define PRS_MASK	(0x3f << 16)
#define NMI		(1 << 22)
#define RSIE		(1 << 23)

u32 exception_handler;
u32 fast_tlb_handler;
u32 break_handler;

u32 isn;
u32 op;
u32 opx;
u32 opx2;
int carry;

char pe, ani;
int mmu_miss_active;

int global_show;
static int local_show;
int trace_ufetch;
int trace_trap;
int trace_int;
int trace_eret;
int trace_tlb;
int trace_tlb_write;
int trace_tlb_read;
int trace_tlb_miss;
int trace_tlb_perm;
int trace_tlb_reg;

unsigned long cycles;

void
show_regs(void)
{
    int i, j;

    for (i = 0; i < 32; i += 8) {
        printf("R%02d: ", i);
        for (j = 0; j < 8; j++) {
            printf("%08x ", reg[i+j]);
        }
        printf("\n");
    }

    printf("ctl  ");
    for (j = 0; j < 8; j++) {
        printf("%08x ", ctl_reg[j]);
    }
    printf("\n");

    printf("     ");
    for (j = 8; j < 16; j++) {
        printf("%08x ", ctl_reg[j]);
    }
    printf("\n");

    printf("irq %08x\n", irq_latch);

}

void
show_state(int verbose)
{
    char *name;
    int offset;

    if (verbose) {
        printf("\n");
        printf("PC:  %08x, RA %08x\n", pc, ra);
        disas(pc, isn);

        if (find_func_loc(pc, &name, &offset) == 0) {
            printf("%s+0x%x\n", name, offset);
        }

        show_regs();
    } else {
        int do_regs;

        if (find_func_loc(pc, &name, &offset) == 0) {
            printf("%s+0x%x\n", name, offset);
            do_regs = 1;
        } else
            do_regs = 0;

        printf("EA %08x RA %08x PC ", ea, ra);
        disas(pc, isn);

//        if (do_regs)
//            show_regs();
    }

}

static inline u32
rotate_left(u32 value, int bitstorotate)
{
    unsigned int tmp;
    int mask;

    /* determine which bits will be impacted by the rotate */
    if (bitstorotate == 0)
        mask = 0;
    else
        mask = (int)0x80000000 >> bitstorotate;
		
    /* save off the affected bits */
    tmp = (value & mask) >> (32 - bitstorotate);
		
    /* perform the actual rotate */
    /* add the rotated bits back in (in the proper location) */
    return (value << bitstorotate) | tmp;
}

static inline u32
rotate_right(u32 value, int bitstorotate)
{
    return rotate_left(value, 32 - bitstorotate);
}

#define TLB_NUM_ENTRIES 128
#define TLB_NUM_WAYS	16
#define TLB_NUM_LINES	8	/* 128/16 */

/*
tag, pid, g, x, w, r, c, pfn

tag vpn[19:3]
line vpn[2:0]
*/
struct {
    u32 vpn;
    u16 pid;
    u8 g,x,w,r,c;
    u32 tlb;
#define TLB_C (1<<24)
#define TLB_R (1<<23)
#define TLB_W (1<<22)
#define TLB_X (1<<21)
#define TLB_G (1<<20)
} tlb[TLB_NUM_LINES][TLB_NUM_WAYS];

int way_ctr;

void
tlbwrite(u32 value)
{
    int way, line, next_way;

    /* use pteaddr.VPN tlbmisc.WAY tlbmisc.pid */

    line = (pteaddr >> 2) & 0x7;
    way = (tlbmisc >> 20) & 0xf;

    tlb[line][way].vpn = (pteaddr >> 2) & 0xfffff;
    tlb[line][way].pid = (tlbmisc >> 4) & 0x3fff;
    tlb[line][way].g = (value & TLB_G) ? 1 : 0;
    tlb[line][way].x = (value & TLB_X) ? 1 : 0;
    tlb[line][way].w = (value & TLB_W) ? 1 : 0;
    tlb[line][way].r = (value & TLB_R) ? 1 : 0;
    tlb[line][way].c = (value & TLB_C) ? 1 : 0;
    tlb[line][way].tlb = value;

    next_way = way + 1;
    if (next_way == TLB_NUM_WAYS)
        next_way = 0;

    if (trace_tlb || trace_tlb_write)
        printf("pc=%08x tlb[%d][%d] write vpn=%x pid=%x tlb=%x <- "
               "pteaddr %x tlbacc %x tlbmisc %x\n", 
               pc, line, way, 
               tlb[line][way].vpn, tlb[line][way].pid, tlb[line][way].tlb,
               pteaddr, value, tlbmisc);

    /* tlbmisc.WAY++ */
    tlbmisc =
        (tlbmisc & ~(0xf << 20)) |
        (next_way << 20);
}

void
tlbread(void)
{
    int way, line;

    if (trace_tlb)
        printf("tlbread pteaddr %x tlbmisc %x\n", pteaddr, tlbmisc);

    way = (tlbmisc >> 20) & 0xf;
    line = (pteaddr >> 2) & 0x7;

    tlbacc = tlb[line][way].tlb;
    pteaddr = (pteaddr & ~(0xfffff << 2)) | (tlb[line][way].vpn << 2);
    tlbmisc = (tlbmisc & ~(0x3fff << 4)) | (tlb[line][way].pid << 4);

    if (trace_tlb || trace_tlb_read)
        printf("tlb[%d][%d] read %x %x -> pteaddr %x tlbacc %x tlbmisc %x\n", 
               line, way, pteaddr, tlbmisc, pteaddr, tlbacc, tlbmisc);
}

void
tlbdump(void)
{
    int w, l;

    for (l = 0; l < TLB_NUM_LINES; l++) {
        for (w = 0; w < TLB_NUM_WAYS; w++) {
            printf("tlb[%d][%d].vpn %08x tlb %08x\n",
                   w, l, tlb[l][w].vpn, tlb[l][w].tlb);
        }
    }
}

#if 0
void
dumppgt(void)
{
    u32 base, table, p, offset;
    int i,l;

    base = 0xd0289000;
    table = mem_read_d32(base);

    printf("base: %08x\n", base);
    printf("table: %08x\n", table);

    offset = 0;
    for (l = 0; l < 32; l++) {
        printf("%04x: ", offset);
        for (i = 0; i < 8; i++) {
            p = mem_read_d32(table + offset);
            offset += 4;
            printf("%08x ", p);
        }
        printf("\n");
    }

    offset = 0xaa000;
    for (l = 0; l < 32; l++) {
        printf("%04x: ", offset);
        for (i = 0; i < 8; i++) {
            p = mem_read_d32(table + offset);
            offset += 4;
            printf("%08x ", p);
        }
        printf("\n");
    }
}
#endif

void tlblookup(u32 addr)
{
    u32 base, table, p, offset, index, final;
    u32 pe, v;

    /*
     * phys = *((*pgd_current) + ((pteaddr >> 12) << 2)) +
     *        (pteaddr & 0xfff))
     */

    base = 0xd0289000;
    table = mem_read_d32(base);

    pe = (addr >> 12) << 2;
    index = (pe >> 12) << 2;
    offset = pe & 0xfff;

    printf("base: %08x\n", base);
    printf("table: %08x\n", table);
    printf("offset %x, index %x\n", offset, index);

    v = mem_read_d32(table + offset);
    printf("%08x -> %08x\n", table + offset, v);
    printf("%08x -> %08x\n", v, mem_read_d32(v));
}

int
tlbcheck(u32 addr, int iord, u32 *newaddr, int *ex)
{
    int way, line, hit;
    u32 vpn;
    u16 pid;

    vpn = addr >> 12;
    line = vpn & 7;
    pid = (tlbmisc >> 4) & 0x3fff;

    if (trace_tlb)
        printf("tlbcheck pc %x addr %x; vpn %x line %x\n", pc, addr, vpn, line);

    hit = 0;
    for (way = 0; way < TLB_NUM_WAYS; way++) {
        if (vpn == tlb[line][way].vpn &&
            (tlb[line][way].g || pid == tlb[line][way].pid))
        {
            if (trace_tlb)
                printf("tlbcheck: match %d -> %08x\n",
                       way, tlb[line][way].tlb);
            hit = 1;
            break;
        }
    }

    if (hit == 0) {
        mmu_miss_active = 1;

        if (trace_tlb) {
            if (trace_tlb) printf("tlbcheck: miss\n");
#if 0
            for (way = 0; way < TLB_NUM_WAYS; way++)
                printf("tlbcheck %d; %08x %08x\n",
                       way, vpn, tlb[line][way].vpn);
            //dumppgt();
            tlblookup(addr);
#endif
        }

        return 1;
    }

#define D_R (1<<23)
#define D_W (1<<22)
#define I_X (1<<21)

    if ((tlb[line][way].tlb & iord) == 0) {
        if (iord & I_X) *ex = 13;
        if (iord & D_R) *ex = 14;
        if (iord & D_W) *ex = 15;
        if (trace_tlb || trace_tlb_perm)
            printf("tlbcheck: perm [%d][%d] tlb %08x iord %08x %08x\n",
                   line, way, tlb[line][way].tlb,
                   iord, tlb[line][way].tlb & (0xf << 20));
        return 2;
    }
    
    *newaddr = ((tlb[line][way].tlb & 0xfffff) << 12) | (addr & 0xfff);

    if (trace_tlb)
        printf("tlb hit %08x -> %08x\n", addr, *newaddr);

    return 0;
}

#define TLBMAP(aa, access) { \
if (((aa >> 24)&0xff) < 0xc0) \
{  int ret, ex; \
    ret = tlbcheck(aa, access, &maddr, &ex); \
    if (ret == 1) { \
        estatus = status;                       \
        status &= ~(U | PIE)/*~U*/;             \
        ea = pc + 4;  \
        if (trace_tlb_miss) printf("addr miss; pc=%08x, addr=%08x\n", pc, aa); \
        pteaddr = (pteaddr & ~(0xfffff << 2)) | ((aa >> 12) << 2); \
        tlbmisc |= 1;                                              \
        tlbmisc &= ~2;                                             \
        badaddr = aa;                                              \
        pc = fast_tlb_handler;                                     \
        break;                                                     \
    }  \
    if (ret == 2) { \
            estatus = status; \
            status &= ~(U | PIE)/*~U*/; \
            ea = pc + 4; \
            exception = ex << 2; \
            if (trace_tlb_miss) printf("addr exception; pc=%08x, addr=%08x\n",pc,aa); \
            pteaddr = (pteaddr & ~(0xfffff << 2)) | ((aa >> 12) << 2);  \
            tlbmisc |= 1;                                               \
            tlbmisc |= 2;                                               \
            badaddr = aa;                                               \
            pc = exception_handler;                                     \
            break;                                                      \
    } \
    trace_tlb = 0;  \
    aa = maddr;  \
} }

static inline void
inst_loop(int show)
{
    u32 a, b, c;
    u32 imm26, addr, maddr, mpc;
    s16 imm16;
    u8 imm5;
    long long llv;

    timer_update();

    /* interrupt? */
    if ((ipending & ienable) && (status & PIE)) {
        estatus = status | PIE;
        status &= ~(U | PIE);
        if (trace_int)
            printf("interrupt: pc %08x, new pc %08x\n", pc, exception_handler);
        ea = pc + 4;
        exception = 2 << 2;
        pc = exception_handler;
    }

    if (trace_ufetch && (status & U))
        printf("USER FETCH! %x\n", pc);

    if (((pc >> 24)&0xff) < 0xc0) {
        int ret, ex;
        ret = tlbcheck(pc, I_X, &mpc, &ex);
        if (ret == 1) {
            estatus = status; 
            status &= ~(U | PIE)/*~U*/;
            ea = pc + 4;
            if (trace_tlb_miss) printf("pc miss; pc %x\n", pc);
            pteaddr = (pteaddr & ~(0xfffff << 2)) | ((pc >> 12) << 2);
            tlbmisc &= ~1;
            tlbmisc &= ~2;
            badaddr = pc;
            pc = fast_tlb_handler;
            return;
        }
        if (ret == 2) {
            estatus = status;
            status &= ~(U | PIE)/*~U*/;
            ea = pc + 4;
            exception = ex << 2;
            if (trace_tlb_miss) printf("pc exception; pc %x\n", pc);
            pteaddr = (pteaddr & ~(0xfffff << 2)) | ((pc >> 12) << 2);
            tlbmisc &= ~1;
            tlbmisc |= 2;
            badaddr = pc;
            pc = exception_handler;
            return;
        }
        trace_tlb = 0;
    } else
        mpc = pc;

    isn = mem_read_i(mpc);
    op = isn & 0x3f;
    
    if (show) {
        show_state(0);
    }

    switch (op) {
    case inst_call:	/* J type */
        imm26 = (isn >> 6) & 0x03ffffff;
        ra = pc + 4;
        pc = (pc & 0xf0000000) | (imm26 << 2);
        break;

    case inst_jmpi:
        imm26 = (isn >> 6) & 0x03ffffff;
        pc = (pc & 0xf0000000) | (imm26 << 2);
        break;

    case inst_ldbuio:
    case inst_ldbu:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        if (op == inst_ldbu)
            TLBMAP(addr, D_R);
        reg[b] = mem_read_d8(addr);
        pc += 4;
        break;

    case inst_addi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] + imm16;
        pc += 4;
        break;

    case inst_stbio:
    case inst_stb:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        if (op == inst_stb)
            TLBMAP(addr, D_W);
        mem_write_d8(addr, reg[b]);
        pc += 4;
        break;

    case inst_br:
        imm16 = (isn >> 6) & 0xffff;
        pc = (pc + 4) + imm16;
        break;
        
    case inst_ldbio:
    case inst_ldb:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        if (op == inst_ldb)
            TLBMAP(addr, D_R);
        reg[b] = (s8)mem_read_d8(addr);
        pc += 4;
        break;

    case inst_cmpgei:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] >= imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_ldhuio:
    case inst_ldhu:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        if (op == inst_ldhu)
            TLBMAP(addr, D_R);
        reg[b] = mem_read_d16(addr);
        pc += 4;
        break;

    case inst_andi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] & (u16)imm16;
        pc += 4;
        break;

    case inst_sthio:
    case inst_sth:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        if (op == inst_sth)
            TLBMAP(addr, D_W);
        mem_write_d16(addr, reg[b]);
        pc += 4;
        break;

    case inst_bge:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] >= (s32)reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_ldhio:
    case inst_ldh:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        if (op == inst_ldh)
            TLBMAP(addr, D_R);
        reg[b] = (s16)mem_read_d16(addr);
        pc += 4;
        break;

    case inst_cmplti:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] < imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_ori:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] | (u16)imm16;
        pc += 4;
        break;

    case inst_stwio:
    case inst_stw:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        if (op == inst_stw)
            TLBMAP(addr, D_W);
        mem_write_d32(addr, reg[b]);
        pc += 4;
        break;

    case inst_blt:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] < (s32)reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_ldwio:
    case inst_ldw:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
        if (op == inst_ldw)
            TLBMAP(addr, D_R);
        reg[b] = mem_read_d32(addr);
        pc += 4;
        break;

    case inst_cmpnei:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] != imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_xori	:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] ^ (u16)imm16;
        pc += 4;
        break;

    case inst_bne:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] != reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_cmpeqi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((s32)reg[a] == imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_muli:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] * imm16;
        pc += 4;
        break;

    case inst_beq:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] == reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_cmpgeui:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] >= (u16)imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

    case inst_andhi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] & (imm16 << 16);
        pc += 4;
        break;


    case inst_bgeu:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if ((u32)reg[a] >= (u32)reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_cmpltui:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] < (u16)imm16) {
            reg[b] = 1;
        } else
            reg[b] = 0;
        pc += 4;
        break;

//    case inst_custom:
    case inst_initd:
        printf("INITD!\n");
        pc += 4;
        break;

    case inst_flushda:
        printf("FLUSHDA!\n");
        pc += 4;
        break;

    case inst_flushd:
        a = (isn >> 27) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        addr = reg[a] + imm16;
//        printf("FLUSHD! %08x\n", addr);
        pc += 4;
#if 0
        {
            int w, l;
            u32 vpn = addr >> 12;
            int line = vpn & 7;

            for (w = 0; w < TLB_NUM_WAYS; w++) {
                tlb[line][w].vpn = 0;
                tlb[line][w].pid = 0;
                tlb[line][w].g = 0;
                tlb[line][w].tlb = 0;
            }

            if (0) printf("flushd line %d, vpn %x\n", line, vpn);
#if 0
            for (l = 0; l < TLB_NUM_LINES; l++) {
                for (w = 0; w < TLB_NUM_WAYS; w++) {
                    if (tlb[l][w].vpn)
                    if (tlb[l][w].vpn == vpn) {
                        printf("FLUSHD! %08x\n", addr);
                        printf("tlb clear tlb[%d][%d].vpn %08x tlb %08x\n",
                               w, l, tlb[l][w].vpn, tlb[l][w].tlb);
                        tlb[l][w].vpn = 0;
                        tlb[l][w].pid = 0;
                        tlb[l][w].g = 0;
                        tlb[l][w].tlb = 0;
                    }
                }
            }
#endif
        }
#endif
        break;

    case inst_orhi:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] | (imm16 << 16);
        pc += 4;
        break;

    case inst_bltu:
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        if (reg[a] < reg[b]) {
            pc = (pc + 4) + imm16;
        } else
            pc += 4;
        break;

    case inst_R_type: /* R-type */
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        c = (isn >> 17) & 0x1f;
        opx = (isn >> 6) & 0x7ff;
        opx2 = (isn >> 11) & 0x3f;
        imm5 = (isn >> 6) & 0x1f;
        switch (opx2) {
        case 0x01: /* eret */
            if (trace_int || trace_eret)
                printf("eret; ea %08x, status %08x -> %08x\n",
                       ea, status, estatus);
            status = estatus;
            pc = ea;
            break;
        case 0x02: /* roli */
            reg[c] = rotate_left(reg[a], imm5);
            pc += 4;
            break;
        case 0x03: /* rol */
            reg[c] = rotate_left(reg[a], reg[b] & 0x1f);
            pc += 4;
            break;
        case 0x04: /* flushp */
            pc += 4;
            break;
        case 0x05: /* ret */
            pc = ra;
            break;
        case 0x06: /* nor */
            reg[c] = ~(reg[a] | reg[b]);
            pc += 4;
            break;
        case 0x07: /* mulxuu */
            llv = (long long)(u32)reg[a] * reg[b];
            reg[c] = llv >> 32;
            pc += 4;
            break;
        case 0x08: /* cmpge */
            if ((s32)reg[a] >= (s32)reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x09: /* bret */
            status = bstatus;
            pc = ba;
            break;
        case 0x0b: /* ror */
            reg[c] = rotate_right(reg[a], reg[b] & 0x1f);
            pc += 4;
            break;
        case 0x0c: /* flushi */
            pc += 4;
            break;
        case 0x0d: /* jmp */
            pc = reg[a];
            break;
        case 0x0e: /* and */
            reg[c] = reg[a] & reg[b];
            pc += 4;
            break;
        case 0x10: /* cmplt */
            if ((s32)reg[a] < (s32)reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x12: /* slli */
            reg[c] = reg[a] << imm5;
            pc += 4;
            break;
        case 0x13: /* sll */
            reg[c] = reg[a] << (reg[b] & 0x1f);
            pc += 4;
            break;
        case 0x16: /* or */
            reg[c] = reg[a] | reg[b];
            pc += 4;
            break;
        case 0x17: /* mulxsu */
            llv = (long long)(s32)reg[a] * reg[b];
            reg[c] = llv >> 32;
            pc += 4;
            break;
        case 0x18: /* cmpne */
            if (reg[a] != reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x1a: /* srli */
            reg[c] = reg[a] >> imm5;
            pc += 4;
            break;
        case 0x1b: /* srl */
            reg[c] = reg[a] >> (reg[b] & 0x1f);
            pc += 4;
            break;
        case 0x1c: /* nextpc */
            reg[c] = pc + 4;
            pc += 4;
            break;
        case 0x1d: /* callr */
            ra = pc + 4;
            pc = reg[a];
            break;
        case 0x1e: /* xor */
            reg[c] = reg[a] ^ reg[b];
            pc += 4;
            break;
        case 0x1f: /* mulxss */
            llv = (long long)(s32)reg[a] * (s32)reg[b];
            reg[c] = llv >> 32;
            pc += 4;
            break;
        case 0x20: /* cmpeq */
            if (reg[a] == reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x24: /* divu */
            reg[c] = reg[a] / reg[b];
            pc += 4;
            break;
        case 0x25: /* div */
            reg[c] = (s32)reg[a] / (s32)reg[b];
            pc += 4;
            break;
        case 0x26: /* rdctl */
            reg[c] = ctl_reg[imm5];
            pc += 4;
            break;
        case 0x27: /* mul */
            reg[c] = reg[a] * reg[b];
            pc += 4;
            break;
        case 0x28: /* cmpgeu */
            if (reg[a] >= reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x29: /* initi */
            pc += 4;
            break;
        case 0x2d: /* trap */
            if (trace_trap) printf("trap; pc=%08x\n", pc);
            estatus = status;
            status &= ~(U | PIE);
            ea = pc + 4;
            exception = 3 << 2;
            pc = exception_handler;
            break;
        case 0x2e: /* wrctl */
            ctl_reg[imm5] = reg[a];
            if (trace_int && imm5) printf("wrctl[%d] <- %08x\n", imm5, reg[a]);
            
            switch (imm5) {
            case 0:
            case 1:
            case 2:
                ctl_reg[imm5] &= 0xf;
                break;

            case 3:
            case 4:
                ipending = irq_latch;
                if (trace_int) printf("ienable %x\n", ienable);
                break;

            case 8:
                if (trace_tlb_reg)
                    printf("pc=%08x ra=%08x pteaddr <- %x (%08x)\n", 
                           pc, ra, reg[a], reg[a] << 10);
                break;

            case 9:
                if (trace_tlb || trace_tlb_reg)
                    printf("pc=%08x tlbacc <- %08x\n", pc, reg[a]);
#if 1
//                if (mmu_miss_active && !(status&1) && ~(tlbmisc & WE))
//                    printf("** pc=%08x status=%08x estatus=%08x tlbacc<-%08x\n",
//                           pc, status, estatus, reg[a]);
                if (!mmu_miss_active && (status&1) && ~(tlbmisc & WE)) {
                    if (0) printf("## pc=%08x status=%08x estatus=%08x tlbacc<-%08x\n",
                           pc, status, estatus, reg[a]);
                    /* hmmm */
                    mmu_miss_active = 1;
                }
#endif
                if ((tlbmisc & WE) || mmu_miss_active) {
                    tlbwrite(reg[a]);
                    mmu_miss_active = 0;
                }
                break;

            case 10:
                /* tlbmisc.RD */
                if (reg[a] & (1<<19)) {
                    tlbread();
                }
                /* tlbmisc.WR */
                if (reg[a] & (1<<18)) {
                }
                break;

            case 13:
                pe = reg[a] & 1;
                ani = reg[a] & 2;
                if (pe) printf("MPU ON!\n");
                if (ani) printf("ANI ON!\n");
                break;
            }
            pc += 4;
            break;
        case 0x30: /* cmpltu */
            if (reg[a] < reg[b])
                reg[c] = 1;
            else
                reg[c] = 0;
            pc += 4;
            break;
        case 0x31: /* add */
            llv = reg[a] + reg[b];
            carry = llv >> 32 ? 1 : 0;
            reg[c] = llv;
            pc += 4;
            break;
        case 0x34: /* break */
            bstatus = status;
            status &= ~(U | PIE);
            ba = pc + 4;
            pc = break_handler;
            break;
        case 0x36: /* sync */
            pc += 4;
            break;
        case 0x39: /* sub */
            reg[c] = reg[a] - reg[b];
            pc += 4;
            break;
        case 0x3a: /* srai */
            reg[c] = (s32)reg[a] >> imm5;
            pc += 4;
            break;
        case 0x3b: /* sra */
            reg[c] = (s32)reg[a] >> (reg[b] & 0x1f);
            pc += 4;
            break;
        default:
            printf("unknown r-type opcode %x isn %08x @ %08x\n",
                   opx2, isn, pc);
            exit(1);
        }
        break;

    case inst_xorhi :
        a = (isn >> 27) & 0x1f;
        b = (isn >> 22) & 0x1f;
        imm16 = (isn >> 6) & 0xffff;
        reg[b] = reg[a] ^ (imm16 << 16);
        pc += 4;
        break;

    default:
        printf("unknown opcode 0x%x isn %08x @ %08x\n", op, isn, pc);
        exit(1);
    }
}

static inline void
inst_poll(void)
{
    jtag_uart_poll();
}

static u32 initial_pc;
static int initial_pc_set;

static u32 initial_reg[32];
static int initial_reg_set[32];

void
inst_set_pc(u32 pc)
{
    initial_pc = pc;
    initial_pc_set = 1;
}

void
inst_set_regs(int first, int last, u32 *regs)
{
    int i;

    for (i = first; i <= last; i++) {
        initial_reg_set[i] = 1;
        initial_reg[i] = regs[i-first];
    }
}

void
inst_run(int max_cycles, int show)
{
    int pollcount, i;
    int max_poll = 2000000;

    if (initial_pc_set)
        pc = initial_pc;

    for (i = 0; i < 32; i++) {
        if (initial_reg_set[i]) {
            reg[i] = initial_reg[i];
            printf("initial r%d = %08x\n", i, reg[i]);
        }
    }

    cycles = 0;
    pollcount = 0;
    while (1) {
        inst_loop(show || local_show || global_show);

        cycles++;

        if (max_cycles && cycles > max_cycles)
            break;

        if (++pollcount > max_poll) {
            pollcount = 0;
            inst_poll();
        }
    }

    if (show) {
        dump_mem();
    }
}

void
inst_final_dump(void)
{
    printf("\n");
    printf("cycles %lu\n", cycles);
    printf("pc %08x\n", pc);
    show_regs();
}

void
inst_init(void)
{
    int i;

    pc = 0x02800000;
    exception_handler = 0xd0000020;
    fast_tlb_handler = 0xc7fff400;
    break_handler = 0xc7fff820;
    status = 0;

    for (i = 0; i < 32; i++)
        reg[i] = 0;

#if 0
    {
        int i;
        u32 addr;

        addr = 0x009db940;
        for (i = 0; i < 16; i++) {
            printf("%08x %08x\n", addr, mem_read_d32(addr));
            addr += 4;
        }
    }
#endif
}

/*
8    ~ISP1362_avalon_slave_1_irq_n_from_sa,
7    ~ISP1362_avalon_slave_0_irq_n_from_sa,
6    DM9000A_avalon_slave_0_irq_from_sa,
5    button_pio_s1_irq_from_sa,
4    timer_1_s1_irq_from_sa,
3    timer_0_s1_irq_from_sa,
2    uart_0_s1_irq_from_sa,
1    jtag_uart_0_avalon_jtag_slave_irq_from_sa,
0    epcs_controller_epcs_control_port_irq_from_sa
*/

void set_local_show(void) { local_show = 1; }

void
interrupt_generate(int irq)
{
    if (trace_int && (irq_latch & (1 << irq)) == 0)
        printf("interrupt_generate %d\n", irq);

#if 1
    {
        extern int trace_mem;
        if (trace_int) {
                global_show = 1;
                trace_mem = 1;
        }
    }
#endif

    irq_latch |= (1 << irq);
    ipending = irq_latch;
}

void
interrupt_ack(int irq)
{
    //if (trace_int) printf("interrupt_ack %d\n", irq);

    irq_latch &= ~(1 << irq);
    ipending = irq_latch;

    if (trace_int) printf("interrupt_ack %d; ipending %x\n", irq, ipending);
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
