/*
 * jtag.c
 */

#include <stdio.h>
#include <memory.h>
#include <termios.h>
#include <sys/poll.h>

#include "sim.h"

static struct termios newterm, oldterm;
int jtag_uart_fd;
int jtag_uart_fd_valid;

u16 jtag_uart_ctl_bits;

int jtag_uart_rx_fifo[256];
u8 jtag_uart_rx_fifo_rptr;
u8 jtag_uart_rx_fifo_wptr;
int jtag_uart_rx_fifo_depth;

int jtag_uart_int;

void
jtag_int(int which)
{
    switch (which) {
    case 1:
        jtag_uart_ctl_bits |= 0x100;
        if (jtag_uart_ctl_bits & 0x1)
            interrupt_generate(jtag_uart_int);
        break;
    case 2:
        jtag_uart_ctl_bits |= 0x200;
        if (jtag_uart_ctl_bits & 0x2)
            interrupt_generate(jtag_uart_int);
        break;
    }
}

void
jtag_unint(int which)
{
    switch (which) {
    case 1:
        jtag_uart_ctl_bits &= ~0x100;
        interrupt_ack(jtag_uart_int);
        break;
    case 2:
        jtag_uart_ctl_bits &= ~0x200;
        interrupt_ack(jtag_uart_int);
        break;
    }
}

int
rx_fifo_depth(void)
{
    return jtag_uart_rx_fifo_depth;
}

int
rx_fifo_get(void)
{
    u32 v = 0;
    if (jtag_uart_rx_fifo_depth) {
        jtag_uart_rx_fifo_depth--;
        v = jtag_uart_rx_fifo[jtag_uart_rx_fifo_rptr++];
        if (0) printf("pull %02x %c\n", v, v);
    }
    return v;
}

void
jtag_uart_rx_put(u32 v)
{
    jtag_uart_rx_fifo[jtag_uart_rx_fifo_wptr++] = v;
    jtag_uart_rx_fifo_depth++;
    jtag_int(1);

    if (0) {
        printf("rx fifo depth %d, wptr %d, rptr %d (%02x %c)\n",
               jtag_uart_rx_fifo_depth,
               jtag_uart_rx_fifo_wptr, jtag_uart_rx_fifo_rptr,
               v, v);
        fflush(stdout);
    }
}

int
tx_fifo_put(u8 v)
{
#if 0
    printf("%c", v);
    fflush(stdout);
#else
    {
        char c = v;
        write(jtag_uart_fd, &c, 1);
    }
#endif
    jtag_int(2);
}

int
tx_fifo_space(void)
{
    return 64;
}

void
jtag_uart_poll(void)
{
    int c;
#if 0
    if ((c = getchar()) != -1) {
        jtag_uart_rx_put(c);
    }
#else
    int ret;
    struct pollfd fd[1];
    fd[0].fd = jtag_uart_fd;
    fd[0].events = POLLIN;
    fd[0].revents = 0;
    ret = poll(fd, 1, 0);
    if (ret > 0 && (fd[0].revents & POLLIN)) {
        char ch;
        ret = read(jtag_uart_fd, &ch, 1);
        jtag_uart_rx_put(ch);
#if 0
        {
            extern int global_show, trace_mem, trace_int;
            if (0) {
                global_show = 1;
                trace_mem = 1;
            }
            if (1) {
                trace_int = 1;
            }
        }
#endif

    }
#endif
}

u32
jtag_uart_read(u32 addr)
{
    u32 v, data, offset;
    int count;
    
#define RVALID 0x00008000

jtag_uart_poll();

    offset = addr & 0xf;
    switch (offset) {
    case 0:
        v = 0;
        jtag_uart_poll();
        count = rx_fifo_depth();
        if (count) {
            data = rx_fifo_get();
            count = rx_fifo_depth();
            v = (count << 16) | RVALID | (data & 0xff);

            if (count == 0) {
                jtag_unint(1);
            }
        }
        return v;
    case 4:
        return (tx_fifo_space() << 16) | jtag_uart_ctl_bits;
    }
}

u32
jtag_uart_write(u32 addr, u32 v)
{
    u32 offset = addr & 0xf;
    switch (offset) {
    case 0:
        tx_fifo_put(v & 0xff);
        return 0;
    case 4:
        //printf("tx: ctl %08x\n", v);
        jtag_uart_ctl_bits = (jtag_uart_ctl_bits & 0xfffc) | (v & 3);
        if (v & 0x400)
            jtag_uart_ctl_bits &= ~0x400;

        if ((jtag_uart_ctl_bits & 0x1) && jtag_uart_rx_fifo_depth) {
            jtag_int(1);
        }
        if ((jtag_uart_ctl_bits & 0x2)) {
            jtag_int(2);
            //printf("tx: enable\n");
        }

        if ((jtag_uart_ctl_bits & 0x2) == 0) {
            //printf("tx: disable\n");
	    jtag_unint(2);
        }

        return 0;
    }

    return 0;
}

void
jtag_uart_init(void)
{
    jtag_uart_fd = fileno(stdin);
    tcgetattr(jtag_uart_fd, &oldterm);

    memcpy(&newterm, &oldterm, sizeof(newterm));
    newterm.c_lflag = newterm.c_lflag & (!ICANON);
newterm.c_lflag |= ISIG;
    newterm.c_cc[VMIN] = 0;
    newterm.c_cc[VTIME] = 1;
    tcsetattr(jtag_uart_fd, TCSANOW, &newterm);

    jtag_uart_fd_valid = 1;

    jtag_uart_int = 1;
}

void
jtag_uart_deinit(void)
{
    if (jtag_uart_fd_valid) {
        tcsetattr(jtag_uart_fd, TCSANOW, &oldterm);
    }
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
