/*
 * mem.c
 */

#define M256		(256*1024*1024)
#define M128		(128*1024*1024)
#define M64		(64*1024*1024)
#define M32		(32*1024*1024)
#define M8		(8*1024*1024)
#define K1		(1024)

#define FLASH_BASE	0x00000000
#define SDRAM_BASE	0x10000000
#define TLB_BASE	0x07fff400
#define UART_BASE	0x08004c80
#define SYSTEM_ID_BASE	0x08004d40
#define JTAG_UART_BASE	0x08004d50
#define TIMER_BASE	0x08400000
		
#include <stdio.h>
#include <fcntl.h>
#include <memory.h>

#include "sim.h"
#include "ops.h"

u32 flash[16*1024*1024];
u32 ddr2[128*1024*1024];
u32 tlb_tci[1024];

extern u32 pc;
extern int trace_mem;
extern int trace_int;

void
byteswap_flash(void)
{
    int i;

    for (i = 0; i < 16*1024*1024; i++) {
        flash[i] = ntohl(flash[i]);
    }
}

/* bulk byte based write to memory, used by elf read code */
int
mem_move_i(u32 addr, u8 *bytes, int size)
{
    char *d;
    int offset;

    printf("mem_move_i(addr=%x, bytes=%x, size=%d)\n",
           addr, (unsigned int)*bytes, size);

    offset = (addr - SDRAM_BASE) & (M128-1);
    memcpy((char *)((u8 *)ddr2) + offset, bytes, size);
}

/* bulk byte based write to data, used by elf read code */
int
mem_move_d(u32 addr, u8 *bytes, int size)
{
    char *d;
    int offset;

    printf("mem_move_d(addr=%x, bytes=%x, size=%d)\n",
           addr, (unsigned int)*bytes, size); fflush(stdout);

    offset = (addr - SDRAM_BASE) & (M128-1);
    memcpy( (char *)((u8 *)ddr2) + offset, bytes, size );
}

/* cpu read from instruction space */
u32
mem_read_i(u32 pc)
{
    int offset;

    if (0) printf("mem_read_i: %08x\n", pc);

    if (pc >= 0xc0000000)
        pc &= 0x1fffffff;

    if (pc >= TLB_BASE && pc < TLB_BASE+K1) {
        offset = pc - TLB_BASE;
        return tlb_tci[offset/4];
    }

    if (pc >= FLASH_BASE && pc < FLASH_BASE+M64) {
        offset = pc - FLASH_BASE;
        return flash[offset/4];
    }

    if (pc >= SDRAM_BASE && pc < SDRAM_BASE+M256) {
        offset = (pc - SDRAM_BASE) & (M128-1);
        return ddr2[offset/4];
    }

    printf("** illegal i_mem read %08x\n", pc);
sigint_handler(1);

    return 0;
}

/* cpu read from data space */
u32
mem_read_d32(u32 addr)
{
    int offset;
    u32 v;

    if (addr >= 0xc0000000)
        addr &= 0x1fffffff;

    if (addr >= FLASH_BASE && addr < FLASH_BASE+M64) {

        if (flash_active())
            return flash_read(addr, 4);

        offset = addr - FLASH_BASE;
        v = flash[offset/4];
        if (trace_mem) printf("%08x <- flash[%x]\n", v, addr);
        return v;
    }

    if (addr >= SDRAM_BASE && addr < SDRAM_BASE+M256) {
        /* ddr2 */
        offset = (addr - SDRAM_BASE) & (M128-1);
        v = ddr2[offset/4];

        if (0) {
            /* induce error */
            if (addr/4 == 5) v = 0;
        }

        if (trace_mem) printf("%08x <- ddr2[%x]\n", v, addr);
        return v;
    }

    if (addr >= TLB_BASE && addr < TLB_BASE+K1) {
        /* 1k tlb tightly coupled */
        offset = addr - TLB_BASE;
        v = tlb_tci[offset/4];
        if (trace_mem) printf("%08x <- tlb_tci[%x]\n", v, addr);
        return v;
    }

    switch (addr) {
    case UART_BASE+8:
    case UART_BASE+12:
        printf("uart read %08x\n", addr);
        return 0;

        /* jtag uart */
    case JTAG_UART_BASE:
    case JTAG_UART_BASE+4:
        return jtag_uart_read(addr);

    case SYSTEM_ID_BASE:
        return 1174346794;
    case SYSTEM_ID_BASE+4:
        return 1233287581;

//    case 0x806810f8:
//    case 0x806810fc:
//        return enet_read(addr);

    case 0x0800403c:
    case 0x08004068:
    case 0x0800406c:
    case 0x08004078:
    case 0x0800407c:
    case 0x08004088:
    case 0x0800408c:
    case 0x08004094:
    case 0x08004208:
    case 0x0800420c:
        return 0;

    case 0x08004040:
    case 0x08004288:
    case 0x0800428c:
        return 0;

        /* timer */
    case TIMER_BASE+0x0: /* 0 - status */
    case TIMER_BASE+0x4: /* 1 - control */
    case TIMER_BASE+0x8: /* 2 - l*/
    case TIMER_BASE+0xc: /* 3 - h */
    case TIMER_BASE+0x10:
    case TIMER_BASE+0x14:
    case TIMER_BASE+0x18:
    case TIMER_BASE+0x1c:
    case TIMER_BASE+0x20:
    case TIMER_BASE+0x24:
    case TIMER_BASE+0x28:
    case TIMER_BASE+0x2c:
    case TIMER_BASE+0x30:
    case TIMER_BASE+0x34:
    case TIMER_BASE+0x38:
    case TIMER_BASE+0x3c:
        return timer_read32((addr & 0x1f) / 4);
        break;
    }

    printf("** illegal d_mem read %08x @ pc %08x\n", addr, pc);

    return 0;
}

u32
mem_read_d16(u32 addr)
{
    u32 v;

    if (addr >= 0xc0000000)
        addr &= 0x1fffffff;

    if (addr >= FLASH_BASE && addr < FLASH_BASE+M64) {
        int offset = addr - FLASH_BASE;

        if (flash_active())
            return flash_read(addr, 2);

        if (trace_mem) printf("%08x <- flash[%x]\n", v, addr);
        v = flash[offset/4];
        if (addr & 2)
            return (v >> 16) & 0xffff;
        return v & 0xffff;
    }

    if (addr >= SDRAM_BASE && addr < SDRAM_BASE+M256) {
        int offset = (addr - SDRAM_BASE) & (M128-1);
        return  ((u16 *)ddr2)[offset/2];
    }

    v = mem_read_d32(addr & 0xfffffffc);
    if (addr & 2)
        return (v >> 16) & 0xffff;
    return v & 0xffff;
}

u32
mem_read_d8(u32 addr)
{
    u32 v;

    if (addr >= 0xc0000000)
        addr &= 0x1fffffff;

    if (addr >= FLASH_BASE && addr < FLASH_BASE+M64) {
        if (flash_active())
            return flash_read(addr, 1);
    }

    if (addr >= SDRAM_BASE && addr < SDRAM_BASE+M256) {
        int offset = (addr - SDRAM_BASE) & (M128-1);
        return  ((u8 *)ddr2)[offset];
    }

    v = mem_read_d32(addr & 0xfffffffc);
    switch (addr & 3) {
    case 3: return (v >> 24) & 0xff;
    case 2: return (v >> 16) & 0xff;
    case 1: return (v >>  8) & 0xff;
    case 0: return (v >>  0) & 0xff;
    }
}

/* cpu write to data space */
int
mem_write(u32 addr, u32 v, int size)
{
    int offset;

    if (addr >= 0xc0000000)
        addr &= 0x1fffffff;

    if (addr >= FLASH_BASE && addr < FLASH_BASE+M64) {
        /* flash emulation */
        flash_write(addr, v, size);
        return 0;
    }

    if (addr >= SDRAM_BASE && addr < SDRAM_BASE+M256) {
        /* ddr2 */
        offset = (addr - SDRAM_BASE) & (M128-1);
        if (trace_mem) printf("ddr2[%x] <- %x\n", addr, v);
//if (addr == 0x10279678) printf("ddr2[%x] <- %x\n", addr, v);
        switch (size) {
        case 1: ((u8 *)ddr2)[offset] = v; break;
        case 2: ((u16 *)ddr2)[offset/2] = v; break;
        case 4: ddr2[offset/4] = v; break;
        }
        return 0;
    }

    if (addr >= TLB_BASE && addr < TLB_BASE+K1) {
        /* 1k tlb tightly coupled */
        offset = addr - TLB_BASE;
        tlb_tci[offset/4] = v;
        if (trace_mem) printf("tlb_tci[%x] <- %08x\n", addr, v);
        return v;
    }

    switch (addr) {
    case UART_BASE+8:
    case UART_BASE+12:
        printf("uart write %08x <- %08x\n", addr, v);
        return 0;

    case JTAG_UART_BASE:
    case JTAG_UART_BASE+4:
        return jtag_uart_write(addr, v);

//    case 0x806810f8:
//    case 0x806810fc:
//        return enet_write(addr, v);

    case 0x0800403c:
    case 0x08004208:
    case 0x0800420c:
        return 0;

    case 0x08004cc4:
        return 0;

    case 0x08004040:
    case 0x08004288:
    case 0x0800428c:
        return 0;

    case 0x08004cc0:
        return 0;

    case TIMER_BASE+0x0: /* 0 - status */
    case TIMER_BASE+0x4: /* 1 - control */
    case TIMER_BASE+0x8: /* 2 - l*/
    case TIMER_BASE+0xc: /* 3 - h */
    case TIMER_BASE+0x10:
    case TIMER_BASE+0x14:
    case TIMER_BASE+0x18:
    case TIMER_BASE+0x1c:
    case TIMER_BASE+0x20:
    case TIMER_BASE+0x24:
    case TIMER_BASE+0x28:
    case TIMER_BASE+0x2c:
    case TIMER_BASE+0x30:
    case TIMER_BASE+0x34:
    case TIMER_BASE+0x38:
    case TIMER_BASE+0x3c:
        return timer_write32((addr & 0x1f) / 4, v);
    }

    printf("** illegal d_mem write %08x (size%d) <- %08x\n", addr, size, v);
    return -1;
}

int
mem_write_d32(u32 addr, u32 v)
{
    return mem_write(addr, v, 4);
}

int
mem_write_d16(u32 addr, u32 v)
{
    return mem_write(addr, v, 2);
}

int
mem_write_d8(u32 addr, u32 v)
{
    return mem_write(addr, v, 1);
}

void
dump_mem(void)
{
    int i, j;

    printf("ddr2:\n");
    for (i = 0;  i < 64; i++) {
        printf("%04x: ", i*8*4);
        for (j = 0; j < 8; j++)
            printf("%08x ", ddr2[i*8+j]);
        printf("\n");
    }

//    printf("sram:\n");
//    for (i = 0;  i < 64; i++) {
//        printf("%04x: ", i*8*4);
//        for (j = 0; j < 8; j++)
//            printf("%08x ", d_mem[i*8+j]);
//        printf("\n");
//    }
}

void
mem_init(void)
{
    int fd, ret;

    fd = open("flash.img", O_RDONLY);
    if (fd >= 0) {
        ret = read(fd, flash, sizeof(flash));
        if (ret < 0)
            perror("flash.img");
        else {
            printf("flash: %d bytes\n", ret);
        }
        close(fd);
    }
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
