/*
 * given an elf file and a cpu trace file (from the nios simulation),
 * correlate the cpu pc's with the elf and print a list of
 * code labels which were executed, along with some register information
 *
 * this is a total hack but does the job...
 *
 * Brad Parker <brad@heeltoe.com>
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define MAX_SYMS	50000
#define MAX_SAMPLES	5000000

typedef unsigned int u32;

struct {
    u32 v;
    int t;
    char *sym;
} syms[MAX_SYMS];
int sym_count;

void
get_syms(char *elfname)
{
    FILE *f;
    u32 addr;
    char t;
    char cmd[256];
    char sym[256];
    char line[256];

    sprintf(cmd, "nm -n %s", elfname);
    f = popen(cmd, "r");
    if (f == NULL) {
        perror(elfname);
        exit(1);
    }

    while (fgets(line, sizeof(line), f)) {
        sscanf(line, "%x %c %s", &addr, &t, sym);

        if (sym_count >= MAX_SYMS) {
            fprintf(stderr, "maximum symbols (%d) exceeded\n", MAX_SYMS);
            exit(1);
        }

        syms[sym_count].v = addr;
        syms[sym_count].t = t;
        syms[sym_count].sym = strdup(sym);
#if 0
        if (addr < 0xd0000100)
            printf("%08x %s\n", syms[sym_count].v, syms[sym_count].sym);
#endif
        sym_count++;
    }
}

struct {
    int ts;
    int rst;
    u32 pcb;
    u32 iw;
    int wr_dst_reg;
    int dst_regnum;
    int wr_data_filtered;
    int full_mem_baddr;
    int st_data;
    int mem_byte_en;
    int cmp_result;

    int sym;
    int offset;
} samples[MAX_SAMPLES];
int t_count;

/* get trace from altera cpu.tr file */
void
get_trace(char *tr_filename)
{
    FILE *f;
    char sym[256];
    char line[256];

    int ts, rst, pcb, iw, fill;
    int wr_dst_reg, dst_regnum, wr_data_filtered, full_mem_baddr;
    int st_data, mem_byte_en, cmp_result;

    f = fopen(tr_filename, "r");
    if (f == NULL) {
        perror(tr_filename);
        exit(1);
    }

    while (fgets(line, sizeof(line), f)) {


        /*
          $time, ~reset_n, W_pcb,
          0, W_valid_intr, W_valid_hbreak, W_iw, ~W_iw_invalid, 
          W_wr_dst_reg, W_dst_regnum, W_wr_data_filtered, W_mem_baddr, W_st_data, W_mem_byte_en, W_cmp_result,
          W_target_pcb, W_status_reg, W_estatus_reg, W_bstatus_reg, W_ienable_reg, W_ipending_reg, W_exception_reg, W_pteaddr_reg_tb, W_tlbacc_reg_tb, W_tlbmisc_reg_tb, W_badaddr_reg, 0, 0, 0, W_pcb_phy, W_mem_baddr_phy, W_valid_exception_no_intr
        */
        sscanf(line, "%d ns: %x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x",
               &ts, &rst, &pcb,
               &fill, &fill, &fill, &iw, &fill, 
               &wr_dst_reg, &dst_regnum, &wr_data_filtered, &full_mem_baddr,
               &st_data, &mem_byte_en, &cmp_result,
               &fill, &fill, &fill, &fill, &fill, &fill, &fill);

        if (t_count > MAX_SAMPLES) {
            fprintf(stderr, "maximum samples (%d) exceeded\n", MAX_SAMPLES);
            exit(1);
        }

        samples[t_count].ts = ts;
        samples[t_count].rst = rst;
        samples[t_count].pcb = pcb;
        samples[t_count].iw = iw;
        samples[t_count].wr_dst_reg = wr_dst_reg;
        samples[t_count].dst_regnum = dst_regnum;
        samples[t_count].wr_data_filtered = wr_data_filtered;
        samples[t_count].full_mem_baddr = full_mem_baddr;
        samples[t_count].st_data = st_data;
        samples[t_count].mem_byte_en = mem_byte_en;
        samples[t_count].cmp_result = cmp_result;
        t_count++;
    }
}

/* get trace from our trace file */
void
get_alt_trace(char *tr_filename)
{
    FILE *f;
    char sym[256];
    char line[256];
    char inst[256];

    int ts, cycle, rst, pcb, iw, fill;
    int wr_dst_reg, dst_regnum, wr_data_filtered, full_mem_baddr;
    int st_data, mem_byte_en, cmp_result;

    f = fopen(tr_filename, "r");
    if (f == NULL) {
        perror(tr_filename);
        exit(1);
    }

    while (fgets(line, sizeof(line), f)) {

        rst = 0;
        fill = 0;
        wr_dst_reg = 0;
        dst_regnum = 0;
        wr_data_filtered = 0;
        full_mem_baddr = 0;
        st_data = 0;
        mem_byte_en = 0;
        cmp_result = 0;

        sscanf(line, "%d ns: %d r0 pc %x iw %x W_inst %s",
               &ts, &cycle, &pcb, &iw, inst);

#if 0
        if (t_count < 10) {
            printf("line: %s", line);
            printf("->  %d ns: %d r0 pc %x iw %x W_inst %s\n",
                   ts, cycle, pcb, iw, inst);
        }
#endif

        if (t_count > MAX_SAMPLES) {
            fprintf(stderr, "maximum samples (%d) exceeded\n", MAX_SAMPLES);
            exit(1);
        }

        samples[t_count].ts = ts;
        samples[t_count].rst = rst;
        samples[t_count].pcb = pcb;
        samples[t_count].iw = iw;
        samples[t_count].wr_dst_reg = wr_dst_reg;
        samples[t_count].dst_regnum = dst_regnum;
        samples[t_count].wr_data_filtered = wr_data_filtered;
        samples[t_count].full_mem_baddr = full_mem_baddr;
        samples[t_count].st_data = st_data;
        samples[t_count].mem_byte_en = mem_byte_en;
        samples[t_count].cmp_result = cmp_result;
        t_count++;
    }
}

int
find_pcb(int pcb, int *psym, int *poffset)
{
    int i, off;

    off = 0;
    for (i = 0; i < sym_count; i++) {
        if (pcb == syms[i].v) 
            break;

        if (syms[i].v > pcb) {
            i--;
            off = pcb - syms[i].v;
            break;
        }
    }

    if (i == sym_count)
        return -1;

    *psym = i;
    *poffset = off;
    return 0;
}

void
correlate(void)
{
    int i;
    int sym, offset;

    for (i = 0; i < t_count; i++) {
        int pcb;
        pcb = samples[i].pcb;

        find_pcb(pcb, &sym, &offset);

        //printf("%08x -> %s %08x\n", pcb, syms[sym].sym, offset);

        samples[i].sym = sym;
        samples[i].offset = offset;
    }
}

void
show(void)
{
    int i;
    char *s, sym[256];

    for (i = 0; i < t_count; i++) {

        if (samples[i].rst || samples[i].ts == 0)
            continue;

        sym[0] = 0;
        s = syms[samples[i].sym].sym;

        if (s && samples[i].offset == 0) {
            strcpy(sym, s);
        } else {
            sprintf(sym, "%s+0x%x", s, samples[i].offset);
        }

        printf("%+5dns %08x %-20s\t [%d]<- %08x %08x %08x\n",
               samples[i].ts,
               samples[i].pcb,
               sym,
               samples[i].dst_regnum,
               samples[i].full_mem_baddr,
               samples[i].wr_data_filtered,
               samples[i].st_data);
    }
}

void
usage(void)
{
    fprintf(stderr, "usage:\n");
    fprintf(stderr, "show -e <elf-filename> -t <cpu-trace-filename>\n");
}

int optint;
char *optarg;


main(int argc, char *argv[])
{
    int c;
    char *elffilename, *tracefilename;
    int alttrace;

    elffilename = "executable.elf";
    elffilename = "vmlinux";
    tracefilename = "linux_cpu.tr";
    alttrace = 0;

    while ((c = getopt(argc, argv, "ae:t:d")) != -1) {
        switch (c) {
        case 'a': alttrace = 1; break;
        case 'e': elffilename = strdup(optarg); break;
        case 't': tracefilename = strdup(optarg); break;
        default:
            usage();
        }
    }

    printf("elf:   %s\n", elffilename);
    printf("trace: %s\n", tracefilename);

    get_syms(elffilename);
    if (alttrace)
        get_alt_trace(tracefilename);
    else
        get_trace(tracefilename);

    printf("symbols: %d\n", sym_count);
    printf("traces:  %d\n", t_count);

    correlate();
    show();
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
 */
