/*
 * (C) Copyright 2002, 2003
 * Daniel Engstrm, Omicron Ceti AB, daniel@omicron.se
 * 
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Alex Zuepke <azu@sysgo.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/io.h>
#include <pci.h>
#include <asm/ic/sc520.h>

#define PROBE_BUFFER_SIZE 1024
static unsigned char buffer[PROBE_BUFFER_SIZE];

#define SC520_FLASH_BANK_BASE 0x38f80000  /* BOOTCS */
//#define SC520_FLASH_BANKSIZE  0x08000000

#define AMD29LV040B_SIZE        0x80000
#define AMD29LV040B_SECTORS     16

#define AT49LV040_SIZE		0x80000
#define AT49LV040_SECTORS	16

#define AMD29LV016B_SIZE        0x200000
#define AMD29LV016B_SECTORS     32

#define SC520_MAX_FLASH_BANKS 1
flash_info_t    flash_info[SC520_MAX_FLASH_BANKS];

#define READY 1
#define ERR   2
#define TMO   4

/*-----------------------------------------------------------------------
 */


static u32 _probe_flash(u32 addr)
{
	u32 result=0;
	u8 b1, b2, b3, b4;

	/* First do an unlock cycle */
	*(volatile u8 *)(addr+0x5555) = 0xaa;
	*(volatile u8 *)(addr+0x2aaa) = 0x55;
	*(volatile u8 *)(addr+0x5555) = 0x90;
		
	/* Read vendor */
	b1 = *(volatile u8 *)addr;
	b2 = *(volatile u8 *)(addr+1);
	b3 = *(volatile u8 *)(addr+2);
	b4 = *(volatile u8 *)(addr+3);
		
	/* Return device to data mode */
	*(volatile u8 *)addr = 0xff;
	*(volatile u8 *)(addr+0x5555) = 0xf0;

	result = (b1 << 16) | b2;

	return result;
}

extern int _probe_flash_end;
asm ("_probe_flash_end:\n"
     ".long 0\n");

static int identify_flash(unsigned address)
{
	int is;	
	int device;
	int vendor;	
	int size;
	unsigned res;

/* 01 4f 00 02 */	
	u32 (*_probe_flash_ptr)(u32 a);
	
	size = (unsigned)&_probe_flash_end - (unsigned)_probe_flash; 
	
	if (size > PROBE_BUFFER_SIZE) {
		printf("_probe_flash() routine too large (%d) %p - %p\n",
		       size, &_probe_flash_end, _probe_flash);
		return 0;
	}
	
	memcpy(buffer, _probe_flash, size);
	_probe_flash_ptr = (void*)buffer;
	
	is = disable_interrupts();
	res = _probe_flash_ptr(address);
	if (is) {
		enable_interrupts();
	}
	
        vendor = res >> 16;
	device = res & 0xffff;
		
	return res;
}

ulong flash_init(void)
{
	int i, j;
	ulong size = 0;

	for (i = 0; i < SC520_MAX_FLASH_BANKS; i++) {
		unsigned id;
		ulong flashbase = 0;
		int sectsize = 0; 
		
		memset(flash_info[i].protect, 0, CFG_MAX_FLASH_SECT);
		switch (i) {
		case 0:
			flashbase = SC520_FLASH_BANK_BASE;
			break;
		default:
			panic("configured to many flash banks!\n");
		}
		
		id = identify_flash(flashbase);
		switch (id & 0x00ff00ff) {
		case 0x0001004f:
			/* 29LV040B */
			flash_info[i].flash_id =
				(AMD_MANUFACT & FLASH_VENDMASK) |
				(AMD_ID_LV040B & FLASH_TYPEMASK);
			
			flash_info[i].size = AMD29LV040B_SIZE;
			flash_info[i].sector_count = AMD29LV040B_SECTORS;
			sectsize = (AMD29LV040B_SIZE)/AMD29LV040B_SECTORS;
			printf("Bank %d: 1 x AMD 29LV040B\n", i);
			break;
		case 0x001f0013:
			/* 49LV040 */
			flash_info[i].flash_id =
				(ATM_MANUFACT & FLASH_VENDMASK) |
				(ATM_ID_LV040 & FLASH_TYPEMASK);
			
			flash_info[i].size = AT49LV040_SIZE;
			flash_info[i].sector_count = AT49LV040_SECTORS;
			sectsize = (AT49LV040_SIZE)/AT49LV040_SECTORS;
			printf("Bank %d: 1 x Atmel 49LV040\n", i);
			break;
		case 0x000100c8:
			/* 29LV016B/29LV017B */
			flash_info[i].flash_id =
				(AMD_MANUFACT & FLASH_VENDMASK) |
				(AMD_ID_LV016B & FLASH_TYPEMASK);
			
			flash_info[i].size = AMD29LV016B_SIZE;
			flash_info[i].sector_count = AMD29LV016B_SECTORS;
			sectsize = (AMD29LV016B_SIZE)/AMD29LV016B_SECTORS;
			printf("Bank %d: 1 x AMD 29LV017B\n", i);
			break;
			
		default:
			printf("Bank %d have unknown flash %08x\n", i, id);
			flash_info[i].flash_id = FLASH_UNKNOWN;
			continue;
		}
		
		for (j = 0; j < flash_info[i].sector_count; j++) {
			flash_info[i].start[j] = flashbase + j * sectsize;
		}
		size += flash_info[i].size;
		
		flash_protect(FLAG_PROTECT_CLEAR,
			      flash_info[i].start[0],
			       flash_info[i].start[0] + flash_info[i].size - 1,
			      &flash_info[i]);
	}
	
	/*
	 * Protect monitor and environment sectors
	 */
	flash_protect(FLAG_PROTECT_SET,
		      i386boot_start,
		      i386boot_end,
		      &flash_info[0]);
#ifdef CFG_ENV_ADDR
	flash_protect(FLAG_PROTECT_SET,
		      CFG_ENV_ADDR,
		      CFG_ENV_ADDR + CFG_ENV_SIZE - 1,
		      &flash_info[0]);
#endif	
	return size;
}

/*-----------------------------------------------------------------------
 */
void flash_print_info(flash_info_t *info)
{
	int i;
	
	switch (info->flash_id & FLASH_VENDMASK) {
		
	case (AMD_MANUFACT & FLASH_VENDMASK):
		printf("AMD:   ");
		switch (info->flash_id & FLASH_TYPEMASK) {
		case (AMD_ID_LV040B & FLASH_TYPEMASK):
			printf("1x AMD29LV040B (1x512k)\n");
			break;
		case (ATM_MANUFACT & FLASH_VENDMASK):
			printf("1x Atmel49LV040 (1x512k)\n");
			break;
		case (AMD_ID_LV016B & FLASH_TYPEMASK):
			printf("1x AMD29LV017B (4x16Mbit)\n");
			break;
		default:
			printf("Unknown Chip Type\n");
			goto done;
			break;
		}
		break;
	default:
		printf("Unknown Vendor ");
		break;
	}
	
	printf("  Size: %ld MB in %d Sectors\n",
	       info->size >> 20, info->sector_count);
	
	printf("  Sector Start Addresses:");
	for (i = 0; i < info->sector_count; i++) {
		if ((i % 5) == 0) {
			printf ("\n   ");
		}
		printf (" %08lX%s", info->start[i],
			info->protect[i] ? " (RO)" : "     ");
	}
	printf ("\n");
	
	done:
	;
}

/*-----------------------------------------------------------------------
 */

/* this needs to be inlined, the SWTMRMMILLI register is reset by each read */
#define __udelay(delay) \
{	\
	unsigned micro; \
	unsigned milli=0; \
	\
	micro = *(volatile u16*)(0xfffef000+SC520_SWTMRMILLI); \
         \
	for (;;) { \
		\
		milli += *(volatile u16*)(0xfffef000+SC520_SWTMRMILLI); \
		micro = *(volatile u16*)(0xfffef000+SC520_SWTMRMICRO); \
		\
		if ((delay) <= (micro + (milli * 1000))) { \
			break; \
		} \
	} \
} while (0) 

static u32 _amd_erase_flash(u32 addr, u32 sector)
{
	unsigned elapsed;
	
	/* Issue erase */
	*(volatile u8 *)(addr + 0x5555) = 0xAA;
	*(volatile u8 *)(addr + 0x2aaa) = 0x55;
	*(volatile u8 *)(addr + 0x5555) = 0x80;
	/* And one unlock */
	*(volatile u8 *)(addr + 0x5555) = 0xAA;
	*(volatile u8 *)(addr + 0x2aaa) = 0x55;
	/* Sector erase command comes last */
	*(volatile u8 *)(addr + sector) = 0x30;
	
	/* dummy read */
	elapsed = *(volatile u16*)(0xfffef000+SC520_SWTMRMILLI);
	elapsed = 0;
	__udelay(50);
	while (((*(volatile u8 *)(addr + sector)) & 0x80) != 0x80) {
		
		elapsed += *(volatile u16*)(0xfffef000+SC520_SWTMRMILLI);
		if (elapsed > ((CFG_FLASH_ERASE_TOUT/CFG_HZ) * 1000)) {
			*(volatile u8 *)(addr) = 0xf0;
			return 1;			
		}
	}
	
	*(volatile u8 *)(addr) = 0xf0;
	
	return 0;
}

extern int _amd_erase_flash_end;
asm ("_amd_erase_flash_end:\n"
     ".long 0\n");

int flash_erase(flash_info_t *info, int s_first, int s_last)
{
	u32 (*_erase_flash_ptr)(u32 a, u32 so);
	int prot;
	int sect;
	unsigned size;
	
	if ((s_first < 0) || (s_first > s_last)) {
		if (info->flash_id == FLASH_UNKNOWN) {
			printf("- missing\n");
		} else {
			printf("- no sectors to erase\n");
		}
		return 1;
	}
	
	if ((info->flash_id & FLASH_VENDMASK) ==
	    (AMD_MANUFACT & FLASH_VENDMASK))
	{
		size = (unsigned)&_amd_erase_flash_end -
			(unsigned)_amd_erase_flash; 
		
		if (size > PROBE_BUFFER_SIZE) {
			printf("_amd_erase_flash() "
			       "routine too large (%d) %p - %p\n",
			       size, &_amd_erase_flash_end, _amd_erase_flash);
			return 0;
		}
		
		memcpy(buffer, _amd_erase_flash, size);
		_erase_flash_ptr = (void*)buffer;
	
	}  else {
		printf ("Can't erase unknown flash type - aborted\n");
		return 1;
	}
	
	prot = 0;
	for (sect = s_first; sect <= s_last; ++sect) {
		if (info->protect[sect]) {
			prot++;
		}
	}
	
	if (prot) {
		printf ("- Warning: "
			"%d protected sectors will not be erased!\n", prot);
	} else {
		printf ("\n");
	}
		
	/* Start erase on unprotected sectors */
	for (sect = s_first; sect <= s_last; sect++) {
		
		if (info->protect[sect] == 0) { /* not protected */
			int res;
			int flag;
			
			/* Disable interrupts which might cause a timeout */
			flag = disable_interrupts();
			
			res = _erase_flash_ptr(info->start[0],
					       info->start[sect] - 
					       info->start[0]);
			
			/* re-enable interrupts if necessary */
			if (flag) {
				enable_interrupts();
			}
			
			if (res) {
				printf("Erase timed out, sector %d\n", sect);
				return res;
			}
			
			putc('.');			
		}		
	}

	return 0;
}

/*-----------------------------------------------------------------------
 * Write a byte to Flash, returns:
 * 0 - OK
 * 1 - write timeout
 * 2 - Flash not erased
 */
static int _amd_write_byte(unsigned start, unsigned dest, unsigned data)
{
	volatile u8 *addr2 = (u8 *)start;
	volatile u8 *dest2 = (u8 *)dest;
	volatile u8 *data2 = (u8 *)&data;
	unsigned elapsed;
	
	/* Check if Flash is (sufficiently) erased */
	if ((*((volatile u8 *)dest) & (u8)data) != (u8)data) {
		return 2;
	}
		
	addr2[0x5555] = 0xAA;
	addr2[0x2aaa] = 0x55;
	addr2[0x5555] = 0xA0;
	
	dest2[0] = data;
	
	elapsed = *(volatile u16*)(0xfffef000+SC520_SWTMRMILLI); /* dummy read */
	elapsed = 0;
	
	/* data polling for D7 */
	while ((dest2[0] & 0x80) != (data2[0] & 0x80)) {
		elapsed += *(volatile u16*)(0xfffef000+SC520_SWTMRMILLI);
		if (elapsed > ((CFG_FLASH_WRITE_TOUT/CFG_HZ) * 1000)) {
			addr2[0] = 0xf0;
			return 1;			
		}
	}
	
	addr2[0] = 0xf0;
	
	return 0;
}

extern int _amd_write_byte_end;
asm ("_amd_write_byte_end:\n"
     ".long 0\n");




/*-----------------------------------------------------------------------
 * Copy memory to flash, returns:
 * 0 - OK
 * 1 - write timeout
 * 2 - Flash not erased
 * 3 - Unsupported flash type
 */

int write_buff(flash_info_t *info, uchar *src, ulong addr, ulong cnt)
{
	int rc, flag;
	u32 (*_write_byte_ptr)(unsigned start, unsigned dest, unsigned data);
	unsigned size;
	
	if ((info->flash_id & FLASH_VENDMASK) ==
	    (AMD_MANUFACT & FLASH_VENDMASK))
	{
		size = (unsigned)&_amd_write_byte_end - 
			(unsigned)_amd_write_byte; 
		
		if (size > PROBE_BUFFER_SIZE) {
			printf("_amd_write_byte() "
			       "routine too large (%d) %p - %p\n",
			       size, &_amd_write_byte_end, _amd_write_byte);
			return 0;
		}
		
		memcpy(buffer, _amd_write_byte, size);
		_write_byte_ptr = (void*)buffer;
	
	} else {
		printf ("Can't program unknown flash type - aborted\n");
		return 3;
	}

	while (cnt >= 0) {
		/* Disable interrupts which might cause a timeout here */
		flag = disable_interrupts();

		rc = _write_byte_ptr(info->start[0], addr, *src++);
		
		/* re-enable interrupts if necessary */
		if (flag) {
			enable_interrupts();
		}
		if (rc != 0) {
			return rc;
		}
		addr++;
		cnt--;
	}
	
	return 0;
}


