/*
 * Sitewatch SC520
 * Brad Parker <brad@heeltoe.com>
 *
 * (C) Copyright 2002
 * Daniel Engstrm, Omicron Ceti AB <daniel@omicron.se>.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <pci.h>
#include <asm/io.h>
#include <asm/pci.h>
#include <asm/ic/sc520.h>
#include <spi.h>

#undef SC520_SW_DEBUG

#ifdef	SC520_SW_DEBUG
#define	PRINTF(fmt,args...)	printf (fmt ,##args)
#else
#define PRINTF(fmt,args...)
#endif

/* ------------------------------------------------------------------------- */

	
/* 
 * Theory:
 * We first set up all IRQs to be non-pci, edge triggered,
 * when we later enumerate the pci bus and pci_sc520_fixup_irq() gets 
 * called we reallocate irqs to the pci bus with sc520_pci_set_irq()
 * as needed. We choose the irqs to grab from a configurable list
 * inside pci_sc520_fixup_irq() (If this list contains stupid irq's
 * such as 0 things will not work)
 */

static void irq_init(void)
{
	/* disable global interrupt mode */
	write_mmcr_byte(SC520_PICICR, 0x40); 
	
	/* set all irqs to edge */
	write_mmcr_byte(SC520_MPICMODE, 0x00);
	write_mmcr_byte(SC520_SL1PICMODE, 0x00);
	write_mmcr_byte(SC520_SL2PICMODE, 0x00);
	
	/* active low polarity on PIC interrupt pins, 
	 *  active high polarity on all other irq pins */
	write_mmcr_word(SC520_INTPINPOL, 0x0000);

	/* set irq number mapping */
	write_mmcr_byte(SC520_GPTMR0MAP, SC520_IRQ_DISABLED);   /* disable GP timer 0 INT */       
	write_mmcr_byte(SC520_GPTMR1MAP, SC520_IRQ_DISABLED);   /* disable GP timer 1 INT */
	write_mmcr_byte(SC520_GPTMR2MAP, SC520_IRQ_DISABLED);   /* disable GP timer 2 INT */
	write_mmcr_byte(SC520_PIT0MAP, SC520_IRQ0);             /* Set PIT timer 0 INT to IRQ0 */ 
	write_mmcr_byte(SC520_PIT1MAP, SC520_IRQ_DISABLED);     /* disable PIT timer 1 INT */
	write_mmcr_byte(SC520_PIT2MAP, SC520_IRQ_DISABLED);     /* disable PIT timer 2 INT */
	write_mmcr_byte(SC520_PCIINTAMAP, SC520_IRQ_DISABLED);  /* disable PCI INT A */
	write_mmcr_byte(SC520_PCIINTBMAP, SC520_IRQ_DISABLED);  /* disable PCI INT B */
	write_mmcr_byte(SC520_PCIINTCMAP, SC520_IRQ_DISABLED);  /* disable PCI INT C */
	write_mmcr_byte(SC520_PCIINTDMAP, SC520_IRQ_DISABLED);  /* disable PCI INT D */
	write_mmcr_byte(SC520_DMABCINTMAP, SC520_IRQ_DISABLED); /* disable DMA INT */ 
	write_mmcr_byte(SC520_SSIMAP, SC520_IRQ_DISABLED);      /* disable Synchronius serial INT */
	write_mmcr_byte(SC520_WDTMAP, SC520_IRQ_DISABLED);      /* disable Watchdog INT */
	write_mmcr_byte(SC520_RTCMAP, SC520_IRQ8);              /* Set RTC int to 8 */
	write_mmcr_byte(SC520_WPVMAP, SC520_IRQ_DISABLED);      /* disable write protect INT */
//	write_mmcr_byte(SC520_ICEMAP, SC520_IRQ1);              /* Set ICE Debug Serielport INT to IRQ1 */
	write_mmcr_byte(SC520_ICEMAP, SC520_IRQ_DISABLED);      /* disable ICE Debug Serialport INT*/
	write_mmcr_byte(SC520_FERRMAP,SC520_IRQ13);             /* Set FP error INT to IRQ13 */
	
	write_mmcr_byte(SC520_UART1MAP, SC520_IRQ4);         /* Set internal UART2 INT to IRQ4 */
	write_mmcr_byte(SC520_UART2MAP, SC520_IRQ3);         /* Set internal UART2 INT to IRQ3 */

//	write_mmcr_byte(SC520_GP3IMAP, SC520_IRQ_DISABLED);  /* disable GPIRQ3 (ISA IRQ3) */ 
//	write_mmcr_byte(SC520_GP4IMAP, SC520_IRQ_DISABLED);  /* disable GPIRQ4 (ISA IRQ4) */
	
	write_mmcr_byte(SC520_GP1IMAP, SC520_IRQ1);             /* Set GPIRQ1 (SIO IRQ1) to IRQ1 (shared) */
	write_mmcr_byte(SC520_GP5IMAP, SC520_IRQ5);             /* Set GPIRQ5 (ISA IRQ5) to IRQ5 */
	write_mmcr_byte(SC520_GP6IMAP, SC520_IRQ6);             /* Set GPIRQ6 (ISA IRQ6) to IRQ6 */
	write_mmcr_byte(SC520_GP7IMAP, SC520_IRQ7);             /* Set GPIRQ7 (ISA IRQ7) to IRQ7 */
	write_mmcr_byte(SC520_GP8IMAP, SC520_IRQ15);             /* Set GPIRQ8 (SIO IRQ8) to IRQ15 */

	write_mmcr_byte(SC520_GP9IMAP, SC520_IRQ12);             /* Set GPIRQ9 (ISA IRQ2) to IRQ12 */

	write_mmcr_byte(SC520_GP0IMAP, SC520_IRQ1);            /* Set GPIRQ0 (ISA IRQ11) to IRQ10 (shared) */
	write_mmcr_byte(SC520_GP2IMAP, SC520_IRQ2);            /* Set GPIRQ2 (ISA IRQ12) to IRQ2 (shared) */
	write_mmcr_byte(SC520_GP10IMAP,SC520_IRQ14);            /* Set GPIRQ10 (ISA IRQ14) to IRQ14 */          

	write_mmcr_byte(SC520_GP3IMAP,SC520_IRQ2);	/* Set GPIRQ3 (ISA IRQ14) to IRQ2 (shared) */

	write_mmcr_byte(SC520_GP4IMAP,SC520_IRQ11);	/* Set GPIRQ4 (ISA IRQ14) to IRQ11 */
	
	write_mmcr_word(SC520_PCIHOSTMAP, 0x11f);                /* Map PCI hostbridge INT to NMI */
	write_mmcr_word(SC520_ECCMAP, 0x100);                    /* Map SDRAM ECC failure INT to NMI */
 
}

	
/* PCI stuff */
static void pci_sc520_sw_fixup_irq(struct pci_controller *hose, pci_dev_t dev)
{
	/* a configurable lists of irqs to steal
	 * when we need one (a board with more pci interrupt pins
	 * would use a larger table */
	static int irq_list[] = {
		CFG_FIRST_PCI_IRQ,
		CFG_SECOND_PCI_IRQ,
		CFG_THIRD_PCI_IRQ,
		CFG_FORTH_PCI_IRQ
	};
	static int next_irq_index=0;
	
	char tmp_pin;	
	int pin;
	
	pci_hose_read_config_byte(hose, dev, PCI_INTERRUPT_PIN, &tmp_pin);
	pin = tmp_pin;
	
	pin-=1; /* pci config space use 1-based numbering */
	if (-1 == pin) {
		return; /* device use no irq */
	}
	
#if 0
	printf("fixup_irq: device %d pin %c\n", 
	       PCI_DEV(dev), 'A' + pin);
#endif
       
	/* map device number +  pin to a pin on the sc520 */
	switch (PCI_DEV(dev)) {
	case 0xd:	/* PCNet */
		pin+=SC520_PCI_INTA;
		break;
		
	default: 
		return;
	}
	
	pin&=3; /* wrap around */
	
	if (sc520_pci_ints[pin] == -1) {
		/* re-route one interrupt for us */		
		if (next_irq_index > 3) {
			return;
		}
		if (pci_sc520_set_irq(pin, irq_list[next_irq_index])) { 
			return;
		}
		next_irq_index++;
	}

	
	if (-1 != sc520_pci_ints[pin]) {
		pci_hose_write_config_byte(hose, dev, PCI_INTERRUPT_LINE, 
					   sc520_pci_ints[pin]);
	}
#if 0
	printf("fixup_irq: device %d pin %c irq %d\n", 
	       PCI_DEV(dev), 'A' + pin, sc520_pci_ints[pin]);
#endif
}
 
static struct pci_controller sc520_sw_hose = {
	fixup_irq: pci_sc520_sw_fixup_irq,
};

void pci_init_board(void)
{
	pci_sc520_init(&sc520_sw_hose);
}

/* set up the ISA bus timing and system address mappings */
static void bus_init(void)
{
	int slot;

	/* set up the GP IO pins */

 	/* set the GPIO pin function 31-16 reg */		   
	write_mmcr_word(SC520_PIOPFS31_16, 0xf7ff);

  	/* set the GPIO pin function 15-0 reg */
	write_mmcr_word(SC520_PIOPFS15_0, 0xfffb);

	/* set the CS pin function  reg */	
	/* enable cs7-cs1 */
	write_mmcr_byte(SC520_CSPFS, 0xfe);

	write_mmcr_byte(SC520_CLKSEL, 0x70);

	
	write_mmcr_byte(SC520_GPCSRT, 1);   /* set the GP CS offset */
	write_mmcr_byte(SC520_GPCSPW, 3);   /* set the GP CS pulse width */
	write_mmcr_byte(SC520_GPCSOFF, 1);  /* set the GP CS offset */
	write_mmcr_byte(SC520_GPRDW, 3);    /* set the RD pulse width */
	write_mmcr_byte(SC520_GPRDOFF, 1);  /* set the GP RD offset */
        write_mmcr_byte(SC520_GPWRW, 3);    /* set the GP WR pulse width */
	write_mmcr_byte(SC520_GPWROFF, 1);  /* set the GP WR offset */

	write_mmcr_word(SC520_BOOTCSCTL, 0x1823);/* set up timing of BOOTCS */ 
	write_mmcr_word(SC520_ROMCS1CTL, 0x1823);/* set up timing of ROMCS1 */
	write_mmcr_word(SC520_ROMCS2CTL, 0x1823);/* set up timing of ROMCS2 */ 
	
	/*
	 * Adjust the memory map.
	 * by default:
	 * the first 256MB  (0x00000000 - 0x0fffffff) is mapped to SDRAM and
	 * 256MB to 1G-128k (0x01000000 - 0x37ffffff) is mapped to PCI mmio
	 * need to map 1G-128k - 1G (0x38000000 - 0x3fffffff) to CS1 (flash)
	 *
	 * GPCS usage:
	 *
	 * gpcs0
	 * gpcs1		romcs#1			mem
	 * gpcs2		romcs#2			mem
	 * gpcs3		siobuscs#		i/o
	 *		a10-12 are decoded as slot #
	 *		000	lio_cs#0 - J25	0x**** 00xx
	 *		001	lio_cs#1 - J26	0x**** 04xx
	 *		010	lio_cs#2 - J27	0x**** 08xx
	 *		011	lio_cs#3 - J28	0x**** 0cxx
	 *		100	lio_cs#4 - J29	0x**** 10xx
	 *		101	lio_cs#5 - J30	0x**** 14xx
	 *		110	lio_cs#6 - J31	0x**** 18xx
	 *		111	lio_cs#7 - J32	0x**** 1cxx
	 *
	 * gpcs4		cfcs#1			mem
	 * gpcs5		cfcs#2			mem
	 * gpcs6		idecs#1			i/o
	 * gpcs7		idecs#2			i/o
	 *	
	 * PAR usage:
	 *
	 * PAR0   legacy_video		flash
	 * PAR1   PCI ROM mapping	
	 * PAR2   (unused)		doc mem, 0x3001xxxx
	 * PAR3   IDE			doc mem, 0x3002xxxx
	 * PAR4   IDE			cf  mem, 0x3004xxxx
	 * PAR5   legacy_video		cf  mem, 0x3005xxxx
	 * PAR6   legacy_video		ide i/o, 0x1f0
	 * PAR7   legacy_video		ide i/o, 0x170
	 *
	 * PAR8   legacy_video		slot 1
	 * PAR9   legacy_video		slot 2
	 * PAR10  legacy_video		slot 3
	 * PAR11  (unused)		slot 4
	 * PAR12  BOOTCS		slot 5
	 * PAR13  ROMCS1		slot 6
	 * PAR14  ROMCS2		slot 7
	 * PAR15  (unused)		slot 8
	 */

#define PAR_TARGET_GPIO		1
#define PAR_TARGET_GPMEM	2
#define PAR_TARGET_PCIMEM	3
#define PAR_TARGET_BOOTCS	4
#define PAR_TARGET_ROMCS1	5
#define PAR_TARGET_ROMCS2	6
#define PAR_TARGET_SDRAM	7

#define MAKE_PAR(target, attr, pg_sz, sz_st_adr) \
	(((target) & 0x7) << 29 | \
	 ((attr) & 0x7) << 26 | \
	 ((pg_sz) & 0x1) << 25 | \
	 ((sz_st_adr) & 0x1ffffff))

	/* --- new --- */

	/* doc1, PAR2, GPCS1 */
	write_mmcr_long(SC520_PAR2,
			MAKE_PAR(PAR_TARGET_GPMEM, 1, 1, 0x3001));

	/* doc2, PAR3, GPCS2 */
	write_mmcr_long(SC520_PAR3,
			MAKE_PAR(PAR_TARGET_GPMEM, 2, 1, 0x3002));

	/* cf1, PAR4, GPCS4 */
	write_mmcr_long(SC520_PAR4,
			MAKE_PAR(PAR_TARGET_GPMEM, 4, 1, 0x3004));

	/* cf2, PAR5, GPCS5 */
	write_mmcr_long(SC520_PAR5,
			MAKE_PAR(PAR_TARGET_GPMEM, 5, 1, 0x3005));

	/* ide, PAR6-7, GPCS6-7 */
	write_mmcr_long(SC520_PAR6,
			MAKE_PAR(PAR_TARGET_GPIO, 6, 0, (16<<16)|0x1f0));

	write_mmcr_long(SC520_PAR7,
			MAKE_PAR(PAR_TARGET_GPIO, 7, 0, (16<<16)|0x170));

	/* chip select data widths - gpcs#4,5 are 16 bit, 6 & 7 are 8 */
	write_mmcr_byte(SC520_GPCSDW, (1 << 5) | (1 << 4));

	/* set up PAR8-15 for serial slots */
	for (slot = 0; slot < 8; slot++) {
		int base, par_reg, par_val;

		base = 0x4000 + (slot << 10);

		par_val = MAKE_PAR(PAR_TARGET_GPIO, 3, 1, (255<<16) | base);

		switch (slot) {
		default:
		case 0: par_reg = SC520_PAR8; break;
		case 1: par_reg = SC520_PAR9; break;
		case 2: par_reg = SC520_PAR10; break;
		case 3: par_reg = SC520_PAR11; break;
		case 4: par_reg = SC520_PAR12; break;
		case 5: par_reg = SC520_PAR13; break;
		case 6: par_reg = SC520_PAR14; break;
		case 7: par_reg = SC520_PAR15; break;
		}

		write_mmcr_long(par_reg, par_val);
	}
	
	write_mmcr_byte(SC520_ADDDECCTL, 0);
	
	/* Flush cache, req. after setting the unchached attribute ona PAR */
	asm ("wbinvd\n");

	/* make sure uart1 & 2 are not disabled */
	write_mmcr_byte(SC520_ADDDECCTL,
			read_mmcr_byte(SC520_ADDDECCTL) &
			~(RTC_DIS|UART2_DIS|UART1_DIS));
}

#ifdef CONFIG_PCI
#define PCI_ROM_TEMP_SPACE 0x10000
/* 
 * This function should map a chunk of size bytes
 * of the system address space to the PCI bus,
 * suitable to map PCI ROMS (bus address < 16M)
 * the function will return the host memory address
 * which should be converted into a bus address
 * before used to configure the PCI rom address 
 * decoder
 */
u32 pci_get_rom_window(struct pci_controller *hose, int size) 
{
#if 0
	u32 par;
	
	par = size;
	if (par < 0x80000) {
		par = 0x80000;
	}
	par >>= 16;
	par--;
	par&=0x7ff;
	par <<= 14;
	par |= (PCI_ROM_TEMP_SPACE>>16);
	par |= 0x72000000;
	
	PRINTF ("setting PAR1 to %x\n", par);
	
	/* Map rom 0x10000 with PAR1 */
	write_mmcr_long(SC520_PAR1,  par);
#endif
	
	return PCI_ROM_TEMP_SPACE;
}

/*
 * this function removed any mapping created
 * with pci_get_rom_window()
 */
void pci_remove_rom_window(struct pci_controller *hose, u32 addr)
{
	PRINTF("pci_remove_rom_window: %x", addr);
	if (addr == PCI_ROM_TEMP_SPACE) {
		write_mmcr_long(SC520_PAR1, 0);
		PRINTF(" done\n");
		return;
	}
	PRINTF(" not ours\n");
	
}

/*
 * This function is called in order to provide acces to the
 * legacy video I/O ports on the PCI bus. 
 * After this function accesses to I/O ports 0x3b0-0x3bb and 
 * 0x3c0-0x3df shuld result in transactions on the PCI bus.
 * 
 */
int pci_enable_legacy_video_ports(struct pci_controller *hose)
{
#if 0
	/* Map video memory to 0xa0000*/
	write_mmcr_long(SC520_PAR0,  0x7200400a);
	
	/* forward all I/O accesses to PCI */
	write_mmcr_byte(SC520_ADDDECCTL, 
			read_mmcr_byte(SC520_ADDDECCTL) | IO_HOLE_DEST_PCI);

	/*
	 * Map away all io ports to pci (only way to access pci io below 0x400.
	 * Then we have to map back the portions that we don't use so that
	 * it generates cycles on the GPIO bus where the sio and ISA slots
	 * are connected; this require the use of several PAR registers...
	 */
	
	/* bring 0x100 - 0x1ef back to ISA using PAR5 */
	write_mmcr_long(SC520_PAR5, 0x30ef0100);		
	
	/* IDE use 1f0-1f7 */
	
	/* bring 0x1f8 - 0x2f7 back to ISA using PAR6 */
	write_mmcr_long(SC520_PAR6, 0x30ff01f8);		
	
	/* com2 use 2f8-2ff */
	
	/* bring 0x300 - 0x3af back to ISA using PAR7 */
	write_mmcr_long(SC520_PAR7, 0x30af0300);		
	
	/* vga use 3b0-3bb */
	
	/* bring 0x3bc - 0x3bf back to ISA using PAR8 */
	write_mmcr_long(SC520_PAR8, 0x300303bc);		
	
	/* vga use 3c0-3df */
	
	/* bring 0x3e0 - 0x3f5 back to ISA using PAR9 */
	write_mmcr_long(SC520_PAR9, 0x301503e0);		
	
	/* ide use 3f6 */
	
	/* bring 0x3f7  back to ISA using PAR10 */
	write_mmcr_long(SC520_PAR10, 0x300003f7);		
	
	/* com1 use 3f8-3ff */		
#endif

	return 0;
}
#endif /* CONFIG_PCI */

/*
 * Miscelaneous platform dependent initialisations
 */

int board_init(void)
{
	DECLARE_GLOBAL_DATA_PTR;
	
	init_sc520();	
	bus_init();
	irq_init();
		
	/* max drive current on SDRAM */
	write_mmcr_word(SC520_DSCTL, 0x0100);
		
	/* enter debug mode after next reset (only if jumper is also set) */
	write_mmcr_byte(SC520_RESCFG, 0x08);

	/* configure the software timer to 33.333MHz */
	write_mmcr_byte(SC520_SWTMRCFG, 0);
	gd->bus_clk = 33333000;
	
	return 0;
}

int dram_init(void)
{
	init_sc520_dram();
	return 0;
}

int last_stage_init(void)
{
	/* find board rev from h/w & printf */
	return 0;
}


void watchdog_reset(void)
{
}

#if (CONFIG_COMMANDS & CFG_CMD_DOC)
extern void doc_probe (ulong physadr);
void doc_init (void)
{
	doc_probe(CFG_DOC_BASE1);
	doc_probe(CFG_DOC_BASE2);
}
#endif
