/*
 * main.c
 *
 * simple event based driver for modified smartlinc firmware
 */

#include <stdio.h>
#include <fcntl.h>
#include <sys/poll.h>

#define MAX_IDLE_SECONDS 15

int debug = 0;
int testing = 0;

int events;
enum {
  null_event = 0,
  idle_event,
  timer_event,
  touch_event
};

int state;
enum {
  null_state = 0,
  idle_state,
  active_state
};

int idle_timer;
int in_idle_timer;
int in_active_timer;

/*
 * move to new state - 
 * add code here for entering each state 
 */
int
new_state(int new_state)
{
  if (new_state == state)
    return 0;

  switch (new_state) {
  case idle_state:
    if (debug) printf("going idle\n");
    backlight(0);
    break;

  case active_state:
    if (debug) printf("going active\n");
    backlight(1);
    break;
  }

  state = new_state;

  return 0;
}

int
do_idle_event(void)
{
  if (debug) printf("do_idle_event()\n");

  new_state(idle_state);
  return 0;
}

int
do_touch_event(void)
{
  if (debug) printf("do_touch_event()\n");

  idle_timer = 0;

  switch (state) {
  case idle_state:
    new_state(active_state);
    break;
  case active_state:
    break;
  }
  return 0;
}

int
do_timer_event(void)
{
  if (debug) printf("do_timer_event()\n");

  switch (state) {

  case idle_state:
    update_weather();
    if (++in_idle_timer >= 60) {
	    update_display(0);
	    in_idle_timer = 0;
    }
    break;

  case active_state:
    if (++idle_timer >= MAX_IDLE_SECONDS)
      post_event(idle_event);
    else {
      if (++in_active_timer >= 2) {
        in_active_timer = 0;
        update_mail();
	update_weather();
	update_display(1);
      }
    }
    break;
  }
  return 0;
}

int
process_event(int event_no)
{
  if (events & (1 << event_no)) {
    events &= ~(1 << event_no);
    return 1;
  }

  return 0;
}

int
post_event(int event_no)
{
  events |= 1 <<event_no;
}

int
touchpad_hit(int r, int c)
{
  if (debug) printf("touchpad(%d,%d)\n",r, c);
  post_event(touch_event);
}

int
poll_things(void)
{
  struct pollfd ufds[2];
  int r;
  extern int fd;

  ufds[0].fd = fd;
  ufds[0].events = POLLIN;
  ufds[0].revents = 0;

  r = poll(ufds, 1, 1000);
  if (r < 0)
    return -1;

  if (r == 0)
    post_event(timer_event);

  if (r > 0) {

    if (ufds[0].revents & POLLIN) {
      if (debug) printf("POLLIN\n");
      readback();
    }
  }

  return 0;
}

main(int argc, char *argv[])
{
  if (debug) printf("init\n");

  setup_smartlinc();
  setup_display();
  new_state(active_state);

  if (debug) printf("init done\n");

  /* loop forever */

  while (1) {
    /* generate events */

    if (poll_things())
      break;

    /* handle events */

    if (debug) printf("loop, events %x\n", events);

    if (events) {
      if (process_event(idle_event))
	  do_idle_event();

      if (process_event(touch_event))
	  do_touch_event();

      if (process_event(timer_event))
	  do_timer_event();
    }
  }
}
