/*
 * smartlink.c
 */

#include <stdio.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>

int fd;
char devname[1024];
int text_addr;
int graphics_addr;

extern int debug;
extern int testing;

#define DEVNAME "/dev/ttyS0"
#define LCD_COLS 30/*32*/

/* T6963C commands */
#define T_SET_CURSOR_PTR	0x21
#define T_SET_OFFSET_REG	0x22
#define T_SET_ADDRESS_PTR	0x24

#define T_SET_TEXT_HOME		0x40
#define T_SET_TEXT_AREA		0x41
#define T_SET_GRAPHICS_HOME	0x42
#define T_SET_GRAPHICS_AREA	0x43

#define T_MODE_OR		0x80
#define T_MODE_EXOR		0x81
#define T_MODE_AND		0x83
#define T_MODE_TEXT_ATTR	0x84
#define T_MODE_INT_CG_ROM	0x87
#define T_MODE_EXT_CG_ROM	0x88

#define T_DISPLAY_OFF		0x90
#define T_CURSOR_ON_BLINK_OFF	0x92
#define T_CURSOR_ON_BLINK_ON	0x93
#define T_TEXT_ON_GRAPHICS_OFF	0x94
#define T_TEXT_OFF_GRAPHICS_ON	0x98
#define T_TEXT_ON_GRAPHICS_ON	0x9c

#define T_CURSOR_1LINE		0xa0
#define T_CURSOR_8LINE		0xa7

#define T_SET_DATA_AUTO_WRITE	0xb0
#define T_SET_DATA_AUTO_READ	0xb1
#define T_AUTO_RESET		0xb2

#define T_DATA_WRITE_INCR_ADP	0xc0
#define T_DATA_WRITE_DECR_ADP	0xc2
#define T_DATA_WRITE		0xc4


int
init_sio()
{
  struct termios tp;

  fd = open(devname, O_RDWR);
  if (fd < 0) {
    perror(devname);
    return -1;
  }

  tcgetattr(fd, &tp);
  cfmakeraw(&tp);
  cfsetispeed(&tp, B9600);
  cfsetospeed(&tp, B9600);
  tcsetattr(fd, TCSANOW, &tp);
}

char tohex(int n)
{
  if (n < 10) return '0'+n;
  return 'a'+n-10;
}

static char rb_buf[512];
static int rb_cnt;

int
parse_readback_line()
{
    if (rb_buf[0] == 'O' && rb_buf[1] == 'K') {
      if (debug > 1) printf("OK\n");
      return 1;
    }
    if (rb_buf[0] == '?') {
      if (debug) printf("error?\n");
      return 2;
    }

    if (debug > 1) printf("%c %c %c\n", rb_buf[0], rb_buf[1], rb_buf[2]);

    if ('0' <= rb_buf[0] && rb_buf[0] <= '9' && rb_buf[2] == 'M') {
      touchpad_hit(rb_buf[0]-'0', rb_buf[1]-'0');
    }
}

int
parse_readback(char *b, int nb)
{
  int i;

  for (i = 0; i < nb; i++) {
    if (rb_cnt < sizeof(rb_buf))
      rb_buf[rb_cnt++] = b[i];

    if (b[i] == '\n') {
      parse_readback_line();
      rb_cnt = 0;
    }
  }

  /* bug in firmware does not sent newline */
  if (nb > 2 && rb_buf[2] == 'M') {
      parse_readback_line(i);
      rb_cnt = 0;
  }

  return 0;
}

int
readback(void)
{
  int i, n;
  char buf[32];

  if (testing) return 0;

  n = read(fd, buf, sizeof(buf));
  if (n > 0) {
    parse_readback(buf, n);
  }

  return 0;
}


int
send_mesg(int v, int c)
{
  char buf[32];
  int n;

  if (testing)
	  return 0;

  buf[0] = tohex(v >> 4);
  buf[1] = tohex(v & 0xf);
  buf[2] = c;

  if (debug > 1) {
    printf("-> %02x %02x %02x\n", buf[0], buf[1], buf[2]);
  }

  write(fd, buf, 3);
  //usleep(0);

  readback();

  return 0;
}

int
send_data(int n)
{
  return send_mesg(n, 'D');
}

int
send_cmd(int n)
{
  return send_mesg(n, 'L');
}

int
set_address_ptr(int ptr)
{
    send_data(ptr & 0xff);
    send_data(ptr >> 8);
    send_cmd(T_SET_ADDRESS_PTR);
    return 0;
}

int
set_cursor(int x, int y)
{
    send_data(x);
    send_data(y);
    send_cmd(T_SET_CURSOR_PTR);
    return 0;
}

int
set_offset(int offset)
{
    send_data(offset);
    send_data(0x0);
    send_cmd(T_SET_OFFSET_REG);
    return 0;
}

int
write_data_incr(int c)
{
  send_data(c);
  send_cmd(T_DATA_WRITE_INCR_ADP);
  return 0;
}

void
set_text_cols(int cols)
{
  send_data(cols);
  send_data(0);
  send_cmd(T_SET_TEXT_AREA);
}

void
set_graphics_cols(int cols)
{
  send_data(cols);
  send_data(0);
  send_cmd(T_SET_GRAPHICS_AREA);
}

void
set_text_home(int home)
{
  send_data(home & 0xff);
  send_data(home >> 8);
  send_cmd(T_SET_TEXT_HOME);
}

void
set_graphics_home(int home)
{
  send_data(home & 0xff);
  send_data(home >> 8);
  send_cmd(T_SET_GRAPHICS_HOME);
}

int
send_line(int line, char *txt)
{
  char lb[32];
  int i, l;

  //  if (line > 2) return;

  for (i = 0; i < 32; i++)
    lb[i] = 0x0;

  l = strlen(txt);
  for (i = 0; i < l; i++)
    lb[i] = txt[i] - 0x20;

  set_address_ptr(text_addr + LCD_COLS*line);
  for (i = 0; i < /*32*/l; i++) {
    write_data_incr(lb[i]);
  }

  return 0;
}

int
send_pos(int line, int col, char *txt, int n)
{
  char lb[32];
  int i, l;

  for (i = 0; i < 32; i++)
    lb[i] = 0x0;

  for (i = 0; i < n; i++)
    lb[i] = txt[i] - 0x20;

  set_address_ptr(text_addr + LCD_COLS*line + col);

  for (i = 0; i < n; i++) {
    write_data_incr(lb[i]);
  }

  return 0;
}

int
backlight(int onoff)
{
  if (onoff) {
    send_mesg(1, 'B');
  } else {
    send_mesg(0, 'B');
  }
}

int
setup_lcd(void)
{
  set_text_home(text_addr);
  set_text_cols(30);
  set_graphics_home(graphics_addr);
  set_address_ptr(text_addr);
  send_cmd(T_MODE_EXOR);
  send_cmd(T_TEXT_ON_GRAPHICS_OFF);
}

int
setup_smartlinc()
{
  int i;

  /* serial port */
  strcpy(devname, DEVNAME);

  if (init_sio())
    exit(1);

  /* reset */
  send_mesg(1, 'R');
  usleep(100);
  send_mesg(0, 'R');

  text_addr = 0x0780;
  graphics_addr = 0x0000;

  setup_lcd();

  for (i = 0; i < 16; i++) {
    if (debug) printf("%d\n", i);
    send_line(i, "                                ");
  }

  if (debug) printf("done\n");

  return 0;
}
