;
; Smartlinc firmware code
; brad@heeltoe.com
;
; P16F73
; 16Mhz osc
;
; replacement code for the PIC16C73 in the Smartlinc device
; Uses a simple 3 byte protocol over the serial port
;  Null bytes reset state machine, so sending 3 null byts will
;  reset input state.
;
; host -> smartlinc
;  XXL		lcd command
;  XXD		lcd data
;  XXB		00->backlight off, 01->backlight on
;  XXR		reset
;  XXC		clear lcd
;  XXS		text auto incr write (send chars followed by null byte)
;  XXZ		beep
;
; smartlinc -> host
;  XXM		touchpad hit event
;

;#define DEBUG 1

	list p=16f73
;	list p=16c73
	__config _CP_OFF & _WDT_OFF & _PWRTE_OFF & _HS_OSC
	radix dec
	
	include "p16f73.inc"
;	include "p16c73.inc"

;
; brg = (xtalfreq / (64 * baud)) - 1
;
#define osc_freq		16000000
#define b9600			((osc_freq/(64*9600))-1)

	; bank 0 only
	org	0x20
irqw:			res	1		; save w for isr - on all pages
irqstat:		res	1
irqfsr:			res	1

#define RX_RING_SIZE	32
rx_ring_cnt:	res 1
rx_ring_ptr:	res	1
rx_ring_get:	res	1
rx_ring:		res	RX_RING_SIZE

#define TX_RING_SIZE	32
tx_ring_cnt:	res 1
tx_ring_ptr:	res	1
tx_ring_put:	res	1
tx_ring:		res	TX_RING_SIZE

temp:			res	1
temp2:			res	1
delaycnt:		res	1
show_data:		res 1
portb_data:		res 1
portc_data:		res 1
bus_data:		res 1
lcd_data:		res 1
lcd_cmd:		res 1		; 1=send command,0=send data
latch1_data:	res 1		; current latch1 data
latch2_data:	res 1
matrix_hit:		res 1		; last matrix hit value
matrix_row:		res	1
matrix_col:		res	1
matrix_data:	res 1

input_cmd:		res 1		; last uart cmd
input_byte:		res 1		; last uart cmd arg
input_state:	res 1		; state of input
input_char:		res 1		; temp

lcd_col:		res 1
lcd_row:		res 1

	org	0x000
	goto	start
	nop
	nop
	nop
	
	org	0x004
isr
	movwf	irqw			; save w (need to be on all pages)
	swapf	irqw,1
	swapf	status,w		; put status in w
	bcf		status,rp0		; page 0
	movwf	irqstat			;  save status
	movf	fsr,w
	movwf	irqfsr			;  save fsr
	
	btfsc	pir1,RCIF
	call	uart_rx_irq
	
	btfsc	pir1,TXIF
	call	uart_tx_irq

	movf	irqfsr,w
	movwf	fsr				; restore fsr
	swapf	irqstat,w
	movwf	status			; restore status (& page)
	swapf	irqw,w			; restore w
	retfie

; -----------------------------------------------------------------------------
;	extern	eeprom_setup

;
; Program start...
;
start
	call	port_setup
	call	ring_setup
	call	uart_setup
	call	eeprom_setup
	call	lcd_setup
	call	matrix_setup

	movlw	prompt					; output initial prompt
	call	puts
	call	ok

loop:
	call	input_check				; check for uart rx
	call	matrix_check			; check for touchpad event
	goto	loop

;
; output touch pad event
;
report_matrix_hit
	movfw	matrix_row
	andlw	0x0f
	addlw	0x30
	call	uart_putc				; 1st hex digit

	movfw	matrix_col
	andlw	0x0f
	addlw	0x30
	call	uart_putc				; 2nd hex digit

	movlw	0x4d
	call	uart_putc				; output "xxM" for matrix hit
	return

matrix_check
	call	matrix_scan				; scan touch pad matrix
	iorlw	0
	btfsc	status,z				; matrix hit?
	return							; no

	movwf	matrix_hit				; yes, save it
	goto	report_matrix_hit

;
; check for input from uart
; wait for 'XXC' were XX is a hex number and C is a command
;
input_check
	call	rx_fifo_empty			; any input waiting?
	btfsc	status,z
	return

	call	rx_fifo_get				; get char
	movwf	input_char

	movfw	input_state				; state machine
	addwf	pcl,f
	goto	state0
	goto	state1
	goto	state2
	goto	state3

input_state0
	clrf	input_state				; restart state machine
	return

input_state3
	movlw	3
	movwf	input_state
	return

state0
	movfw	input_char				; 1st hex digit
	btfsc	status,z
	goto	input_state0			; zero byte resets state machine
	call	ishex
	btfss	status,z
	goto	error_cmd
	andlw	0x0f
	movwf	input_byte
	bcf		status,c
	rlf		input_byte,f			; put in high nibble
	rlf		input_byte,f
	rlf		input_byte,f
	rlf		input_byte,f
	incf	input_state,f
	return

state1
	movfw	input_char
	btfsc	status,z
	goto	input_state0			; zero byte resets state machine
	call	ishex
	btfss	status,z
	goto	error_cmd
	andlw	0x0f
	iorwf	input_byte,f			; put in low nibble
	incf	input_state,f
	return

state2
	movfw	input_char
	btfsc	status,z
	goto	input_state0			; zero byte resets state machine
	movwf	input_cmd				; save command
	call	interpret_command		; do the command
;	call	show_command			; show the command (debug)
	call	ok 
	clrf	input_state				; restart state machine
	return

state3
	movfw	input_char
	btfsc	status,z
	goto	input_state0			; zero byte resets state machine
	call	lcd_text_write_data
	return

;
; debug - show command
;
show_command
	movlw	'i'
	call	uart_putc
	movfw	input_byte
	call	show_byte
	movlw	'c'
	call	uart_putc
	movfw	input_cmd
	call	show_byte
	call	crlf
	return
	
;
;
;
ok 
	movlw	'O'						; output "OK" + cr,lf
	call	uart_putc
	movlw	'K'
	call	uart_putc
	goto	crlf

error_cmd
	movlw	0x3f
	call	uart_putc				; output "?" + cr,lf
crlf
	movlw	0x0d
	call	uart_putc
	movlw	0x0a
	call	uart_putc
	return

;
; interpret command characters
;
interpret_command
	movfw	input_cmd
	sublw	'B'
	btfsc	status,z
	goto	backlight_cmd

	movfw	input_cmd
	sublw	'D'
	btfsc	status,z
	goto	lcdd_cmd

	movfw	input_cmd
	sublw	'L'
	btfsc	status,z
	goto	lcdc_cmd

	movfw	input_cmd
	sublw	'M'
	btfsc	status,z
	goto	matrix_check

	movfw	input_cmd
	sublw	'R'
	btfsc	status,z
	goto	reset_cmd

	movfw	input_cmd
	sublw	'C'
	btfsc	status,z
	goto	lcd_clear_text

	movfw	input_cmd
	sublw	'S'
	btfsc	status,z
	goto	input_state3
					
	movfw	input_cmd
	sublw	'Z'
	btfsc	status,z
	goto	buzz
			
	movfw	input_cmd
	sublw	'W'
	btfsc	status,z
	goto	loop_cmd

	goto	error_cmd				

;
; test whether character in W is a hex digit
;   returns with status.Z clear if not hex
;   returns with status.Z set if hex, and value in W
;
                            ; 0-9      A-F       a-f       Z
                            ; $30-39   $41-$46   $61-$66   $4a
ishex
	addlw   0xc6			; $f6-$ff  $07-$0c   $27-$2c   $10
	addlw   0x0a			; $00-$09  $11-$16   $31-$36   $1a
	btfsc   status,C
	goto    is09

	addlw   0xe9			;          $fa-$ff   $1a-$1f   $03
	addlw   0x06			;          $00-$05   $20-$25   $09
	btfsc   status,C
	goto    isaf

	addlw   0xda			;                    $fa-$ff   $e3
	addlw   0x06			;                    $00-$05   $e9
	btfsc   status,C
	goto    isaf

	addlw   0x61			;                              $4a
	bcf     status,Z
	return

isaf
	addlw   0x0a
is09
	bsf		status,Z
	return


;
; seup up ports A,B,C for the various smartlinc connections
; 
port_setup
	; setup option reg?

	; port A
	movlw	0x00
	movwf	porta
	
	bsf		status,rp0		; page 1
	movlw	0x3f			; make ra0-5 inputs
	movwf	trisa
	bcf		status,rp0		; page 0


; port B bits 
#define	matrix_cn41	0
#define lcd_we		3		; lcd WE#
#define	back_vdd	5		; sipex vdd
#define latch1_le	6		; active high
#define latch2_le	7		; active low, sort of

	; port B
	movlw	(1<<lcd_we) | (1<<latch2_le)
	movwf	portb_data
	movwf	portb			; disable control signals
	
	bsf		status,rp0		; page 1
	movlw	0x00			; make rb0-7 outputs
	movwf	trisb
	bcf		status,rp0		; page 0

; port C bits
#define	databus6_rc	2
#define	atmel_p6	3		; SCL
#define	atmel_p5	4		; SDA
#define	databus7_rc	5
#define	matrix_cn49	5

	; port C
	movlw	0x00
	movwf	portc_data
	movwf	portc

	bsf		status,rp0		; page 1
	movlw	(0x80 | (1<<atmel_p5) | (1<<atmel_p6) | (1<<databus6_rc) | (1<<databus7_rc))
	movwf	trisc			; make rc6 & rc7 input
	
	movlw	0x00
	movwf	indf	

	clrf	intcon

	bcf		status,rp0		; page 0

; latch1 bits 
#define	lcd_ce		2		; asserts low
#define	back_hon	3		; asserts high
#define	lcd_cd		4		; asserts low
#define	lcd_reset	5		; asserts low
#define	buzz_out	6

	movlw	0x00
	movwf	latch1_data		; latch1 initial state
	bsf		latch1_data,lcd_ce
	bsf		latch1_data,lcd_cd
	bsf		latch1_data,lcd_reset
	call	write_latch1
;
	call	big_delay
;	bcf		latch1_data,lcd_reset
;	call	write_latch1	; reset lcd
	
	call	buzz
	call	big_delay
	
;	bsf		latch1_data,lcd_reset
;	call	write_latch1	; out of reset
	call	big_delay
;

	movlw	0x00
	movwf	latch2_data		; latch2 initial state
	call	write_latch2

	clrf	input_state

	return

;
; set up databus
;
setup_databus_quick
	movfw	bus_data
	movwf	porta				; w->ra0-5

	movfw	portc_data
	
	btfsc	bus_data,6			; set rc2,5 data
	iorlw	1<<databus6_rc
	
	btfsc	bus_data,7
	iorlw	1<<databus7_rc
	
	movwf	portc				; ->rc2,rc5

	return

setup_databus
	movwf	bus_data			; save

	bsf		status,rp0			; page 1
	movlw	0x00				; make ra0-5 outputs
	movwf	trisa
	bcf		trisc,databus6_rc
	bcf		trisc,databus7_rc	; make rc6 & rc7 output
	bcf		status,rp0			; page 0

	call	setup_databus_quick
	return

reset_databus
	clrf	bus_data
	clrf	porta				; w->ra0-5
	movfw	portc_data
	movwf	portc

turn_databus	
	bsf		status,rp0			; page 1
	movlw	0x3f				; make ra0-5 inputs
	movwf	trisa
	bsf		trisc,databus6_rc
	bsf		trisc,databus7_rc	; make rc6 & rc7 inputs
	bcf		status,rp0			; page 0
	return
		
write_latch1_leavebus
	movfw	latch1_data
	call	setup_databus
	movfw	portb_data
	iorlw	1<<latch1_le		; assert latch1 le (assert high)
	movwf	portb
	nop
;	nop
;	nop
	xorlw	1<<latch1_le
	movwf	portb
	nop
;	nop
;	nop
	return	

write_latch1
	call	write_latch1_leavebus
	call	reset_databus
	return	

write_latch2
	movfw	latch2_data
	call	setup_databus
	
	movfw	portb_data
	xorlw	1<<latch2_le		; strobe latch2 le (assert low)
	movwf	portb
	nop
	iorlw	1<<latch2_le
	movwf	portb

	return

set_latch2
	movfw	latch2_data
	call	setup_databus
	
	movfw	portb_data
	xorlw	1<<latch2_le		; assert latch2 le (assert low)
	movwf	portb
	
	return

unset_latch2
	movfw	portb_data
	iorlw	1<<latch2_le		; deassert latch2 le
	movwf	portb
	return

write_portb
	movfw	portb_data
	movwf	portb
	return

;
; write to lcd, w=data, lcd_cd -> 1=command,0=data
;
lcd_write
	movwf	lcd_data			; save

	bsf		portb_data,lcd_we	; make sure lcd WE# is deasserted
	movfw	portb_data
	movwf	portb

	bsf		latch1_data,lcd_cd	; deassert latch 1, bit 5, C/D#
	btfss	lcd_cmd,1			; check what we want
	bcf		latch1_data,lcd_cd	; assert C/D#
	
	bcf		latch1_data,lcd_ce	; assert latch 1, bit 3, CE#
	call	write_latch1_leavebus

	movfw	lcd_data
	movwf	bus_data
	call	setup_databus_quick

	movfw	portb_data
	xorlw	1<<lcd_we			; strobe WE#
	movwf	portb
	
;	nop
;	nop
;	nop
	iorlw	1<<lcd_we
	movwf	portb
	
;	nop
;	nop
;	nop
	bsf		latch1_data,lcd_ce	; deassert latch 1, bit 3, CE#
	call	write_latch1

	return

lcd_write_cmd
	bsf		lcd_cmd,1
	goto	lcd_write

lcd_write_data
	bcf		lcd_cmd,1
	goto	lcd_write

lcd_text_write_data
	call	lcd_write_data
	movlw	0xc0
	goto	lcd_write_cmd			; text data write

;
; set up LCD controller
;
#define TEXT_ADDR	0x0780
#define GRAPH_ADDR	0x0000
#define LCD_ROWS	16
#define LCD_COLS	30
#define GRAPH_COLS	30

lcd_setup:
	movlw	low(TEXT_ADDR)
	call	lcd_write_data
	movlw	high(TEXT_ADDR)
	call	lcd_write_data
	movlw	0x40
	call	lcd_write_cmd			; set text home

	movlw	LCD_COLS
	call	lcd_write_data
	movlw	0x00
	call	lcd_write_data
	movlw	0x41
	call	lcd_write_cmd			; 20 columns
	
	movlw	low(GRAPH_ADDR)
	call	lcd_write_data
	movlw	high(GRAPH_ADDR)
	call	lcd_write_data
	movlw	0x42
	call	lcd_write_cmd			; set graphics home

	movlw	GRAPH_COLS
	call	lcd_write_data
	movlw	0x00
	call	lcd_write_data
	movlw	0x43
	call	lcd_write_cmd			; graphics columns

	movlw	0x81
	call	lcd_write_cmd			; set "exor" mode
;
	call	lcd_clear_text
;	
	movlw	0x72-0x20				; "Reset."
	call	lcd_text_write_data
	movlw	0x65-0x20
	call	lcd_text_write_data
	movlw	0x73-0x20
	call	lcd_text_write_data
	movlw	0x65-0x20
	call	lcd_text_write_data
	movlw	0x74-0x20
	call	lcd_text_write_data
	movlw	0x2e-0x20
	call	lcd_text_write_data

	movlw	0x94
	call	lcd_write_cmd			; text on
		
	call	backlight_on			; turn on backlight

	return

;
lcd_reset_text_ptr
	movlw	low(TEXT_ADDR)
	call	lcd_write_data
	movlw	high(TEXT_ADDR)
	call	lcd_write_data
	movlw	0x24
	call	lcd_write_cmd			; address ptr
	return

;
; clear lcd text memory
;
lcd_clear_text
	call	lcd_reset_text_ptr

	movlw	LCD_ROWS
	movwf	lcd_row	
lcd_cl_1	
	movlw	LCD_COLS						; clear one row
	movwf	lcd_col
lcd_cl_2:
	movlw	0x00
	call	lcd_text_write_data
	decfsz	lcd_col,f
	goto	lcd_cl_2

	decfsz	lcd_row,f
	goto	lcd_cl_1

	call	lcd_reset_text_ptr
	return


;
; turn on backlight
; enable Vdd to sipex inverter, assert HON signal
;
backlight_on
	bsf		portb_data,back_vdd		; turn on vdd to sipex inverter
	call	write_portb
	bsf		latch1_data,back_hon	; assert latch 1, HON bit
	call	write_latch1
	return

backlight_off
	bcf		portb_data,back_vdd		; turn off vdd to sipex inverter
	call	write_portb
	bcf		latch1_data,back_hon	; deassert latch 1, HON bit
	call	write_latch1
	return

;
; simple command routines for uart input parser
;
lcdc_cmd
	movfw	input_byte
	call	lcd_write_cmd
	return

lcdd_cmd
	movfw	input_byte
	call	lcd_write_data
	return

lcd_active
	bsf		latch1_data,lcd_reset
	call	write_latch1
	return

lcd_inactive
	bcf		latch1_data,lcd_reset
	call	write_latch1
	return

backlight_cmd
	movfw	input_byte
	iorlw	0
	btfsc	status,z
	goto	backlight_off
	goto	backlight_on

reset_cmd
	movfw	input_byte
	iorlw	0
	btfsc	status,z
	goto	lcd_active
	goto	lcd_inactive

loop_cmd
	movlw	0xaa
	call	lcd_write_cmd
	movlw	0x55
	call	lcd_write_data
	goto	loop_cmd
		

;
;
;
matrix_setup
	return

show_hex
	call	hex_table
	clrf	pclath
	call	uart_putc
	return

show_byte
	movwf	show_data
	movwf	temp
	rrf		temp,f
	rrf		temp,f
	rrf		temp,f
	rrf		temp,f
	movfw	temp
	andlw	0x0f
	call	show_hex
	
	movfw	show_data
	andlw	0x0f
	call	show_hex
	return
					
show_matrix_data
	movlw	'b'
	call	uart_putc
	movfw	portb
	call	show_byte
	movlw	' '
	call	uart_putc

	movlw	'c'
	call	uart_putc
	movfw	portc
	call	show_byte
	movlw	' '
	call	uart_putc

	movlw	'l'
	call	uart_putc
	movfw	matrix_data
	call	show_byte

	call	crlf

	return

matrix_delay
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	return				
		
;debug
matrix_test
	clrf	latch2_data
	call	set_latch2
	call	unset_latch2

	movfw	input_byte
	movwf	latch2_data
	call	write_latch2

	call	reset_databus		; make bus input to read touchpad

	call	matrix_delay
	call	matrix_delay
	call	matrix_delay
	call	matrix_delay

	movfw	porta				; read ra0-5
	andlw	0x3f
	movwf	matrix_data
	call	show_matrix_data
		
	return

scan_table:
	movwf	temp2
	movlw	high(scan_table)
	movwf	pclath
	movfw	temp2

	addwf	pcl,f
	dt	~0x40,~0x20,~0x10,~0x08,~0x04,~0x02,~0x01,0
;	dt	~0x01,~0x02,~0x04,~0x08,~0x10,~0x20,~0x40,0
;	dt	 0xfe, 0xfd, 0xfb, 0xf7, 0xef, 0xdf, 0xbf,0
;	dt	 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01,0
;	dt	 0xfe, 0xfc, 0xf8, 0xf0, 0xe0, 0xc0, 0x80,0

;
; touch matrix scanner
;
; Start by setting all 373 #2 outputs high
; Scan by setting one 373 #2 output low
; Check inputs - if any low, you've got a hit
;
matrix_scan
	clrf	matrix_row

matrix_loop
	movfw	matrix_row
	call	scan_table			; get next value for latch2
	clrf	pclath
	iorlw	0
	btfsc	status,z			; value zero?
	goto	matrix_done

	movwf	latch2_data
	call	set_latch2			; set latch2 (one bit will be 0)
	
	call	reset_databus		; make bus input to read touchpad
;	call	turn_databus	

	call	matrix_delay
	call	matrix_delay
				
	movfw	porta				; read ra0-5
	andlw	0x3f
	movwf	matrix_data

	btfsc	portc,databus6_rc
	bsf		matrix_data,6	
	btfsc	portc,databus7_rc
	bsf		matrix_data,7

;	call	show_matrix_data

	call	unset_latch2		; set latch2 (one bit will be 0)


	movfw	matrix_data
	sublw	0xd0
	btfsc	status,z			; key down?
	goto	matrix_next

	btfss	matrix_data,0
	movlw	1
	btfss	matrix_data,1
	movlw	2
	btfss	matrix_data,2
	movlw	3
	btfss	matrix_data,3
	movlw	4
	btfss	matrix_data,4
	movlw	5
	btfss	matrix_data,5
	movlw	6
	btfss	matrix_data,6
	movlw	7
	movwf	matrix_col

	movlw	0x01
	return
		
matrix_next
	incf	matrix_row,f

	goto	matrix_loop

; done
matrix_done
	movlw	0x00
	movwf	latch2_data
	call	write_latch2		; set CN4 1-7 high

	retlw	0

;------------------------------------------------------------------------------

;
; setup uart tx & rx ring
;
ring_setup:
	clrf	rx_ring_cnt		; reset rx ring
	movlw	rx_ring
	movwf	rx_ring_ptr
	movwf	rx_ring_get
	
	clrf	tx_ring_cnt		; reset tx ring
	movlw	tx_ring
	movwf	tx_ring_ptr
	movwf	tx_ring_put

	return
		
;
; init uart
;
uart_setup:

	clrf	status
	bsf		status,rp0		; page 1
	
	movlw	b9600
	movwf	spbrg			; set baud rate

	bcf		status,rp0		; page 0
	movlw	(1<<SPEN) | (1<<CREN)
	movwf	rcsta			; enable receiver

	movf	rcreg,w
	bsf		intcon,GIE
	bsf		intcon,PEIE

	bsf		status,rp0		; page 1
	bsf		txsta,TXEN		; enable transmitter

	bsf		pie1,RCIE		; enable receiver interrupt

	bcf		status,rp0		; page 0
	
;	bsf intcon, PEIE
	
	return

;
; uart rx interrupt service
;
uart_rx_irq:
	movf	rx_ring_cnt,w		; buffer full?
	xorlw	RX_RING_SIZE
	btfsc	status,z
	goto	uart_rx_full		; yes, drop chars
	
	movf	rx_ring_ptr,w		; get ring ptr
	movwf	fsr
	movf	rcreg,w
	movwf	indf				; save char in ring
	
	incf	rx_ring_cnt,f		; incr count
		
	incf	rx_ring_ptr,f		; advance ring ptr
	movf	rx_ring_ptr,w
	xorlw	rx_ring+RX_RING_SIZE
	movlw	rx_ring
	btfsc	status,z			; at end?
	movwf	rx_ring_ptr			; reset to top if at end
	
	return

uart_rx_full:
	movf	rcreg,w				; empty buffer, ignore
	return

;
;
rx_fifo_empty
	movf	rx_ring_cnt,f
	return

;
; get char from input ring
;
rx_fifo_get:
	movf	rx_ring_cnt,f
	btfsc	status,Z
	goto	rx_fifo_get			; loop

	movf	rx_ring_get,w
	movwf	fsr

	incf	rx_ring_get,1		; advance ring ptr
	movf	rx_ring_get,w
	xorlw	rx_ring+RX_RING_SIZE
	movlw	rx_ring
	btfsc	status,z			; at end?
	movwf	rx_ring_get			; reset to top if at end

	movf	indf,w				; get char
	decf	rx_ring_cnt,1		; decr count

	return

;
; uart tx interrupt service
;
uart_tx_irq:
	movf	tx_ring_cnt,1		; empty?
	btfsc	status,z
	goto	uart_tx_empty
	
	movf	tx_ring_ptr,w		; no, get next char
	movwf	fsr
	movf	indf,w
	movwf	txreg				; transmit
	
	decf	tx_ring_cnt,1		; decr count
	
	incf	tx_ring_ptr,1		; advance ring ptr
	movf	tx_ring_ptr,w
	xorlw	tx_ring+TX_RING_SIZE
	movlw	tx_ring
	btfsc	status,z			; at end?
	movwf	tx_ring_ptr			; reset to top if at end

	return

uart_tx_empty:
	movlw	pie1
	movwf	fsr
	bcf		indf,TXIE		; turn off xmit interrupt
	return
	
;
; output a single char to uart
;
uart_putc
;	btfsc	tx_ring_ptr,0	; fifo in use?
;	goto	push_chr
putc_loop
	btfsc	pir1,TXIF		; xmit busy?
	goto	putc_chr
	bsf		status,rp0		; page 1
	btfss	txsta,TRMT		; shift reg empty?
	goto	retest_tsr
	bcf		status,rp0		; page 0
	goto 	putc_chr
;
retest_tsr
	bcf		status,rp0
	goto	putc_loop
;
putc_chr
	movwf	txreg			; send char
	return

;
;
; 
puts
	movwf	temp
	
puts_loop
	movfw	temp
	call	msg_table
	clrf	pclath
	xorlw	0
	btfsc	status,z
	return
	call	uart_putc
	incf	temp,f
	goto	puts_loop

	org 0x300
hex_table
	movwf	temp2
	movlw	high(hex_table)
	movwf	pclath
	movfw	temp2
	addwf	pcl,f
	dt		"0123456789abcdef"
	
msg_table
	movwf	temp2
	movlw	high(msg_table)
	movwf	pclath
	movfw	temp2
	addwf	pcl,f
prompt	equ	$-(msg_table+5)
	dt		13,10,"reset",13,10,0

; ----------------------------------------------------------------------------

;
;
;
eeprom_setup
	return

;
; make a short delay for the beeper
;
buzz_delay
	movlw	255
	movwf	temp2
bdl
	decfsz	temp2,f
	goto	bdl
	return

; 
; make a big delay
; 
big_delay
	movlw	255
	movwf	temp
bd1
	call	buzz_delay
	decfsz	temp,f
	goto	bd1
	return

lcd_reset_delay
	movlw	100
	movwf	delaycnt
lrd1
	call	big_delay
	decfsz	delaycnt,f
	goto	lrd1

	return

;		
; beep the beeper
;		
buzz
	movlw	255
	movwf	temp
	
bl1
	bsf		latch1_data,buzz_out
	call	write_latch1
	call	buzz_delay	
	bcf		latch1_data,buzz_out
	call	write_latch1
	call	buzz_delay	

	decfsz	temp,f
	goto	bl1
	return

	end
