/*
 * info.c
 *
 * answer telnet requests to a local port for "info" on the server
 *
 * $Id$
 */

#include <stdio.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "list.h"
#include "log.h"

#define DEFAULT_LOCAL_PORT 22222

extern int local_ip;
static int local_port;

char *
ipport_text(unsigned long nip, unsigned nport)
{
        unsigned long ip = ntohl(nip);
        unsigned short p = ntohs(nport);
        static char s[64];
        sprintf(s, "%d.%d.%d.%d:%-5u",
               (ip >> 24) & 0xff,
               (ip >> 16) & 0xff,
               (ip >>  8) & 0xff,
               (ip      ) & 0xff,
               ntohs(p));
        return s;
}

/* use this to printf to a cli session */
int
info_printf(int fd, char *fmt, ...)
{
    char string[512];
    va_list ap;
    
    va_start(ap, fmt);
    vsprintf(string, fmt, ap);
    va_end(ap);

    write(fd, string, strlen(string));
    return 0;
}


void
info_dump(int fd)
{
    info_printf(fd, "telnet info:\r\n");

    telnet_dump(fd);
}

/*
 * after a socket has indicated it has half a connection, call accept()
 * to make a new socket and create the second half of the connection.
 */
int
info_accept(int listenfd, int *newfd)
{
    struct sockaddr_in in_addr;
    int len, fd;
    char text[32];
    
    debugf(DBG_LOW, "info_accept()\n");

    /* accept the new tcp connection */
    len = sizeof(in_addr);
    if ((fd = accept(listenfd, (struct sockaddr *)&in_addr, &len)) < 0)
    {
        perror("accept(listenfd)");
        return -1;
    }

    debugf(DBG_LOW, "info_accept() len %d, fd %d\n", len, fd,
           sockaddr_in_to_text(&in_addr, text));
	
    *newfd = fd;
    return 0;
}

/*
 * accept a new connection, create a new info and
 * connect the new info to new fd
 *
 * called by fd code when socket gets a tcp connection request
 */
int
info_accept_new_connection(int fd)
{
    int new_fd;
    void *cli;
    
    debugf(DBG_LOW, "info_accept_new_connection(fd=%d)\n", fd);
    
    /* do actual accept and make a new fd */
    if (info_accept(fd, &new_fd) != -1) {

        info_dump(new_fd);
        close(new_fd);
    }

    return 0;
}

/*
 * create a new tcp socket and set it up to listen for connections
 */
int
info_listen(int *newfd)
{
    struct sockaddr_in in_addr;
    int fd;

    /* create a IP socket */
    if ((fd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
        perror("socket(PF_INET, SOCK_STREAM)");
        return -1;
    }

#if 1
    {
        int one = 1;
        if ((setsockopt(fd, SOL_SOCKET, SO_REUSEADDR,
                        &one, sizeof(one))) == -1)
        {
            perror("setsockopt(PF_INET, SOCK_STREAM)");
            return -1;
        }
    }
#endif

    memset(&in_addr, 0, sizeof(in_addr));
    in_addr.sin_family = AF_INET;
    in_addr.sin_addr.s_addr = local_ip ? htonl(local_ip) : htonl(INADDR_ANY);
    in_addr.sin_port = htons(local_port);

    debugf(DBG_INFO, "bind to port %d\n", local_port);
        
    if (bind(fd, (struct sockaddr *)&in_addr, sizeof(in_addr)) < 0) {
        perror("bind(PF_INET, SOCK_STREAM)");
        return -1;
    }
    
    if (listen(fd, 5) < 0) {
	    perror("listen(..., SOCK_STREAM)");
	    return -1;
    }

    *newfd = fd;
    return 0;
}

int
info_init_sockets(void)
{
    int fd, new_fd;

    if (local_port == 0)
        local_port = DEFAULT_LOCAL_PORT;

    /* create a listening socket */
    if (info_listen(&fd)) {
	return -1;
    }

    /* tell socket code to call us back when fd gets a connection */
    fd_add_listen(fd, info_accept_new_connection);

    return 0;
}

int
info_init()
{
    if (info_init_sockets())
        return -1;

    return 0;
}



/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
