/*
 * log.c
 *
 * support for syslog and debug logging
 *
 * Brad Parker <brad@@heeltoe.com>
 */

#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <syslog.h>
#include <time.h> 

#include "list.h"
#include "server.h"
#include "log.h"

#define LOG_IDENT "cipeserver"

extern int flag_daemon;
extern int flag_debug_level;
extern int flag_trace_level;
extern int flag_debug_time;
extern int flag_debug_abs_time;

int
log_init(void)
{
    if (flag_daemon) {
        openlog(LOG_IDENT, LOG_CONS, LOG_DAEMON);
    }
    return 0;
}

void
log(int level, char *fmt, ...)
{
    char string[512];
    va_list ap;
    
    va_start(ap, fmt);
    vsprintf(string, fmt, ap);
    va_end(ap);

    if (flag_daemon) {
        int syslog_level = LOG_DAEMON;

        switch (level) {
        case DBG_PANIC: syslog_level |= LOG_EMERG; break;
        case DBG_WARN: syslog_level |= LOG_WARNING; break;
        case DBG_INFO: syslog_level |= LOG_INFO; break;
        case DBG_MED:
        case DBG_LOW: syslog_level |= LOG_DEBUG; break;
        }
        syslog(syslog_level, "%s", string);
    } else {
        printf("log: %s\n", string);
    }
}

int
log_shutdown(void)
{
    if (flag_daemon) {
        closelog();
    }
    return 0;
}

void
debugf(int level, char *fmt, ...)
{
    char string[512], intro[64], *tail, *head;
    va_list ap;
    int len;
    int perror_no = 0;
    const char *perror;

    if (DBG_LEVEL(level) > flag_debug_level && DBG_LEVEL(level) != DBG_PANIC)
        return;

    /* if system error, note number and string */
    if (level & DBG_ERRNO) {
        perror_no = errno;
        perror = strerror(errno);
    }

    va_start(ap, fmt);
    vsprintf(string, fmt, ap);
    va_end(ap);

    len = strlen(string);
    tail = string[len-1] == '\n' ? "" : "\n";

    switch (DBG_LEVEL(level)) {
    case DBG_LOW:  head = "      "; break;
    case DBG_MED:  head = "    "; break;
    case DBG_INFO: head = "  "; break;
    case DBG_WARN: head = ""; break;
    case DBG_PANIC: head = ""; break;
    }

    strcpy(intro, "debug:");

    if (flag_debug_time) {
        if (flag_debug_abs_time) {
            /* absolute time */
            time_t t;
            struct tm *tm;
            t = time(NULL);
            tm = localtime(&t);
            sprintf(intro, "%02d:%02d:%02d debug:",
                    tm->tm_hour, tm->tm_min, tm->tm_sec);
        } else {
            /* relative time */
            static struct timeval tv_last;
            struct timeval tv1, tv2;
            gettimeofday(&tv1, NULL);
            if (tv_last.tv_sec == 0) {
                tv2.tv_sec = tv2.tv_usec = 0;
            } else {
                if (tv_last.tv_usec <= tv1.tv_usec) {
                    tv2.tv_usec = tv1.tv_usec - tv_last.tv_usec;
                    tv2.tv_sec = tv1.tv_sec - tv_last.tv_sec;
                } else {
                    tv2.tv_usec = (tv1.tv_usec + 1 * 1000 * 1000)
                        - tv_last.tv_usec;
                    tv2.tv_sec = (tv1.tv_sec - 1) - tv_last.tv_sec;
                }
            }
            sprintf(intro, "%4d:%06d debug:",
                    tv2.tv_sec, tv2.tv_usec);
            tv_last = tv1;
        }
    }

    if (!flag_daemon) {
        printf("%s %s%s%s", intro, head, string, tail);
        if (perror_no) {
            printf("%s errno %d: %s\n", intro, perror_no, perror);
        }
    }

    if (flag_daemon) {
        if (level <= DBG_WARN) {
            syslog(LOG_DAEMON | LOG_WARNING, "debug: %s", string);
            if (perror_no) {
                syslog(LOG_DAEMON | LOG_WARNING, "debug: errno %d: %s\n",
                       perror_no, perror);
            }
        }
    }

#if 0 //XXX DEBUG
    if (DBG_LEVEL(level) == DBG_PANIC) {
        while (1)
            ;
    }
#endif
}

void
tracef(int level, char *fmt, ...)
{
    char string[512], intro[64], *tail, *head;
    va_list ap;
    int len;
    int perror_no = 0;
    const char *perror;

    if (TRACE_LEVEL(level) > flag_trace_level)
        return;

    /* if system error, note number and string */
    if (level & TRACE_ERRNO) {
        perror_no = errno;
        perror = strerror(errno);
    }

    va_start(ap, fmt);
    vsprintf(string, fmt, ap);
    va_end(ap);

    len = strlen(string);
    tail = string[len-1] == '\n' ? "" : "\n";

    switch (TRACE_LEVEL(level)) {
    case TRACE_LOW:  head = "    "; break;
    case TRACE_MED: head = "  "; break;
    case TRACE_HIGH: head = ""; break;
    case TRACE_ALL: head = ""; break;
    }

    strcpy(intro, "trace:");

    if (!flag_daemon) {
        printf("%s %s%s%s", intro, head, string, tail);
        if (perror_no) {
            printf("%s errno %d: %s\n", intro, perror_no, perror);
        }
    }

    if (flag_daemon) {
        if (level <= TRACE_HIGH) {
            syslog(LOG_DAEMON | LOG_WARNING, "trace: %s", string);
            if (perror_no) {
                syslog(LOG_DAEMON | LOG_WARNING, "trace: errno %d: %s\n",
                       perror_no, perror);
            }
        }
    }
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
