/*
 * output.c
 *
 * output scheduler
 *
 * $Id$
 */

#include <stdio.h>
#include <assert.h>
#include <sys/time.h>

#include "log.h"
#include "list.h"

/*
 * pending is output which is not due yet
 * active is output which is due, but has multiple bytes to delay
 * free is free buffers
 *
 * lines of output sit on pending queue until they expire, then they move
 * to the active queue.  
 * 
 * output on the active queue is spit out at the delay rate.
 *
 * when output is done it is put on the free queue
 * 
 */

struct list_head o_pending;
struct list_head o_active;
struct list_head o_free;

#define MAX_BUFFERED 1024
struct out_s {
	struct list_element link;
	int fd;
	int line_due;
	int ch_due;
	int ch_delay;
	int count;
	char *ptr;
	char buffer[MAX_BUFFERED];
};

int o_active_fd[1024];

static struct timeval last_tv;
static int skip_delay;

void
output_service_active(int delta)
{
	struct list_element *l, *next;

	for (l = list_head(&o_active); l; l = next) {
		struct out_s *o = (struct out_s *)l;
		int count;

		next = list_next(l);

		if (0) debugf(DBG_MED,
			      "output_service_active(delta=%d) "
			      "%p active, count %d, delay %d\n",
			      delta, l, o->count, o->ch_due);


		if (o->ch_due > delta) {
			o->ch_due -= delta;
			continue;
		}

		/* figure out how many bytes go out this quantum */
		if (o->ch_delay) {

			count = (delta + o->ch_delay-1) / o->ch_delay;
			if (count == 0) count = 1;

			if (count > o->count)
				count = o->count;
		} else
			count = o->count;

		if (0) debugf(DBG_LOW,
			      "output_service_active(delta=%d) "
			      "%p active, count %d, left %d, delay %d\n",
			      delta, l, count, o->count, o->ch_delay);

		if (count > 0) {
			write(o->fd, o->ptr, count);
			o->ptr += count;
			o->count -= count;
		}

		if (o->count == 0) {
			debugf(DBG_LOW, "output_service_active() %p done\n",l);

			list_remove(&o_active, l);
			list_add(&o_free, l);

			o_active_fd[o->fd]--;
		} else {
			o->ch_due += o->ch_delay;
		}
	}
}

void
output_dump()
{
	struct list_element *l, *next;

	debugf(DBG_MED, "o_pending:\n");
	for (l = list_head(&o_pending); l; l = list_next(l)) {
		struct out_s *o = (struct out_s *)l;
		debugf(DBG_MED, "%p: due %d, delay %d\n",
		       o, o->line_due, o->ch_delay);
	}

	debugf(DBG_MED, "o_active:\n");
	for (l = list_head(&o_active); l; l = list_next(l)) {
		struct out_s *o = (struct out_s *)l;
		debugf(DBG_MED, "%p: due %d, due %d, delay %d\n",
		       o, o->line_due, o->ch_due, o->ch_delay);
	}
}

void
output_check_pending(int delta)
{
	struct list_element *l, *next;

#if 1
	for (l = list_head(&o_pending); l; l = next) {
		struct out_s *o = (struct out_s *)l;

		if (o->line_due > delta)
			o->line_due -= delta;
		else
			o->line_due = 0;

		next = list_next(l);

		if (o->line_due == 0 && o_active_fd[o->fd] == 0) {
			debugf(DBG_LOW,
			       "output_check_pending() %p line_due\n", l);

			o_active_fd[o->fd]++;

			o->ch_due = o->ch_delay;

			list_remove(&o_pending, l);
			list_add(&o_active, l);

			if (0) output_dump();
		}
	}
#else

	if ((l = list_head(&o_pending))) {
		struct out_s *o = (struct out_s *)l;

		if (o->line_due > delta)
			o->line_due -= delta;
		else
			o->line_due = 0;

		if (o->line_due == 0 && o_active_fd[o->fd] == 0) {
			debugf(DBG_LOW,
			       "output_check_pending(delta=%d) %p line_due\n",
			       delta, l);

			o_active_fd[o->fd]++;

			o->ch_due = o->ch_delay;

			list_remove(&o_pending, l);
			list_add(&o_active, l);
		}
	}
#endif
}

void
output_insert(struct list_element *new)
{
	struct list_element *l, *next;

	for (l = list_head(&o_pending); l; l = list_next(l)) {
		struct out_s *o = (struct out_s *)l;
		struct out_s *onew = (struct out_s *)new;

		if (o->line_due > onew->line_due) {
			debugf(DBG_LOW, "output_insert() %p before %p\n",
			       new, l);

			list_insert_before(l, new);
			return;
		}
	}

	debugf(DBG_LOW, "output_insert() added to end %p\n", new);
	list_add(&o_pending, new);

	if (0) output_dump();

// not sure I want to do this yet...
//	skip_delay++;
}

void
output_add_pending(int fd, int l_delay, int ch_delay, char *buffer, int count)
{
	struct list_element *first;
	struct out_s *o;

#if 0
	write(fd, buffer, count);
	return;
#endif

	if (!list_empty(&o_free)) {
		first = list_head(&o_free);
		list_remove(&o_free, first);

		o = (struct out_s *)first;

		debugf(DBG_LOW, "output_add_pending() from free %p\n", o);
	} else {
		o = (struct out_s *)malloc(sizeof(struct out_s));
		if (o == NULL)
			return;

		first = (struct list_element *)o;
		o->link.prev = 0;
		o->link.next = 0;

		debugf(DBG_LOW, "output_add_pending() new %p\n", o);
	}
	
	if (count > MAX_BUFFERED)
		count = MAX_BUFFERED;

	o->fd = fd;
	o->line_due = l_delay;
	o->ch_delay = ch_delay;
	o->count = count;
	o->ptr = o->buffer;
	memcpy(o->buffer, buffer, count);

	output_insert(first);
}

void
output_service(void)
{
	int delta_ms;
	struct timeval tv;
	int sec;
	int usec;

	gettimeofday(&tv, NULL);

	if (last_tv.tv_sec == 0) {
		last_tv = tv;
		return;
	}

	sec = tv.tv_sec - last_tv.tv_sec;
	usec = tv.tv_usec - last_tv.tv_usec;

	if (usec < 0) {
		sec -= 1;
		usec += 1000000;
	}

	delta_ms = usec / 1000;
	if (delta_ms <= 0 && !skip_delay)
		return;

	skip_delay = 0;
	last_tv = tv;

//debugf(DBG_MED, "delta_ms %d\n", delta_ms);

	output_service_active(delta_ms);
	output_check_pending(delta_ms);
}
