/*
 * params.c
 *
 * parse config file, store params and find them later
 *
 * $Id$
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "log.h"
#include "param.h"

#define MAX_PARAMS 10
#define MAX_IDS 1024

struct param_s {
    int count;
    struct {
        char *key;
        char *string;
        value_t value;
    } params[MAX_PARAMS];
};

struct param_s default_params;
struct param_s params[MAX_IDS];
int param_max_id;

int
param_get(int id, char *name, char **pstring, value_t *pvalue)
{
    if (pvalue) {
        pvalue->is_random = 0;
        pvalue->value = 0;
    }
    if (pstring)
        *pstring = NULL;

    if(id > MAX_IDS)
        return -1;

#if 0
    if (strcmp(name, "login") == 0)
        *pvalue = 1;
    if (strcmp(name, "negotiation") == 0)
        *pvalue = 1;

    if (strcmp(name, "echo-delay") == 0)
        *pvalue = 200;

    if (strcmp(name, "line-delay") == 0)
        *pvalue = 200;

    if (strcmp(name, "command-delay") == 0)
        *pvalue = 0;

    return 0;
#endif

    struct param_s *param = &params[id];
    int i;

    debugf(DBG_MED, "param_get() id %d, name %s\n", id, name);

    for (i = 0; i < param->count; i++) {
        if (strcmp(name, param->params[i].key) == 0) {
            if (pvalue)
                *pvalue = param->params[i].value;
            if (pstring)
                *pstring = param->params[i].string;
            debugf(DBG_MED, "found under id\n");
            return 0;
        }
    }

    /* no match, check defaults */
    for (i = 0; i < default_params.count; i++) {
        if (strcmp(name, default_params.params[i].key) == 0) {

            debugf(DBG_MED, "found default\n");

            if (pvalue) {
                *pvalue = default_params.params[i].value;
                debugf(DBG_MED, "found default; %s = %d\n", name, *pvalue);
            }
            if (pstring) {
                *pstring = default_params.params[i].string;
            }

            return 0;
        }
    }
    
    debugf(DBG_MED, "not found\n");
    return -1;
}

int
param_valid_user(int id, char *username, char *password)
{
    struct param_s *param;
    char *uname, *pw;
    int no_config = 0;

    if (param_get(id, "user", &uname, NULL))
        no_config++;

    if (param_get(id, "password", &pw, NULL))
        no_config++;

    if (no_config) {
        /* default username if no config */
        if (strcmp(username, "user") == 0 &&
            strcmp(password, "password") == 0)
            return 0;
    } else {
        /* config username */
        if (strcmp(username, uname) == 0 &&
            strcmp(password, pw) == 0)
            return 0;
    }

    return -1;
}

int
getnum(char **pp, int *pv)
{
    char *p = *pp;
    int v = 0;

    //debugf(DBG_MED, "getnum() %s\n", p);
    while (1) {
        int c = *p++;
        if (!isdigit(c))
            break;

        v = (v * 10) + (c - '0');
    }

    *pv = v;
    *pp = p;
    return 0;
}

int
skipwhite(char **pp)
{
    char *p = *pp;
    while (*p && (*p == ' ' || *p == '\t' || *p == ','))
        p++;

    if (*p == 0)
        return -1;

    *pp = p;
    return 0;
}

int
mustbe(char **pp, char ch)
{
    char *p = *pp;

    //debugf(DBG_MED, "mustbe want %c, got %c\n", ch, *p);
    if (*p && *p != ch)
        return -1;

    *pp = ++p;
    return 0;
}

/* parse a single word */
int
getword(char **pp, char *word)
{
    char *p = *pp;
    int len = 0;

    while (*p && (isalnum(*p) || *p == '-')) {
        *word++ = *p++;
        if (++len == 255)
            break;
    }

    *word = 0;

    *pp = p;
    return 0;
}

/* parse a <tag>=<value> pair */
int
gettag(char **pp, char *tag, char *value)
{
    if (skipwhite(pp))
        return -1;

    getword(pp, tag);
    //debugf(DBG_MED, "tag '%s'\n", tag);

    skipwhite(pp);
    if (mustbe(pp, '='))
        return -1;

    skipwhite(pp);
    getword(pp, value);
    //debugf(DBG_MED, "value '%s'\n", value);

    return 0;
}

int
param_file_parse(FILE *f)
{
    char line[1024], *p;

    while (fgets(line, sizeof(line), f)) {
        int len = strlen(line);
        int id;
        char tag[256], value[256];
        struct param_s *param;

        if (len > 0)
            line[--len] = 0;

        debugf(DBG_LOW, "len %d '%s'\n", len, line);

        if (line[0] == '#' || len == 0)
            continue;

        /* line starts with id (or '*' for default) */
        p = line;
        if (p[0] == '*') {
            id = -1;
            p++;
        } else {
            if (getnum(&p, &id)) {
                fprintf(stderr, "bad id\n");
                fprintf(stderr, "%s\n", line);
                return -1;
            }
        }

        if (id > MAX_PARAMS) {
            fprintf(stderr, "id %d exceeds compiled max %d\n",
                    id, MAX_PARAMS);
            return -1;
        }

        /* id '*' is the default */
        if (id == -1)
            param = &default_params;
        else {
            param = &params[id];
            if (id > param_max_id)
                param_max_id = id;
        }

        param->count = 0;

        /* collect the tag=value pairs */
        while (gettag(&p, tag, value) == 0) {
            int n = atoi(value);
            char *dash;

            param->params[param->count].key = strdup(tag);
            param->params[param->count].string = strdup(value);

            /* handle values like "10-20" */
            if ((dash = strchr(value, '-'))) {
                param->params[param->count].value.is_random = 1;
                param->params[param->count].value.min = n;
                param->params[param->count].value.max = atoi(dash+1);
                param->params[param->count].value.value = 0;

                debugf(DBG_MED, "id %d, tag=%s str=%s value=%d-%d\n",
                       id, tag, value,
                       param->params[param->count].value.min,
                       param->params[param->count].value.max);

            } else {
                param->params[param->count].value.is_random = 0;
                param->params[param->count].value.min = 0;
                param->params[param->count].value.max = 0;
                param->params[param->count].value.value = n;

                debugf(DBG_MED, "id %d, tag=%s str=%s value=%d\n",
                       id, tag, value, n);
            }


            param->count++;
            if (param->count >= MAX_PARAMS) {
                fprintf(stderr, "too many parameters\n");
                fprintf(stderr, "%s\n", line);
                return -1;
            }
        }
    }

    return 0;
}

/* return float between 0 and 1 */
double
rvalue(void)
{
    return ((double)random()) / (double)RAND_MAX;
}

int
pick_value(value_t *pvalue)
{
    if (!pvalue->is_random) {
        return pvalue->value;
    }

    return pvalue->min + ((pvalue->max - pvalue->min) * rvalue());
}

char *show_value(value_t *pvalue)
{
    static char b[128];

    if (!pvalue->is_random) {
        sprintf(b, "fixed-value %d", pvalue->value);
    } else {
        sprintf(b, "random-value from %d to %d", pvalue->min, pvalue->max);
    }

    return b;
}

/*
 * read in config file 
 */

int
param_init(char *filename)
{
    FILE *f;
    int ret;

    if (filename == NULL) {
        debugf(DBG_INFO, "no configuration file\n");
        return 0;
    }

    debugf(DBG_INFO, "configuration file '%s'\n", filename);

    f = fopen(filename, "r");
    if (f == NULL) {
        perror(filename);
        return -1;
    }

    ret = param_file_parse(f);
    fclose(f);

    if (ret)
        return -1;

    return 0;
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
