/*
 * server.c
 *
 * telnet test server
 * Brad Parker <brad@@heeltoe.com>
 *
 */

#include <stdio.h>
#include <fcntl.h>

#include <netdb.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <netinet/in.h>

#include <sys/ioctl.h>
#include <linux/if.h>

#include "list.h"
#include "server.h"
#include "log.h"

static char rcs_id[] = "$Id$";

int server_running;
int local_ip;
int local_port;
int server_count;

int flag_daemon;
int flag_debug_level;
int flag_trace_level;
int flag_debug_time;
int flag_debug_abs_time;

char *config_filename;

#define SERVER_VERSION 007

int server_version = SERVER_VERSION;

int
parse_ip(char *name)
{
    struct hostent *h;
    struct sockaddr_in a;
    struct in_addr *ia;
    char str[64];

    h = gethostbyname(name);
    if (h) {
        debugf(DBG_LOW, "name: %s\n", h->h_name);
        ia = (struct in_addr *)(h->h_addr_list[0]);

        memset((char *)&a, 0, sizeof(a));
        a.sin_addr.s_addr = ia->s_addr;
        debugf(DBG_LOW, "found %s\n", sockaddr_in_to_text(&a, str));
        return ntohl(ia->s_addr);
    }

    return -1;
}

int
server_init_telnet(void)
{
    if (telnet_init())
        return -1;

    return 0;
}

int
server_init_signals(void)
{
    if (signal_init())
        return -1;
    return 0;
}

int
server_init_syslog(void)
{
    if (log_init())
        return -1;

    return 0;
}

int
server_init_info(void)
{
    if (info_init())
        return -1;

    return 0;
}

int
server_shutdown(void)
{
    log_shutdown();
    server_running = 0;
    return 0;
}

int
server_init(void)
{
    debugf(DBG_LOW, "server_init()\n");

    if (server_init_telnet())
        return-1;

    if (server_init_signals())
        return-1;

    if (server_init_syslog())
        return-1;

    if (server_init_info())
        return-1;

    debugf(DBG_LOW, "server_init() done\n");

    server_running = 1;
    return 0;
}

int
server(void)
{
    /* init server machinery */
    if (server_init())
        return -1;

    debugf(DBG_LOW, "starting server loop\n");

    /* be a server... */
    while (server_running) {
        fd_poll();
        signal_poll();
        output_service();
    }

    return 0;
}

char *
server_get_version_text(void)
{
    static char buf[64];
    sprintf(buf, "v%d.%d",
            server_version / 100, server_version % 100);
    return buf;
}

/*
 * fork the server process and set up to be a good background daemon
 *
 * disconnect from terminal, etc...
 */
int
daemonize(char *what)
{
    int r;

    chdir("/tmp");

    if ((r = fork()) > 0)
        exit(0);

    if (r == -1) {
	fprintf(stderr,"%s: unable to fork new process\n", what);
	perror("fork");
	exit(1);
    }

    /* close stdin, out, err */
    close(0);
    close(1);
    close(2);

    /* and reopen as bit buckets */
    open("/dev/null", O_RDONLY);
    open("/dev/null", O_WRONLY);
    open("/dev/null", O_WRONLY);

    log(5, "TELNET test server startin, %s", server_get_version_text());

    debugf(DBG_WARN, "TELNET test server starting");
    debugf(DBG_WARN, "%s", rcs_id);

    return 0;
}

void
usage(void)
{
    fprintf(stderr, "TELNET test server %s\n", server_get_version_text());

    fprintf(stderr, "usage:\n");
    fprintf(stderr, "-c <config-filename>  name of config file\n");
    fprintf(stderr, "-a <local-address>    local address of server\n");
    fprintf(stderr, "-p <local-port>       local port # of server\n");
    fprintf(stderr, "-s                    run as a background daemon\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "-d        increment debug level\n");
    fprintf(stderr, "-t        increment trace level\n");
    fprintf(stderr, "-D<level> set debug level\n");
    fprintf(stderr, "-T<level> set trace level\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "-l        show timestamps in log messages\n");
    fprintf(stderr, "-r        show timestamps as absolute time\n");

    exit(1);
}

extern char *optarg;

int
main(int argc, char **argv)
{
    int c;

    flag_debug_abs_time = 1;

    while ((c = getopt(argc, argv, "a:c:dD:p:stT:lr")) != -1) {
        switch (c) {
        case 'a':
            local_ip = parse_ip(optarg);
            break;
        case 'p':
            local_port = atoi(optarg);
            break;
        case 'c':
            config_filename = strdup(optarg);
            break;
        case 's':
            flag_daemon++;
            break;
        case 'd':
            flag_debug_level++;
            break;
        case 'l':
            flag_debug_time++;
            break;
        case 'r':
            flag_debug_abs_time = 0;
            break;
        case 't':
            flag_trace_level++;
            break;
        case 'T':
            flag_trace_level = atoi(optarg);
            break;
        case 'D':
            flag_debug_level = atoi(optarg);
            break;

        default:
            usage();
        }
    }

    if (param_init(config_filename))
        exit(1);

    if (flag_daemon) {
        daemonize(argv[0]);
    }

    server();
    
    log(5, "TELNET test server exiting\n");

    exit(0);
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
