/*
 * telnet.c
 *
 * basic tcp socket support code
 *
 * $Id$
 */

#include <stdio.h>
#include <fcntl.h>
#include <time.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/udp.h>

#include "list.h"
#include "param.h"
#include "server.h"
#include "log.h"
#include "telnet.h"

int telnet_count;
struct list_head telnet_list;
struct timeval wall_time;

extern int local_ip;
extern int local_port;


void
mark_time()
{
    gettimeofday(&wall_time, NULL);
}

int
get_marked_time()
{
    return wall_time.tv_sec;
}

/* convert a struct sockaddr_in to an ascii IP address */
char *
sockaddr_in_to_text(struct sockaddr_in *in, char *text)
{
    uint32_t ipa;
    
    text[0] = 0;
    ipa = in->sin_addr.s_addr;

    sprintf(text, "%d.%d.%d.%d",
            (ipa >> 24) & 0xff,
            (ipa >> 16) & 0xff,
            (ipa >>  8) & 0xff,
            (ipa      ) & 0xff);
    
    return text;
}

/*
 * make a new telnet object
 *
 * each telnet session creates a new telnet object
 */
int
telnet_new(void **ptelnet)
{
    struct telnet_s *s;

    s = (struct telnet_s *)malloc(sizeof(struct telnet_s));
    if (s == NULL)
        return -1;

    memset((char *)s, 0, sizeof(struct telnet_s));

    telnet_count++;
    *ptelnet = (void *)s;
    s->session_id = telnet_count;

    list_add(&telnet_list, (struct list_element *)s);

    return 0;
}

int
telnet_destroy(void *telnet)
{
    list_remove(&telnet_list, (struct list_element *)telnet);

    telnet_count--;
    free((char *)telnet);
}

int
telnet_set_socket(void *telnet, int fd)
{
    ((struct telnet_s *)telnet)->socket_fd = fd;
}

int
telnet_set_parameters(struct telnet_s *ts)
{
    param_get(ts->session_id, "login", NULL, &ts->cfg_login);
    param_get(ts->session_id, "negotiate", NULL, &ts->cfg_negotiation);
    param_get(ts->session_id, "echo-delay", NULL, &ts->echo_delay);
    param_get(ts->session_id, "command-delay", NULL, &ts->command_delay);
    param_get(ts->session_id, "line-delay", NULL, &ts->line_delay);

    ts->needs_login = pick_value(&ts->cfg_login);
    ts->needs_negotiation = pick_value(&ts->cfg_negotiation);

    if (ts->needs_negotiation)
        ts->iac_mode = 1;

    return 0;
}

int
telnet_set_source(struct telnet_s *ts, int ip, int port)
{
    ts->source_ip = ip;
    ts->source_port = port;
    return 0;
}

/*
 * callback routine called when an socket is closed down;
 * resets the state of the telnet object and erases the telnet
 */
int
telnet_close(int fd, int context)
{
    int ret, xfd;
    void *telnet;

    debugf(DBG_LOW, "telnet_close(fd=%d)\n", fd);
    
    /* don't need to close the fd - transport does this */
    
    if (fd_context_valid(context, &xfd, &telnet))
        return -1;

    telnet_destroy(telnet);
    
    return 0;
}

/*
 * after a socket has indicated it has half a connection, call accept()
 * to make a new socket and create the second half of the connection.
 */
int
telnet_accept(int listenfd, int *newfd, int *pip, int *pport)
{
    struct sockaddr_in in_addr;
    int len, fd;
    char text[32];
    
    debugf(DBG_LOW, "telnet_accept()\n");

    /* accept the new tcp connection */
    len = sizeof(in_addr);
    if ((fd = accept(listenfd, (struct sockaddr *)&in_addr, &len)) < 0)
    {
        perror("accept(listenfd)");
        return -1;
    }

    *pip = ntohl(in_addr.sin_addr.s_addr);
    *pport = ntohs(in_addr.sin_port);

    debugf(DBG_LOW, "telnet_accept() len %d, fd %d\n", len, fd,
           sockaddr_in_to_text(&in_addr, text));
	
    *newfd = fd;
    return 0;
}

/*
 * accept a new connection, create a new telnet and
 * connect the new telnet to new fd
 *
 * called by fd code when socket gets a tcp connection request
 */
int
telnet_accept_new_connection(int fd)
{
    int new_fd;
    void *ts;
    int ip, port;

    debugf(DBG_LOW, "telnet_accept_new_connection(fd=%d)\n", fd);
    
    /* do actual accept and make a new fd */
    if (telnet_accept(fd, &new_fd, &ip, &port) != -1) {

        if (telnet_new(&ts)) {
            close(new_fd);
            return -1;
        }

        telnet_set_parameters(ts);
        telnet_set_source(ts, ip, port);
        telnet_set_socket(ts, new_fd);

        telnet_prompt(ts);

        /* tell fd code to start reading the new socket */
        fd_add_reader(new_fd,
                      "telnet session",
                      telnet_stream_reader,
                      telnet_close,
                      (void *)ts);
    }

    return 0;
}

/* create a new socket and set it up to receive packets */
int
telnet_listen(int *newfd)
{
    struct sockaddr_in in_addr;
    int fd, len;
    char name[1024];

    if ((fd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
        perror("socket(PF_INET, SOCK_STREAM)");
        return -1;
    }

    debugf(DBG_INFO, "socket fd %d\n", fd);

#if 1
    {
        int one = 1;
        if ((setsockopt(fd, SOL_SOCKET, SO_REUSEADDR,
                        &one, sizeof(one))) == -1)
        {
            perror("setsockopt(PF_INET, SOCK_STREAM)");
            return -1;
        }
    }
#endif

    memset(&in_addr, 0, sizeof(in_addr));
    in_addr.sin_family = AF_INET;
    in_addr.sin_addr.s_addr = local_ip ? htonl(local_ip) : htonl(INADDR_ANY);
    in_addr.sin_port = htons(local_port);
    
    debugf(DBG_INFO, "bind to port %d\n", local_port);
        
    if (bind(fd, (struct sockaddr *)&in_addr, sizeof(in_addr)) < 0) {
        perror("bind(PF_INET, SOCK_STREAM)");
        return -1;
    }
    
    if (listen(fd, 5) < 0) {
	    perror("listen(..., SOCK_STREAM)");
	    return -1;
    }

    *newfd = fd;
    return 0;
}


int
telnet_init_sockets(void)
{
    int fd;

    if (local_port == 0) {
        local_port = DEFAULT_TCP_PORT;
    }

    if (telnet_listen(&fd))
        return -1;

    /* tell socket code to call us back when fd gets a connection */
    fd_add_listen(fd, "telnet listener", telnet_accept_new_connection);

    return 0;
}


int
telnet_init(void)
{
    if (telnet_init_sockets())
        return -1;

    return 0;
}

void
telnet_dump(int fd)
{
#if 0
    struct src_s *s;
    int i;

    info_printf(fd,
                "Addr                  Id             Last\t\trx\ttx\terr\n");

    for (i = 0; i < HASH_TAB_SIZE; i++) {
        for (s = (struct src_s *)list_head(&hashtable[i]);
             s;
             s = (struct src_s *)list_next(&s->link))
        {
            char b[64];

            sprintf(b, "%u(%u.%u)", s->id, s->id >> 8, s->id & 0xff);

            info_printf(fd,
                        "%-21s %-14s %s\t%u\t%u\t%u\n",
                        ipport_text(s->ipaddr, s->port),
                        b,
                        time_txt(s->last_time_heard),
                        s->rx_packets,
                        s->tx_packets,
                        s->rx_crc_errors);
        }
    }
#endif
    info_printf(fd, "telnet dump\n");
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
