/*
 * telnet_cmd.c
 *
 * command code for telnet command line interface
 *
 * $Id$
 */

#include <stdio.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "list.h"
#include "param.h"
#include "telnet.h"
#include "output.h"
#include "log.h"

int cmd_quit(struct telnet_s *telnet);
int cmd_test(struct telnet_s *telnet);
int cmd_id(struct telnet_s *telnet);
int cmd_show(struct telnet_s *telnet);
int cmd_type(struct telnet_s *telnet);
int cmd_help(struct telnet_s *telnet);

struct {
    char *cmd;
    int (*func)(struct telnet_s *);
    char *help;
} commands[] = {
    { "help", cmd_help, "help function" },
    { "id", cmd_id, "print session id" },
//    { "test", cmd_test, "test" },
    { "show", cmd_show, "show" },
    { "type", cmd_type, "type out a file" },
    { "quit", cmd_quit, "quit the shell" },
    { 0, 0 }
};

int
telnet_putc(struct telnet_s *telnet, int c)
{
    char cc = c;
#if 0
    write(telnet->socket_fd, &cc, 1);
#else
    output_add_pending(telnet->socket_fd,
                       0, pick_value(&telnet->echo_delay),
                       &cc, 1);
#endif
    return 0;
}

/* use this to printf to a telnet session */
int
telnet_printf(struct telnet_s *telnet, char *fmt, ...)
{
    char string[512];
    va_list ap;
    
    va_start(ap, fmt);
    vsprintf(string, fmt, ap);
    va_end(ap);

#if 0
    write(telnet->socket_fd, string, strlen(string));
#else
    output_add_pending(telnet->socket_fd,
                       pick_value(&telnet->line_delay),
		       pick_value(&telnet->echo_delay),
                       string, strlen(string));
#endif

    return 0;
}

void
beep(struct telnet_s *ts)
{
    telnet_putc(ts, 7);
}

/* add a null buffer with the command delay to simulate the command delay */
void
add_command_delay(struct telnet_s *ts)
{
    output_add_pending(ts->socket_fd, 0, pick_value(&ts->command_delay),
		       "", 0);
}


int cmd_quit(struct telnet_s *telnet)
{
    telnet_printf(telnet, "bye!\r\n");
    telnet->wants_to_quit = 1;
    return 0;
}


int cmd_show(struct telnet_s *telnet)
{
    return 0;
}

int cmd_type(struct telnet_s *telnet)
{
    FILE *f;

    if (telnet->argc < 2) {
        telnet_printf(telnet, "usage: type <filename>\r\n");
        return 1;
    }

    f = fopen(telnet->argv[1], "r");
    if (f != NULL) {
        char line[1024];
        while (fgets(line, sizeof(line), f)) {
            int len = strlen(line);
#if 0
            write(telnet->socket_fd, line, len);
#else
            line[len-1] = '\r';
            line[len++] = '\n';
            output_add_pending(telnet->socket_fd,
                               pick_value(&telnet->line_delay),
			       pick_value(&telnet->echo_delay),
                               line, len);
#endif
        }
        fclose(f);
    }

    return 0;
}

int cmd_id(struct telnet_s *telnet)
{
    telnet_printf(telnet, "session-id: %d\r\n", telnet->session_id);

    telnet_printf(telnet, "source: %d.%d.%d.%d:%u\r\n",
                  (telnet->source_ip >> 24) & 0xff,
                  (telnet->source_ip >> 16) & 0xff,
                  (telnet->source_ip >>  8) & 0xff,
                  (telnet->source_ip      ) & 0xff,
                  telnet->source_port);

    telnet_printf(telnet, "config: negotiate %s\r\n",
		  show_value(&telnet->cfg_negotiation));

    telnet_printf(telnet, "config: login %s\r\n",
		  show_value(&telnet->cfg_login));

    telnet_printf(telnet, "delay: echo %s\r\n",
		  show_value(&telnet->echo_delay));

    telnet_printf(telnet, "delay: line %s\r\n",
		  show_value(&telnet->line_delay));

    telnet_printf(telnet, "delay: command %s\r\n",
		  show_value(&telnet->command_delay));

    return 0;
}

#if 0
/* called when module returns data */
int
cmd_test_callback(struct telnet_s *telnet, struct module *m, int status)
{
    char data[16];
    int data_len;

    if (status != 0) {
        telnet_printf(telnet, "module receive timeout\r\n");
        return 1;
    }

    /* get the data */
    if (module_receive(m, &data, sizeof(data), &data_len)) {
        telnet_printf(telnet, "module send error\r\n");
        return 1;
    }

    telnet_printf(telnet, "module responds with %d bytes\r\n", data_len);

    return 0;
}
#endif

int cmd_test(struct telnet_s *telnet)
{
    struct module *m;
    char data[16];
    int data_len;
    int timeout;

    if (telnet->argc < 2) {
        telnet_printf(telnet, "usage: test <module-name>\r\n");
        return 1;
    }

#if 0
    if (module_find_name(&m, telnet->argv[1], NULL)) {
        telnet_printf(telnet, "can't find module '%s'\r\n");
        return 1;
    }

    if (module_send(m, data, sizeof(data))) {
        telnet_printf(telnet, "module send error\r\n");
        return 1;
    }

    /* mark this telnet as pending */
    telnet_set_pending(telnet);

    /* and set routine to call when data arrives */
    timeout = 5;

    telnet_set_receive_upcall(telnet, m, cmd_test_callback, timeout);
#endif

    return 0;
}

int cmd_help(struct telnet_s *telnet)
{
    int i;
    char buffer[256];

    telnet_printf(telnet, "Available commands:\r\n");

    for (i = 0; commands[i].cmd; i++) {
        telnet_printf(telnet, "%s\t%s\r\n", commands[i].cmd, commands[i].help);
    }

    return 0;
}

/*
 * find the first word of the input in the command list and run the
 * command function...
 */
int
telnet_parse_command(struct telnet_s *telnet)
{
    int i, hit;

    hit = 0;
    for (i = 0; commands[i].cmd; i++) {
        if (strncasecmp(commands[i].cmd, telnet->argv[0],
                        strlen(telnet->argv[0])) == 0)
        {
            hit++;
	    add_command_delay(telnet);

            (*commands[i].func)(telnet);
            break;
        }
    }

    if (!hit) {
        telnet_printf(telnet, "unknown command? '%s'\n\r", telnet->argv[0]);
    }

    return 0;
}



/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
