/*
 * telnet_neg.c
 *
 * $Id$
*/

#include <stdio.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "list.h"
#include "param.h"
#include "telnet.h"
#include "log.h"

#define DBG(...) if (1) debugf(DBG_MED, __VA_ARGS__)

char *telcmds[] = {
	"EOF", "SUSP", "ABORT", "EOR",
	"SE", "NOP", "DMARK", "BRK", "IP", "AO", "AYT", "EC",
	"EL", "GA", "SB", "WILL", "WONT", "DO", "DONT", "IAC", 0,
};

char *telopts[256] = {
	"BINARY", "ECHO", "RCP", "SUPPRESS GO AHEAD", "NAME",
	"STATUS", "TIMING MARK", "RCTE", "NAOL", "NAOP",
	"NAOCRD", "NAOHTS", "NAOHTD", "NAOFFD", "NAOVTS",
	"NAOVTD", "NAOLFD", "EXTEND ASCII", "LOGOUT", "BYTE MACRO",
	"DATA ENTRY TERMINAL", "SUPDUP", "SUPDUP OUTPUT",
	"SEND LOCATION", "TERMINAL TYPE", "END OF RECORD",
	"TACACS UID", "OUTPUT MARKING", "TTYLOC",
	"3270 REGIME", "X.3 PAD", "NAWS", "TSPEED", "LFLOW",
	"LINEMODE", "XDISPLOC", "OLD-ENVIRON", "AUTHENTICATION",
	"ENCRYPT", "NEW-ENVIRON",
	0,
};

char *
telnet_opt_text(u_char opt)
{
    char *on = telopts[opt];
    if (on)
        return on;
    return "unknown";
}

void
telnet_send3(struct telnet_s *ts, int c1, int c2, int c3)
{
    char buf[3];

    buf[0] = c1;
    buf[1] = c2;
    buf[2] = c3;
    write(ts->socket_fd, buf, 3);
}

void
telnet_put_do(struct telnet_s *ts, u_char what)
{
    DBG("->DO(%2x,%s)\n", what, telnet_opt_text(what));
    telnet_send3(ts, TELNET_IAC, TELNET_DO, what);
}

void
telnet_put_dont(struct telnet_s *ts, u_char what)
{
    DBG("->DONT(%2x,%s)\n", what, telnet_opt_text(what));
    telnet_send3(ts, TELNET_IAC, TELNET_DONT, what);
    ts->iac_dont[what]++;
}

void
telnet_put_will(struct telnet_s *ts, u_char what)
{
    DBG("->WILL(%2x,%s)\n", what, telnet_opt_text(what));
    telnet_send3(ts, TELNET_IAC, TELNET_WILL, what);
    ts->iac_will[what]++;
}

void
telnet_put_wont(struct telnet_s *ts, u_char what)
{
    DBG("->WONT(%2x,%s)\n", what, telnet_opt_text(what));
    telnet_send3(ts, TELNET_IAC, TELNET_WONT, what);
    ts->iac_wont[what]++;
}

void
telnet_send_subneg(struct telnet_s *ts )
{
    DBG("telnet_send_subneg()\n");

    /* SE? */
    if (ts->iac_cmd == 240) {
        /* yes, reset subnegotiation */
        ts->iac_flags[1] = 0;
        return;
    }
}

void
telnet_send_response(struct telnet_s *ts)
{
    DBG("telnet_send_response()\n");

    switch (ts->iac_cmd) {
    case 250:	/* SB? */
        ts->iac_flags[1] = 1;
        break;

    case TELNET_DO:
        DBG("<-DO(%2x,%s)\n", ts->iac_arg, telnet_opt_text(ts->iac_arg));

        if (ts->iac_will[ts->iac_arg] > 1)
            break;

        /* check list of options we want */
        switch (ts->iac_arg) {
	case TELOPT_ECHO:
	    telnet_put_will(ts, ts->iac_arg);
            ts->echo_mode = 1;
            break;
        default:
            /*telnet_put_wont(ts, ts->iac_arg);*/
            break;
        }
        break;

    case TELNET_WILL:
        DBG("<-WILL(%2x,%s)\n", ts->iac_arg, telnet_opt_text(ts->iac_arg));

        if (ts->iac_dont[ts->iac_arg] > 1)
            break;

        switch (ts->iac_arg) {
        default:
            telnet_put_dont(ts, ts->iac_arg);
            break;
        }
        break;

    case TELNET_WONT:
        DBG("<-WONT(%2x,%s)\n", ts->iac_arg, telnet_opt_text(ts->iac_arg));
        break;

    case TELNET_DONT:
        DBG("<-DONT(%2x,%s)\n", ts->iac_arg, telnet_opt_text(ts->iac_arg));

        if (ts->iac_wont[ts->iac_arg] > 1)
            break;

        switch (ts->iac_arg) {
	case TELOPT_SGA:
		telnet_put_wont(ts, ts->iac_arg);
		break;
        }
        break;

    default:
        break;
    }
}

int
telnet_iac_in(struct telnet_s *ts, int c)
{
    debugf(DBG_MED, "telnet_iac_in() iac %d, c %02x\n", ts->iac_mode, c);

    switch (ts->iac_mode) {
    case 0:
        /* saw IAC, set command state */
        ts->iac_mode = 2;

        /* have we sent initial options? */
        if (ts->iac_flags[0] == 0) {
            /* no */
            ts->iac_flags[0] = 1;

            /* Make the telnet client understand we will echo
             * characters so it should not do it locally. We don't
             * tell the client to run linemode, because we want to
             * handle line editing and tab completion and other stuff
             * that requires char-by-char support.
             */
            telnet_put_do(ts, TELOPT_ECHO);
            telnet_put_will(ts, TELOPT_ECHO);
            telnet_put_will(ts, TELOPT_SGA);
        }
        break;

    case 1:
        if (c == TELNET_IAC)
            ts->iac_mode = 2;
        break;

    case 2:
        ts->iac_cmd = c;
        ts->iac_mode = 3;
        break;

    case 3:
        ts->iac_arg = c;
        ts->iac_mode = 0;
        telnet_send_response(ts);
        break;
    }

    return 0;
}



/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
