/*
 * telnet_stream.c
 *
 * collect lines of input and parse them for commands
 *
 * $Id$
 */

#include <stdio.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "list.h"
#include "param.h"
#include "telnet.h"
#include "output.h"
#include "log.h"


/* print prompt on telnet session */
int
telnet_prompt(struct telnet_s *telnet)
{
    if (!telnet->has_prompt) {
        telnet_printf(telnet, "%s %s", TELNET_GREETING,
                      server_get_version_text());

        telnet->has_prompt++;
    }

    if (telnet->needs_login) {
        switch(telnet->login_state) {
        case 0:
            telnet_printf(telnet, "\r\n");
        case 1:
            if (telnet->login_tries >= 3) {
                telnet->wants_to_quit = 1;
                return -1;
            }

            telnet_printf(telnet, "%s", LOGIN_PROMPT);
            telnet->login_state = 1;
            break;
        case 2:
            telnet_printf(telnet, "%s", PASSW_PROMPT);
            break;
        }

        return 0;
    }

    telnet_printf(telnet, "\r\n%s", TELNET_PROMPT);

    return 0;
}

/*
 * mark session as pending so it won't get a prompt,
 * since we're waiting for some output to come back
 */
void
telnet_set_pending(struct telnet_s *telnet)
{
    telnet->pending = 1;
}

void
telnet_reset_pending(struct telnet_s *telnet)
{
    telnet->pending = 0;
    telnet->line_len = 0;
    telnet_prompt(telnet);
}

#if 0
/*
 * internal callback routine;
 * called by module code when a module responds with some data
 */
static int
telnet_module_callback(struct module *m, void *telnet_p, int status)
{
    struct telnet_s *telnet = (struct telnet_s *)telnet_p;
    int ret;

    if (telnet->callback == NULL)
        return -1;

    /* call our local telnet function */
    ret = (*telnet->callback)(telnet, telnet->callback_context, status);

    telnet_reset_receive_upcall(telnet);

    /* turn shell prompt back on */
    telnet_reset_pending(telnet);

    return ret;
}

/* inform the module code we want to be called back when a module responds */
int
telnet_set_receive_upcall(struct telnet_s *telnet, struct module *m,
                       int (*callback)(struct telnet_s *, struct module *, int),
                       int timeout)
{
    /* remember who to call back */
    telnet->callback = callback;
    telnet->callback_context = (void *)m;

    /* tell module code who to call */
    if (module_set_receive_upcall(m, telnet_module_callback,
                                  (void *)telnet, timeout))
    {
        return -1;
    }

    return 0;
}

int
telnet_reset_receive_upcall(struct telnet_s *telnet)
{
    telnet->callback = NULL;
    return 0;
}
#endif

/*
 * parse new input line, finding word boundaries and creating
 * an argv vector
 */
int
telnet_create_argv(struct telnet_s *telnet)
{
    char *p, c, t;

    debugf(DBG_LOW, "telnet_create_argv(telnet=%p)\n", telnet);

    strcpy(telnet->line2, telnet->line);

    p = telnet->line2;
    telnet->argc = 0;

    while (*p) {
        if (telnet->argc == MAX_ARGV) {
            debugf(DBG_WARN, "input exceeds max # of args (%d)", MAX_ARGV);
            debugf(DBG_WARN, "'%s'\n", telnet->line);
            break;
        }

        /* save start of word */
        telnet->argv[telnet->argc++] = p;
        telnet->argv[telnet->argc] = NULL;

        /* quoted string? */
        if (*p == '\'' || *p == '\"') {
            t = *p;

            /* adjust pointer to skip over starting quote */
            telnet->argv[telnet->argc-1]++;
            p++;

            while (c = *p) {
                if (c == t) {
                    *p++ = 0;
                    break;
                }
                p++;
            }
        } else {
            /* not quoted string, find end of word */
            while (c = *p) {

                if (c == ' ' || c == '\t') {
                    *p++ = 0;
                    break;
                }

                p++;
            }
        }

        /* skip over whitespace */
        while (c = *p) {
            if (c != ' ' && c != '\t')
                break;

            p++;
            continue;
        }
    }

    return 0;
}

int
telnet_parse_login(struct telnet_s *telnet)
{
    debugf(DBG_LOW, "telnet_parse_login(telnet=%p)\n", telnet);

    switch (telnet->login_state) {
    case 0:
        telnet->login_tries = 0;
    case 1:
        strcpy(telnet->user, telnet->line);
        telnet->login_state = 2;
        telnet->echo_mode_save = telnet->echo_mode;
        telnet->echo_mode = 0;
        break;
    case 2:
        if (param_valid_user(telnet->session_id,
                             telnet->user, telnet->line) == 0)
        {
            telnet->login_state = 0;
            telnet->needs_login = 0;
        } else {
            telnet_printf(telnet, "%s\r\n", PASSW_BAD);
            telnet->login_tries++;
            telnet->login_state = 1;
        }

        telnet->echo_mode = telnet->echo_mode_save;
        break;
    }

    return 0;
}

/*
 * turn the raw character line into words and then look for a known command
 */
int
telnet_parse_input(struct telnet_s *telnet)
{
    debugf(DBG_LOW, "telnet_parse_input(telnet=%p)\n", telnet);

    if (telnet_create_argv(telnet))
        return -1;

#if 1
    {
        int i;
        debugf(DBG_LOW, "argc %d\n", telnet->argc);
        for (i = 0; i < telnet->argc; i++) {
            debugf(DBG_LOW, "arg[%d]='%s'\n", i, telnet->argv[i]);
        }
    }
#endif

    if (telnet_parse_command(telnet))
        return -1;

    return 0;
}

/* read a message serialized on a stream */
int
telnet_stream_reader(int fd, int context)
{
    int ret, done;
    int xfd;
    u_char c;
    struct telnet_s *telnet;

    debugf(DBG_LOW, "telnet_stream_reader(fd=%d,context=0x%x)\n", fd, context);

    /* check for valid context - get telnet ptr */
    if (fd_context_valid(context, &xfd, &telnet))
        return -1;

    done = 0;

    /* read next */
    ret = read(fd, &c, 1);
    if (ret <= 0) {
        debugf(DBG_INFO | DBG_ERRNO, "read() error");
        return -1;
    }

    if (telnet->iac_mode) {
        c = telnet_iac_in(telnet, c);
    }

    if (telnet->cr_mode) {
        switch (telnet->cr_state) {
        case 0:
            if (c == 13) {
                telnet->cr_state = 1;
                return 0;
            }
        case 1:
            telnet->cr_state = 0;
            c = 13;
            break;
        }
    }

    switch (c) {
    case 0:
        return 0;

    case '\r':
        if (1) debugf(DBG_LOW, "ch %d\n", c);
        done = 1;
        break;

    case '\n':
        break;

    case '\b':
    case 0x7f:
        if (telnet->line_len == 0) {
            beep(telnet);
        } else {
            telnet->line_len--;
            telnet_printf(telnet, "\b \b");
        }
        break;

    case TELNET_IAC:
        telnet->iac_mode = 0;
        telnet_iac_in(telnet, c);
        break;

    default:
        if (telnet->echo_mode)
            telnet_putc(telnet, c);

        if (telnet->line_len < MAX_LINE) {
            telnet->line[telnet->line_len++] = c;
        }
        break;
    }

    /* do we have a line? */
    if (done) {
        if (telnet->echo_mode)
            telnet_printf(telnet, "\r\n");

        /* non-blank line? */
        if (telnet->line_len > 0) {
            debugf(DBG_LOW, "telnet line recieved; len %d\n",
		   telnet->line_len);

            telnet->line[telnet->line_len] = 0;

            if (telnet->login_state) {
                if(telnet_parse_login(telnet))
                return -1;
            } else {
                if(telnet_parse_input(telnet))
                    return -1;
            }
        }

        /* blank or not, output prompt (if not pending) */
        if (!telnet->pending && !telnet->wants_to_quit) {
            telnet->line_len = 0;
            telnet_prompt(telnet);
        }

        if (telnet->wants_to_quit)
            return -1;
    }
    
    return 0;
}



/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
