/* init.c */

#include "inout.h"

/* IDT_ADDR must align to 16 */
#define IDT_ADDR	0x6000
#define IDT_SIZE	(256*16)

/* code selector in gdt */
#define CODE_SEL	0x08

#define VALID_ISR	(32+2)

struct idt64_s {
    unsigned int word[4];
};

struct idt64_s *idt = ((struct idt64_s *)IDT_ADDR);

void default_isr(void);
extern void (*isr[(VALID_ISR)<<1])(void);

/* fill in a single IDT entry */
static void 
isr_entry(int index, unsigned long long offset)
{
    idt[index].word[0] = (CODE_SEL << 16) | (offset & 0xffff);
    idt[index].word[1] = (offset & 0xffff0000) | 0x8e00;
    idt[index].word[2] = 0;
    idt[index].word[3] = 0;
}

struct IDT_DESCR {
    unsigned short length;
    unsigned long long address;
} __attribute__((packed)) idt_descr = {256*16-1, IDT_ADDR};

void 
idt_install(void)
{
#if 1
    unsigned int i = 0;
    struct IDT_DESCR {
        unsigned short length;
        unsigned long long address;
    } __attribute__((packed)) idt_descr = {256*16-1, IDT_ADDR};

    for (i = 0; i < VALID_ISR; i++) {
        isr_entry(i, (unsigned long long)(isr[(i<<1)+1]));
    }

    for (++i; i < 256; i++) {
        isr_entry(i, (unsigned long long)default_isr);
    }

    asm volatile ("lidt	%0" :: "m"(idt_descr));
#else
    unsigned int i = 0;

    for (i = 0; i < VALID_ISR; i++) {
        isr_entry(i, (unsigned long long)(isr[(i<<1)+1]));
    }

    for (++i; i < 256; i++) {
        isr_entry(i, (unsigned long long)default_isr);
    }

    asm volatile ("lidt	%0" :: "m"(idt_descr));
#endif
}

#define cli() asm volatile ("cli")
#define sti() asm volatile ("sti")

#define PIC1		0x20	/* IO base address for master PIC */
#define PIC2		0xA0	/* IO base address for slave PIC */
#define PIC1_COMMAND	PIC1
#define PIC1_DATA	(PIC1+1)
#define PIC2_COMMAND	PIC2
#define PIC2_DATA	(PIC2+1)

pic_install(void)
{
    outb(0x11, 0x20);	/* icw1 */
    outb(0x11, 0xa0);
    outb(0x20, 0x21);	/* icw2 */
    outb(0x28, 0xa1);
    outb(0x04, 0x21);
    outb(0x02, 0xa1);

    outb(0x01, 0x21);
    outb(0x01, 0xa1);

    outb(0xff, 0x21);
    outb(0xff, 0xa1);
}

#define PIC_EOI		0x20	/* End-of-interrupt command code */

void pic_send_eoi(unsigned char irq)
{
    if(irq >= 8)
        outb(PIC2_COMMAND, PIC_EOI);
    
    outb(PIC1_COMMAND, PIC_EOI);
}

/*
 * reinitialize the PIC controllers, giving them specified vector offsets
 * rather than 8 and 70, as configured by default
 */

#define ICW1_ICW4	0x01	/* ICW4 (not) needed */
#define ICW1_SINGLE	0x02	/* Single (cascade) mode */
#define ICW1_INTERVAL4	0x04	/* Call address interval 4 (8) */
#define ICW1_LEVEL	0x08	/* Level triggered (edge) mode */
#define ICW1_INIT	0x10	/* Initialization - required! */

#define ICW4_8086	0x01	/* 8086/88 (MCS-80/85) mode */
#define ICW4_AUTO	0x02	/* Auto (normal) EOI */
#define ICW4_BUF_SLAVE	0x08	/* Buffered mode/slave */
#define ICW4_BUF_MASTER	0x0C	/* Buffered mode/master */
#define ICW4_SFNM	0x10	/* Special fully nested (not) */

/*
arguments:
  offset1 - vector offset for master PIC
	    vectors on the master become offset1..offset1+7
  offset2 - same for slave PIC: offset2..offset2+7

  (typically 0x20 & 0x28)
*/
void PIC_remap(int offset1, int offset2)
{
    unsigned char a1, a2;
    
    a1 = inb(PIC1_DATA);			/* save masks */
    a2 = inb(PIC2_DATA);
    
    outb(PIC1_COMMAND, ICW1_INIT+ICW1_ICW4);	/* starts the initialization sequence */
    outb(PIC2_COMMAND, ICW1_INIT+ICW1_ICW4);
    outb(PIC1_DATA, offset1);			/* define the PIC vectors */
    outb(PIC2_DATA, offset2);
    outb(PIC1_DATA, 4);				/* continue initialization sequence */
    outb(PIC2_DATA, 2);
    
    outb(PIC1_DATA, ICW4_8086);
    outb(PIC2_DATA, ICW4_8086);
    
    outb(PIC1_DATA, a1);			/* restore saved masks */
    outb(PIC2_DATA, a2);
}

static void
clear_bss(void)
{
    extern char __bss_start;
    extern char _end;
    volatile char *p;

    for (p = &__bss_start; p <= &_end; p++)
        *p = 0x0;
}

extern void timer_install(int);
extern void run(void);

void 
init(void)
{
    clear_bss();

    idt_install();
    pic_install();

    timer_install(100);
//    keyboard_install();
    sti();

    run();
}


/*
 * Local Variables:
 * indent-tabs-mode:nil
 * c-basic-offset:4
 * End:
*/
